/*
 * FTPClient.java
 *
 * Created on February 25, 2004, 5:58 PM
 */

package lu.tudor.santec.ftp;

import ftp.FtpBean;
import ftp.FtpListResult;
import ftp.FtpException;

import java.io.FileOutputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import java.util.logging.Logger;
import java.util.logging.Level;

import lu.tudor.santec.ftp.FTPFileDescriptor;
import lu.tudor.santec.ftp.FTPClientException;

//---------------------------------------------------------------------------
/** Client to access FTP servers
 * @author lu.tudor.santec/Nico MACK
 * @version 1.0
 */
//---------------------------------------------------------------------------

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

public class FTPClient 
    {
    private static Logger m_Logger = Logger.getLogger (FTPClient.class.getName());

    public static final char c_PathSeparator = '/';

    public static final int c_BufferSize = 1024;
        
    public static final int c_ASCII  = 0;
    public static final int c_Binary = 1;
        
    private FtpBean         m_FTPClient;
    private Pattern        	m_MatchPattern;
    private FtpListResult   m_FileList;
    
    private boolean         m_isConnected;

	private boolean m_Write2Logger;
    
//***************************************************************************
//* Constructor                                                             *
//***************************************************************************

//---------------------------------------------------------------------------
/** Creates a new instance of FTPClient class */    
//---------------------------------------------------------------------------

    public FTPClient() {
    	this(true);
    }
    
    
    public FTPClient(boolean write2logger) 
    {
    m_FTPClient   = new FtpBean ();
    m_isConnected = false;
    m_Write2Logger = write2logger;
    }

//***************************************************************************
//* Class Primitives                                                        *
//***************************************************************************

//---------------------------------------------------------------------------

private int getFTPFileType (int p_InternalFileType)    
    {    
    int l_FileType;
    
    switch (p_InternalFileType)
        {
        case FTPFileDescriptor.c_File      : l_FileType = FtpListResult.FILE;
                                             break;
        case FTPFileDescriptor.c_Link      : l_FileType = FtpListResult.LINK;
                                             break;
        case FTPFileDescriptor.c_Directory : l_FileType = FtpListResult.DIRECTORY;
                                             break;
        default                            : l_FileType = 0;    
        }             
    
    return l_FileType;
    }    
        
//---------------------------------------------------------------------------

private int getInternalFileType (int p_FTPFileType)    
    {    
    int l_FileType;
    
    switch (p_FTPFileType)
        {
        case FtpListResult.FILE      : l_FileType = FTPFileDescriptor.c_File;
                                       break;
        case FtpListResult.LINK      : l_FileType = FTPFileDescriptor.c_Link;
                                       break;
        case FtpListResult.DIRECTORY : l_FileType = FTPFileDescriptor.c_Directory;
                                       break;
        default                      : l_FileType = FTPFileDescriptor.c_NotSet;    
        }             
    
    return l_FileType;
    }    

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************

//---------------------------------------------------------------------------
/** attempts to establish a connection with the specified FTP server
 * @param p_FTPServer specifies the name of the FTP server or its IP Address
 * @param p_FTPUser Username of user having FTP access to specified server
 * @param p_Password Password of specified user to gain FTP access on specified server
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public void connectToServer (String p_FTPServer,
                             String p_FTPUser,
                             String p_Password)
                             
                             throws FTPClientException
    {
    FTPClientException l_Exception;
        
    if (m_isConnected) return;
    
    if (m_Write2Logger) {
    	m_Logger.log (Level.FINE,"Connecting to server " + p_FTPServer + 
    			" as user " + p_FTPUser +
    			" with password " + p_Password );    	
    }
    
    try {
     	m_FTPClient.ftpConnect (p_FTPServer,p_FTPUser,p_Password);
        m_isConnected = true;
        }
    catch (Exception p_Exception)
        {
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Connection to FTP server failed",p_Exception);
    	}
    	
        l_Exception = new FTPClientException ("ConnectFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;       
        }
    }

//---------------------------------------------------------------------------
/** retrieves the list of files stored in the specified directory
 * @param p_FTPDirectory specifies the directory on FTP server to retreive file list of
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public void buildFileList (String p_FTPDirectory)
    
                           throws FTPClientException
    {
 	FTPClientException  l_Exception;
        
    if (!m_isConnected) return;
    
    if (m_Write2Logger) {
    	m_Logger.log (Level.FINE,"Building file list of FTP directory " + p_FTPDirectory);
    }
    
    try {
         if ((p_FTPDirectory != null) && (p_FTPDirectory.length() > 0))
            {
            m_FTPClient.setDirectory (p_FTPDirectory);
            }
        
        m_FileList = m_FTPClient.getDirectoryContent();
    	}
    catch (Exception p_Exception)
        {
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Failed to retrieve file list",p_Exception);
    	}
    	
        l_Exception = new FTPClientException ("ListFilesFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;       
        }
     }

//---------------------------------------------------------------------------
/** changes current directory on FTP server
 * @param p_NewDirectory specifies the new directory to change to
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public void changeDirectory (String p_NewDirectory)
    
                           throws FTPClientException
   {
    FTPClientException l_Exception;
    
    if (!m_isConnected) return;
    
    if (m_Write2Logger) {
    	m_Logger.log (Level.FINE,"Changing FTP directory to " + p_NewDirectory);
    }	
    
    try {
         if ((p_NewDirectory != null) && (p_NewDirectory.length() > 0))
            {
            m_FTPClient.setDirectory (p_NewDirectory);
            }
        }
    catch (Exception p_Exception)
        {
    	
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Failed to change directory",p_Exception);
    	}

        l_Exception = new FTPClientException ("ChangeDirectoryFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;       
        }
    }    
        
//---------------------------------------------------------------------------
/** presets the file match pattern to be used for subsequent calls to
 * getNextFilename.
 * @param p_MatchPattern specifies an optional file match pattern. If specified, only files matching the specified filter
 * pattern will be returned. Setting p_MatchPattern to null will return all files.
 * @see java.util.regex
 */
//---------------------------------------------------------------------------

public void setFileMatchPattern (String p_MatchPattern)
    {
    if (p_MatchPattern != null) 
         m_MatchPattern = Pattern.compile (p_MatchPattern,Pattern.CASE_INSENSITIVE);
    else m_MatchPattern = null;
     }    

//---------------------------------------------------------------------------
/**
 * Allows to check whether this FTP client is currently connected or not.
 * @return <code>true</code> if this client is currently connected, <code>
 * false</code> otherwise.
 */
//---------------------------------------------------------------------------

public boolean isConnected ()
	{
	return m_isConnected;	
	}

//---------------------------------------------------------------------------
/**
 * Sets the port this FTP client should use to connect
 * @param p_Port specifies the port to be used to establish this FTP connection.
 */
//---------------------------------------------------------------------------

public void setPort (int p_Port)
	{
	m_FTPClient.setPort(p_Port);
	}

//---------------------------------------------------------------------------
/**
 * Sets the transfer more for this FTP connection to either passive or active
 * mode.
 * @param p_PassiveMode specifies whether to use passive mode or active mode
 * to transfer data. Set p_PassiveMode to <code>true</code> use passive mode,
 * respectively <code>false</code> for active mode.
 */
//---------------------------------------------------------------------------

public void setPassiveMode (boolean p_PassiveMode)
	{
	m_FTPClient.setPassiveModeTransfer(p_PassiveMode);
	}

//---------------------------------------------------------------------------
/** returns the next filename from previously retrieved file list.
 * @return The name of the next matching file from previously retrieved file list. If no
 * matching file can be found, <CODE>null</CODE> will be returned
 */
//---------------------------------------------------------------------------

public FTPFileDescriptor getNextFileDescriptor (int p_FileType)
    {
    int                 l_FileType   = 0;
     FTPFileDescriptor   l_Descriptor = null;
    boolean             l_FoundMatch = false;
    
    if (!m_isConnected || (m_FileList == null)) return null;
   
    p_FileType = getFTPFileType (p_FileType);
    
    while (!l_FoundMatch && m_FileList.next())
        {
        l_FileType = m_FileList.getType();
            
        if ((p_FileType > 0) && (l_FileType != p_FileType)) continue;
         
        if (m_MatchPattern != null)
            {
            Matcher l_Matcher = m_MatchPattern.matcher (m_FileList.getName());
            l_FoundMatch = l_Matcher.matches();
            }
        else l_FoundMatch = true;       
        }     
          
    if (l_FoundMatch)        	
    	{
        l_Descriptor = new FTPFileDescriptor ();
        l_Descriptor.setFileType  (getInternalFileType (l_FileType));    
        l_Descriptor.setFileName  (m_FileList.getName());    
        l_Descriptor.setFileSize  (m_FileList.getSize());    
        l_Descriptor.setFileDate  (m_FileList.getDate());    
        l_Descriptor.setFileOwner (m_FileList.getOwner());    
        l_Descriptor.setFileGroup (m_FileList.getGroup());    
        l_Descriptor.setFilePermission (m_FileList.getPermission());    
        }
    else l_Descriptor = null;
    
    return l_Descriptor;
    }

//---------------------------------------------------------------------------
/**
 * returns a filtered list of file from a previously retrieved file list.
 * To enable filtering, you set the applicable file match pattern before
 * calling this method.
 * @param p_FileType specifies the type of file to look for.
 * @return a List of FTPFileDescriptor objects matching the specified 
 * criteria.
 * @see #setFileMatchPattern(String)
 */
//---------------------------------------------------------------------------

public List <FTPFileDescriptor> getFilteredList (int p_FileType)
	{
	List <FTPFileDescriptor>	l_FilteredList;
	FTPFileDescriptor			l_Descriptor;
	
	l_FilteredList = new ArrayList <FTPFileDescriptor> ();
	
	do	{
		l_Descriptor = this.getNextFileDescriptor(p_FileType);
		if (l_Descriptor != null) l_FilteredList.add(l_Descriptor);
		} 
	while (l_Descriptor != null);
	
	return l_FilteredList;
	}

//---------------------------------------------------------------------------
/** fetches data from the file specified by p_Filename from current FTP directory 
 * and returns its content.
 * @param p_Filename specifies the filename to download data of
 * @param p_Mode specifies how to transfer the data. Possible values are c_ASCII or c_Binary
 * @return the content of the fetched file if successful, <code>null</code> otherwise
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public byte[] fetchData (String p_Filename, int p_Mode) throws FTPClientException
    {
    String                  l_ASCIIBuffer;
    byte[]                  l_BinaryBuffer;
     
    FTPClientException      l_Exception;
    
	if ((!m_isConnected) || (p_Filename == null)) return null;	
    
	if (m_Write2Logger) {
    m_Logger.log (Level.FINE,"Fetching data from file " + p_Filename );
	}
	
    try {
        // --- Get Data from FTP Server and store it in a byte array
        
        switch (p_Mode)
            {
            case c_ASCII    : l_ASCIIBuffer  = m_FTPClient.getAsciiFile (p_Filename,"\n"); 
                              l_BinaryBuffer = l_ASCIIBuffer.getBytes();
                              break;                      
            case c_Binary   : l_BinaryBuffer = m_FTPClient.getBinaryFile (p_Filename);
                              break;
            default         : l_ASCIIBuffer  = m_FTPClient.getAsciiFile (p_Filename,"\n"); 
                              l_BinaryBuffer = l_ASCIIBuffer.getBytes();
            }
        }
    catch (Exception p_Exception)        
        {
    	
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Failed to fetch data from file " + p_Filename,p_Exception);
    	}
    	
        l_Exception = new FTPClientException ("DownloadFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;              
        }
    
    return l_BinaryBuffer;
    }    


//---------------------------------------------------------------------------
/** uploads the data specified by p_Content and stores it as a file with the
 * name specified by p_Filename to the current FTP Directory
 * @param p_Filename specifies the name of file data should be stored under on
 * server.
 * @param p_Content specifies the data to put to server
 * @param p_Mode specifies how to transfer the data. Possible values are c_ASCII or c_Binary
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public void putData (String p_Filename, byte[] p_Content, int p_Mode) throws FTPClientException
	{
	String				l_ASCIIContent;
	FTPClientException	l_Exception;
	if ((!m_isConnected) || (p_Filename == null) || (p_Content == null)) return;	
	
	if (m_Write2Logger) {
		m_Logger.log (Level.FINE,"Putting file " + p_Filename );
	}
    try	{
		switch (p_Mode)
            {
            case c_ASCII    : l_ASCIIContent = String.valueOf(p_Content);
            				  m_FTPClient.putAsciiFile (p_Filename,l_ASCIIContent,"\n"); 
                              break;                      
            case c_Binary   : m_FTPClient.putBinaryFile (p_Filename,p_Content);
                              break;
            default         : l_ASCIIContent = String.valueOf(p_Content);
            				  m_FTPClient.putAsciiFile (p_Filename,l_ASCIIContent,"\n"); 
            }
        }
    catch (Exception p_Exception)        
        {
    	
    	if (m_Write2Logger) {
        m_Logger.log (Level.WARNING,"Failed to upload file " + p_Filename,p_Exception);
    	}
        
        l_Exception = new FTPClientException ("UploadFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;              
        }
}
//---------------------------------------------------------------------------
/** downloads the file specified by p_Filename from current FTP directory into local
 * directory specified by p_DownloadPath.
 * @param p_Filename specifies the filename to download
 * @param p_DownloadPath specifies the local destination path
 * @param p_Mode specifies how to transfer the file. Possible values are c_ASCII or c_Binary
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public void downloadFile (String p_Filename, String p_DownloadPath, int p_Mode)

                         throws FTPClientException
    {
    byte[]                  l_Buffer;
    BufferedOutputStream    l_Download;    
    FileOutputStream        l_FileStream;    
    
    FTPClientException      l_Exception;
    
    if (!m_isConnected) return;
    
    if (m_Write2Logger) {
    m_Logger.log (Level.FINE,"Downloading file " + p_Filename + 
                             " to local folder " + p_DownloadPath);
    }
      
    try {
        l_FileStream = new FileOutputStream (p_DownloadPath + c_PathSeparator + p_Filename);
    
        if (l_FileStream == null) return;

        l_Buffer = this.fetchData(p_Filename, p_Mode);

        // --- Write byte array to file
 
        l_Download = new BufferedOutputStream (l_FileStream); 
        l_Download.write (l_Buffer);    
        l_Download.flush();
        l_Download.close();        
        }
    catch (Exception p_Exception)        
        {
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Failed to download file",p_Exception);
    	}
    	
        l_Exception = new FTPClientException ("DownloadFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;              
        }
    }    

//---------------------------------------------------------------------------
/** uploads the file specified by p_Filename from the directory specified by
 * p_SourcePath to the current FTP Directory
 * @param p_Filename specifies the filename to upload
 * @param p_DownloadPath specifies the local source path
 * @param p_Mode specifies how to transfer the file. Possible values are c_ASCII or c_Binary
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public void uploadFile (String p_Filename, String p_SourcePath, int p_Mode)

                         throws FTPClientException
    {
    byte[]                  l_BinaryBuffer;
    ByteArrayOutputStream   l_FileContent;    
    BufferedInputStream     l_Upload;    
    FileInputStream         l_FileStream;    
    
    FTPClientException      l_Exception;
    int                     l_Read;
    int                     l_Offset;
    
    if (!m_isConnected) return;
    
    if (m_Write2Logger) {
    m_Logger.log (Level.FINE,"Uploading file " + p_Filename + 
                             " from local folder " + p_SourcePath);
    }
    
    try {
        l_BinaryBuffer = new byte [c_BufferSize];
        l_Offset       = 0;

        // --- Read Data from file and store it in a byte array
        
        l_FileStream  = new FileInputStream (p_SourcePath + c_PathSeparator + p_Filename);
        l_Upload      = new BufferedInputStream (l_FileStream);
        l_FileContent = new ByteArrayOutputStream ();
        
        do  {
            l_Read = l_Upload.read (l_BinaryBuffer,l_Offset,c_BufferSize);
            if (l_Read > 0)
                {   
                l_FileContent.write (l_BinaryBuffer,l_Offset,l_Read);
                l_Offset += l_Read;
                }
            } 
            while (l_Read > 0);         
        
        l_Upload.close ();
        
        if (l_Offset == 0) return;
       
        // --- Transfer byte array to FTP server
        
        this.putData (p_Filename, l_FileContent.toByteArray(), p_Mode);
         }
    catch (Exception p_Exception)        
        {
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Failed to upload file",p_Exception);
    	}
    	
        l_Exception = new FTPClientException ("UploadFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;              
        }
    }    

//---------------------------------------------------------------------------
/** Renames the file specified by p_OldFilename to p_NewFilename. Renaming operation
 * takes place in the current FTP directory
 * @param p_OldFilename specifies the file in current FTP directory to rename
 * @param p_NewFilename specifies the new filename to give to file specifed by p_OldFilename
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public void renameFile (String p_OldFilename, String p_NewFilename)

                         throws FTPClientException
    {
    FTPClientException      l_Exception;

    if (!m_isConnected) return;
    
    if (m_Write2Logger) {
    m_Logger.log (Level.FINE,"Renaming file " + p_OldFilename + 
                             " to " + p_NewFilename);
    }
    
    try {
        m_FTPClient.fileRename (p_OldFilename,p_NewFilename); 
        }
     catch (Exception p_Exception)        
        {
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Failed to rename file",p_Exception);
    	 }

        l_Exception = new FTPClientException ("RenameFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;              
        }
    }

//---------------------------------------------------------------------------
/** deletes the file specified by p_Filename in the current FTP directory
 * @param p_Filename specifies the file to delete
 * @throws FTPClientException
 */
//---------------------------------------------------------------------------

public void deleteFile (String p_Filename)

                        throws FTPClientException
    {
    FTPClientException      l_Exception;

    if (!m_isConnected) return;
    
    if (m_Write2Logger) {
    	m_Logger.log (Level.FINE,"Deleting file " + p_Filename);
    }
    
    try {
        m_FTPClient.fileDelete (p_Filename); 
        }
    catch (FtpException p_Exception)
        {
        // Exception will be thrown if specified filename does not
        // not exist. No reason to bother caller
        }
    catch (Exception p_Exception)        
        {
    	
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Failed to delete file",p_Exception);
    	}
    	
        l_Exception = new FTPClientException ("DeleteFailed");
        l_Exception.initCause(p_Exception);
        throw l_Exception;              
        }
    }

//---------------------------------------------------------------------------
/** terminates a currently open FTP connection */
//---------------------------------------------------------------------------

public void disconnectFromServer ()
    {
    
	if (m_Write2Logger) {
		m_Logger.log (Level.FINE,"Disconnecting from server");    
	}
    try {
        m_FTPClient.close();
        m_isConnected = false;
        }
    catch (Exception p_Exception)
        {
    	if (m_Write2Logger) {
    		m_Logger.log (Level.WARNING,"Disconnect failed");
        }
    }
    }

//***************************************************************************
//* End of Class                                                            *
//***************************************************************************

}
