/*
 * FTPFolderSynchronizer.java
 *
 * Created on July 2, 2004, 11:44 AM
 */

package lu.tudor.santec.ftp;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.FileNotFoundException;

import java.util.Date;
import java.util.Hashtable;
import java.util.Enumeration;

import java.util.regex.Pattern;
import java.util.regex.Matcher;

import java.util.logging.Logger;
import java.util.logging.Level;

import lu.tudor.santec.ftp.FTPClient;
import lu.tudor.santec.ftp.FTPFileDescriptor;
import lu.tudor.santec.ftp.FTPClientException;
import lu.tudor.santec.ftp.FTPFolderSynchronizerException;

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

//---------------------------------------------------------------------------
/**
 * The FTPFolderSynchronizer class implements functionality required to synchronize
 * a master folder with a slave folder. The folder whose content serves as a
 * reference is called the <b>master</b> folder whereas the folder supposed to be
 * updated is refered to as the <b>slave<b> folder. The Class distinguishes between
 * remote folders, i.e. FTP directories, and local folders. Four synchronization
 * scenarios are possible:
 * <li>Transfer from remote (FTP) master folder to local slave folder.</li>
 * <li>Transfer from local master folder to remote (FTP) slave folder.</li>
 * <li>Transfer from remote master folder to remote slave folder. <i>Operation
 * requires <b>temporary</b> folder to be set!</i></li>
 * <li>Transfer from local master folder to local slave folder.</li>
 * @author CRPHT/SANTEC/NMack
 * @version 1.0.0
 */
//---------------------------------------------------------------------------

public class FTPFolderSynchronizer 
{    
private static Logger m_Logger = Logger.getLogger (FTPFolderSynchronizer.class.getName());

private static final int c_None     = 0;
private static final int c_Add      = 1;
private static final int c_Remove   = 2;
private static final int c_Replace  = 3;
    
private File m_TempFolder;

private File m_LocalMaster;
private File m_LocalSlave;

private FTPClient m_RemoteMaster;
private FTPClient m_RemoteSlave;
  
private Pattern     m_MatchPattern;
private FileFilter  m_FileFilter;

//***************************************************************************
//* Constructor                                                             *
//***************************************************************************
    
//---------------------------------------------------------------------------
/** Creates a new instance of FTPFolderSynchronizer */
//---------------------------------------------------------------------------

public FTPFolderSynchronizer() 
    {
    setFileMatchPattern (null);
    }
            
//---------------------------------------------------------------------------
/**
 * builds a hashtable containing the list of files stored in local folder specified
 * by p_LocalFolder. Previously set File match pattern is effective.
 * @param p_LocalFolder specifies the local folder to get the file list of
 * @return the list of files contained in folder specified by p_LocalFolder. Hashtable uses
 * the filename as a key and file modification date as value.
 */
//---------------------------------------------------------------------------

private Hashtable getLocalFileList (File p_LocalFolder)
    {
    Hashtable l_FileList;
    File[]    l_Files; 
    int       l_FileCount;
    File      l_File;
    Date      l_LastModified; 
    
    m_Logger.log (Level.FINE,"Retrieving file list for local folder " + p_LocalFolder.getPath());
    
    l_Files = p_LocalFolder.listFiles (m_FileFilter);

    if (l_Files.length > 0)
        {
        l_FileList = new Hashtable (l_Files.length);

        for(l_FileCount = 0; l_FileCount < l_Files.length; l_FileCount++) 
            {
            l_File = l_Files [l_FileCount];    

            l_LastModified = new Date (l_File.lastModified());    

            l_FileList.put (l_File.getName(),l_LastModified);
            }
        }
    else l_FileList = new Hashtable ();     
    
    return l_FileList;
    }    
        
//---------------------------------------------------------------------------
/**
 * builds a hashtable containing the list of files stored in remote folder specified
 * by p_RemoteFolder. Previously set File match pattern is effective.
 * @param p_RemoteFolder specifies the remote folder to get the file list of
 * @return the list of files contained in folder specified by p_RemoteFolder. Hashtable uses
 * the filename as a key and file modification date as value.
 */
//---------------------------------------------------------------------------

private Hashtable getRemoteFileList (FTPClient p_RemoteFolder)
    {
    Hashtable         l_FileList;
    FTPFileDescriptor l_Descriptor;
        
    m_Logger.log (Level.FINE,"Retrieving file list for remote folder");

    try {
        p_RemoteFolder.buildFileList(null);
        }
    catch (FTPClientException l_Exception)    
        {
        m_Logger.log (Level.WARNING,"Failed to retrieve remote file list");
        }
            
    l_FileList = new Hashtable ();

    if (m_MatchPattern != null) p_RemoteFolder.setFileMatchPattern (m_MatchPattern.pattern());

    do  {
        l_Descriptor = p_RemoteFolder.getNextFileDescriptor (FTPFileDescriptor.c_File);

        if (l_Descriptor != null) 
            l_FileList.put (l_Descriptor.getFileName(),l_Descriptor.getFileDate());
        }
    while (l_Descriptor != null);

    return l_FileList;
    }

//---------------------------------------------------------------------------
/**
 * using the given file lists for Master folder, respectively for Slave folder, the
 * method determines the steps required to synchronize slave folder with master
 * folder.
 * @param p_MasterFileList file list of master folder
 * @param p_SlaveFileList file list of slave folder
 * @return a hashtable, containing the steps it takes to synchronize slave folder with
 * master folder. Hashtable format is similar to file lists, i.e. filename as key,
 * but operation code as value. Possible Opcodes are c_Add, c_Remove or c_Replace.
 */
//---------------------------------------------------------------------------

private Hashtable getSynchronizationSteps ( Hashtable p_MasterFileList,
                                            Hashtable p_SlaveFileList )
    {
    Hashtable   l_SynchronizationSteps;
    Enumeration l_Filenames;  
    String      l_Filename;    
    Date        l_MasterDate;
    Date        l_SlaveDate;
    
    
    m_Logger.log (Level.FINE,"Generating synchronization steps");

    l_SynchronizationSteps = new Hashtable ();
    
    l_Filenames = p_MasterFileList.keys();
    
    //==================================================================
    // Step 1
    // Determine which files are missing in slave folder and which ones
    // are out of date.

    while (l_Filenames.hasMoreElements())
        {
        l_Filename = (String)l_Filenames.nextElement();
                
        if (p_SlaveFileList.containsKey(l_Filename))
            {
            l_MasterDate = (Date) p_MasterFileList.get (l_Filename);
            l_SlaveDate  = (Date) p_SlaveFileList.get (l_Filename);
            
            if (l_MasterDate.after (l_SlaveDate)) 
                l_SynchronizationSteps.put (l_Filename,new Integer (c_Replace));   
            }
        else l_SynchronizationSteps.put (l_Filename,new Integer (c_Add));
        }
        
    //==================================================================
    // Step 2
    // Determine files that are lingering in slave folder but are no
    // no longer in master folder. These files ought to be deleted.

    l_Filenames = p_SlaveFileList.keys ();
    
    while (l_Filenames.hasMoreElements())
        {
        l_Filename = (String)l_Filenames.nextElement();
    
        if (!p_MasterFileList.containsKey(l_Filename))
            l_SynchronizationSteps.put (l_Filename,new Integer(c_Remove));
        }
    
    return l_SynchronizationSteps;
    }

//---------------------------------------------------------------------------
/**
 * copies the specified file from local master folder to local slave folder
 * @param p_Filename specifies the filename to be copied
 * @throws FTPFolderSynchronizationException Exception is thrown in any of the three following cases:
 * <ul>
 * <li>If master file could not be read, either because it does not exist or user
 * does not have the required privileges</li>
 * <li>If slave file could not be written. User probabely doesn't have write
 * permission for slave folder</li>
 * <li>A system error occured during copy operation</li>
 * </ul>
 */
//---------------------------------------------------------------------------

private void copyLocalFile (String p_Filename) throws FTPFolderSynchronizerException
    {
    File             l_MasterFile;
    File             l_SlaveFile;    
    String           l_MasterFilePath;
    String           l_SlaveFilePath;
    
    FileInputStream  l_MasterStream = null;
    FileOutputStream l_SlaveStream  = null;
    
    byte []          l_Buffer;
    int              l_BytesRead;
    
    FTPFolderSynchronizerException l_Exception;
    
    m_Logger.log (Level.FINE,"Copying local file " + p_Filename);

    l_MasterFilePath = m_LocalMaster.getPath() + FTPClient.c_PathSeparator + p_Filename;
    l_SlaveFilePath  = m_LocalSlave.getPath()  + FTPClient.c_PathSeparator + p_Filename;
       
    l_MasterFile = new File (l_MasterFilePath);
    l_SlaveFile  = new File (l_SlaveFilePath);
    
    if (!(l_MasterFile.exists() && l_MasterFile.isFile() && l_MasterFile.canRead()))
        {
        m_Logger.log (Level.WARNING,"Can't access file in master folder");
        l_Exception = new FTPFolderSynchronizerException ("FTPFolderSynchronizer:MasterUnreadable");
        throw l_Exception;
        }    
    
    if (!l_SlaveFile.canWrite())
        {
        m_Logger.log (Level.WARNING,"Can't store file into slave folder");
        l_Exception = new FTPFolderSynchronizerException ("FTPFolderSynchronizer:SlaveUnwriteable");
        throw l_Exception;
        }    
    
    try {
        try {
            l_MasterStream = new FileInputStream  (l_MasterFile);
            l_SlaveStream  = new FileOutputStream (l_SlaveFile);

            l_Buffer = new byte[1024];

            while(true) 
                {
                l_BytesRead = l_MasterStream.read(l_Buffer);
                if (l_BytesRead == -1) break;
                l_SlaveStream.write(l_Buffer, 0, l_BytesRead); 
                }
            }    
        catch (FileNotFoundException p_Exception)
            {
            // Shouldn't occure because existence of files has already been checked
            // beforehand.
            }
        catch (IOException p_Exception)
            {
            m_Logger.log (Level.WARNING,"Copy operation failed",p_Exception);
            l_Exception = new FTPFolderSynchronizerException ("FTPFolderSynchronizer:CopyFileFailed");
            throw l_Exception;
            }
        finally
            {
            if (l_MasterStream != null) l_MasterStream.close ();
            if (l_SlaveStream != null)  l_SlaveStream.close ();
            }    
        }    
    catch (IOException p_Exception) {}  
    }

//---------------------------------------------------------------------------
/**
 * transfers specified file from master folder to slave folder. The method
 * automatically determines how to transfer the file. Four different scenarios are
 * possible:
 * <ol>
 * <li>Transfer from remote (FTP) master folder to local slave folder.</li>
 * <li>Transfer from local master folder to remote (FTP) slave folder.</li>
 * <li>Transfer from remote master folder to remote slave folder. <i>Operation
 * requires <b>temporary</b> folder to be set!</i></li>
 * <li>Transfer from local master folder to local slave folder.</li>
 * @param p_Filename specifies the filename to be transfered
 * @throws FTPClientException
 * @throws FTPFolderSynchronizerException
 */
//---------------------------------------------------------------------------

private void transferFile (String p_Filename) throws FTPClientException,
                                                     FTPFolderSynchronizerException
    {
    File l_TempFile;
        
    m_Logger.log (Level.FINE,"Transfering file " + p_Filename);
    
    //==================================================================
    // Case 1
    // Remote Master Folder, Local Slave Folder

    if ((m_RemoteMaster != null) && (m_LocalSlave != null))
        {
        m_RemoteMaster.downloadFile (p_Filename,
                                     m_LocalSlave.getPath(),
                                     FTPClient.c_Binary);
        return;
        } 
        
    //==================================================================
    // Case 2
    // Local Master Folder, Remote Slave Folder
    
    if ((m_LocalMaster != null) && (m_RemoteSlave != null))
        {
        m_RemoteSlave.uploadFile (p_Filename,
                                  m_LocalMaster.getPath(),
                                  FTPClient.c_Binary);
        return;
        } 
 
    //==================================================================
    // Case 3
    // Remote Master and Slave Folder

    if ((m_RemoteMaster != null) && (m_RemoteSlave != null))
        {
        l_TempFile = new File (m_TempFolder.getPath() + FTPClient.c_PathSeparator + p_Filename);
            
        m_RemoteMaster.downloadFile (p_Filename,
                                     m_TempFolder.getPath(),
                                     FTPClient.c_Binary);
    
        m_RemoteSlave.uploadFile    (p_Filename,
                                     m_TempFolder.getPath(),
                                     FTPClient.c_Binary);
        
        if (l_TempFile.exists() && l_TempFile.isFile()) l_TempFile.delete();
        
        return;
        }  
            
    //==================================================================
    // Case 4
    // Local Master and Slave Folder
            
    if ((m_LocalMaster != null) && (m_LocalSlave != null))
        copyLocalFile (p_Filename);
    }    
        
//---------------------------------------------------------------------------        
/**
 * deletes the file specified by p_Filename in slave folder
 * @param p_Filename specifies the file to be deleted
 * @throws FTPClientException thrown if delete on remote slave folder failed
 */
//---------------------------------------------------------------------------

private void deleteFile (String p_Filename) throws FTPClientException
    {
    File l_File;    
        
    m_Logger.log (Level.FINE,"Deleting file " + p_Filename);

    if (m_RemoteSlave != null)
        m_RemoteSlave.deleteFile (p_Filename);
    
    else if (m_LocalSlave != null)
        {
        l_File = new File (m_LocalSlave.getPath() + FTPClient.c_PathSeparator + p_Filename);
        
        System.out.println (l_File.getPath());
        
        if (l_File.exists() && l_File.isFile()) l_File.delete ();    
        }
    }

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************

//---------------------------------------------------------------------------
/**
 * sets the temporary folder to the local folder specified by p_TempFolder.
 * Temporary folder is required for transfers between remote master and remote
 * slave folders.
 * @param p_TempFolder specifies the path the local folder to be used as temporary folder
 */
//---------------------------------------------------------------------------

public void setTemporaryFolder (File p_TempFolder)
    {
    if ((p_TempFolder != null) && (p_TempFolder.isDirectory()))        
        m_TempFolder = p_TempFolder;

    m_Logger.log (Level.FINE,"Temporary Folder is " + m_TempFolder);
    }

//---------------------------------------------------------------------------
/**
 * sets the master folder. You may either specify a local folder or a remote
 * folder.
 * @param p_LocalFolder specifies a local folder to be used as master folder
 * @param p_RemoteFolder specifies a remote folder to be used as master folder
 */
//---------------------------------------------------------------------------

public void setMasterFolder (File p_LocalFolder, FTPClient p_RemoteFolder)
    {
    m_LocalMaster  = null;
    m_RemoteMaster = null;    
        
    if ((p_LocalFolder != null) && (p_LocalFolder.isDirectory()))
        m_LocalMaster = p_LocalFolder;
    else if (p_RemoteFolder != null) m_RemoteMaster = p_RemoteFolder;
    }    
        
//---------------------------------------------------------------------------
/**
 * sets the slave folder. You may either specify a local folder or a remote
 * folder.
 * @param p_LocalFolder specifies a local folder to be used as slave folder
 * @param p_RemoteFolder specifies a remote folder to be used as slave folder
 */
//---------------------------------------------------------------------------

public void setSlaveFolder (File p_LocalFolder, FTPClient p_RemoteFolder)
    {
    m_LocalSlave  = null;
    m_RemoteSlave = null;    

    if ((p_LocalFolder != null) && (p_LocalFolder.isDirectory()))
        m_LocalSlave = p_LocalFolder;
    else if (p_RemoteFolder != null) m_RemoteSlave = p_RemoteFolder;
    }    
 
//---------------------------------------------------------------------------
/** presets the file match pattern to be used for building file lists
 * @param p_MatchPattern specifies an optional file match pattern. If specified, only files matching the specified filter
 * pattern will be returned. Setting p_MatchPattern to null will return all files.
 * @see java.util.regex
 */
//---------------------------------------------------------------------------

public void setFileMatchPattern (String p_MatchPattern)
    {
    if (p_MatchPattern != null)
        {
        m_Logger.log (Level.FINE,"File match pattern set to " + p_MatchPattern);
            
        m_MatchPattern = Pattern.compile (p_MatchPattern,Pattern.CASE_INSENSITIVE);
        if (m_RemoteMaster != null) m_RemoteMaster.setFileMatchPattern (p_MatchPattern);
        if (m_RemoteSlave  != null) m_RemoteSlave.setFileMatchPattern (p_MatchPattern);
                
        m_FileFilter = new FileFilter() 
            {     
            public boolean accept (File p_File) 
                {    
                Matcher l_Matcher = m_MatchPattern.matcher (p_File.getName());
                    
                return (p_File.exists() && p_File.isFile() && l_Matcher.matches());    
                }   
             };
        }     
     else 
        {   
        m_MatchPattern = null;
        
        m_FileFilter = new FileFilter() 
            {     
            public boolean accept (File p_File) 
                {    
                return (p_File.exists() && p_File.isFile());    
                }   
            };
        }     
    }    

//---------------------------------------------------------------------------
/**
 * synchronizes previously set master and slave folders.
 * @param p_DeleteFilesNotInMaster specifies whether files not found in master folder should be deleted from slave
 * folder.
 * @throws FTPClientException
 * @throws FTPFolderSynchronizerException
 * @see transferFile and deleteFile for more information about thrown Exceptions
 */
//---------------------------------------------------------------------------

public void synchronize (boolean p_DeleteFilesNotInMaster) throws FTPClientException,
                                                                  FTPFolderSynchronizerException
    {
    Hashtable   l_MasterFileList;    
    Hashtable   l_SlaveFileList;    
    Hashtable   l_SynchronizationSteps;
    Enumeration l_Filenames;
    
    String      l_Filename;
    int         l_Operation;
    
    //==================================================================
    // Step 1
    // Build file lists for master and slave folder first.

    if (m_LocalMaster != null)       l_MasterFileList = getLocalFileList (m_LocalMaster);
    else if (m_RemoteMaster != null) l_MasterFileList = getRemoteFileList (m_RemoteMaster);  
    else l_MasterFileList = new Hashtable ();
    
    if (m_LocalSlave != null)        l_SlaveFileList = getLocalFileList (m_LocalSlave);
    else if (m_RemoteSlave != null)  l_SlaveFileList = getRemoteFileList (m_RemoteSlave);  
    else l_SlaveFileList = new Hashtable ();
        
    //==================================================================
    // Step 2
    // Determine the steps required to synchronize slave folder with
    // master folder

    l_SynchronizationSteps = getSynchronizationSteps (l_MasterFileList,l_SlaveFileList);
    
    if (l_SynchronizationSteps.size() == 0)
        {
        m_Logger.log (Level.FINE,"Slave Folder is up-to-date. No synchronization required!");
        return;
        }    
            
    //==================================================================
    // Step 3
    // Perform required steps

    l_Filenames = l_SynchronizationSteps.keys ();
    
    while (l_Filenames.hasMoreElements())
        {
        l_Filename  = (String)l_Filenames.nextElement();
        l_Operation = ((Integer)l_SynchronizationSteps.get (l_Filename)).intValue();

        switch (l_Operation)
            {
            case c_Add      :      
            case c_Replace  : transferFile (l_Filename);
                              break;
            case c_Remove   : if (p_DeleteFilesNotInMaster) deleteFile (l_Filename);
                              break;
            }
        }
    }    

//***************************************************************************
//* End Of Class                                                            *
//***************************************************************************

}
