/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.usermanagement.gui.settings;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Locale;

import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedLocale;
import lu.tudor.santec.gecamed.core.ejb.session.beans.ListManagerBean;
import lu.tudor.santec.gecamed.core.ejb.session.interfaces.ListManagerInterface;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.widgets.IntegerField;
import lu.tudor.santec.gecamed.core.gui.widgets.LocaleChooser;
import lu.tudor.santec.gecamed.core.gui.widgets.PasswordChangeDialog;
import lu.tudor.santec.gecamed.core.utils.GECAMedUtils;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.usermanagement.ejb.entity.beans.GecamedUser;
import lu.tudor.santec.gecamed.usermanagement.ejb.session.beans.UserAdminBean;
import lu.tudor.santec.gecamed.usermanagement.ejb.session.interfaces.UserAdminInterface;
import lu.tudor.santec.gecamed.usermanagement.gui.AdminModule;
import lu.tudor.santec.i18n.Translatrix;
import lu.tudor.santec.settings.SettingsPlugin;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * Settings plugin to show/edit some general per-user settings in GECAMed
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @Version
 * <br>$Log: UserSettingsPlugin.java,v $
 * <br>Revision 1.29  2013-12-27 18:09:26  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.28  2013-07-15 06:18:38  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.27  2013-01-16 13:00:38  troth
 * <br>remove scanner options from the user settings.
 * <br>
 * <br>Revision 1.26  2013-01-16 11:32:48  ferring
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.25  2013-01-15 17:57:42  ferring
 * <br>Add new scan dialog for mac.
 * <br>
 * <br>Revision 1.24  2013-01-15 11:50:53  troth
 * <br>Add scanner function for mac os. (Testversion)
 * <br>
 * <br>Revision 1.23  2010-12-20 15:42:05  troth
 * <br>GUI redesign of Waitingroom and the Waitingroom get his own usersettingpanel
 * <br>
 * <br>Revision 1.22  2010-12-03 10:18:08  troth
 * <br>Complete - # 649: Wartesaal nur f�r einen Arzt anzeigen
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/649
 * <br>
 * <br>Revision 1.21  2010-11-17 14:34:51  troth
 * <br>add translation for the user setting 'open new prescription after saving / printing'
 * <br>
 * <br>Revision 1.20  2010-11-16 16:50:49  troth
 * <br>Complete - # 211: create nicer accident list dialog
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/211
 * <br>And Ticket #355 - http://santec.tudor.lu/trac/gecamed/ticket/355
 * <br>
 * <br>Revision 1.19  2010-11-15 10:26:38  troth
 * <br>add new user option 'open new prescription after saving/printing'
 * <br>
 * <br>Revision 1.18  2010-05-05 12:16:10  hermen
 * <br>fixed logout bug after changing password
 * <br>
 * <br>Revision 1.17  2009-05-07 08:26:45  hermen
 * <br>handle password changing without restarting
 * <br>
 * <br>Revision 1.16  2009-03-24 11:10:09  heinemann
 * <br>Ticket #270 make shortcut column in presc templates resizable
 * <br>fixed.
 * <br>the width of the column is now saved in the user settings.
 * <br>
 * <br>Revision 1.15  2008-09-25 09:43:07  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.14  2008-08-04 10:06:13  hermen
 * <br>moved toolbarsize setting to usersettings
 * <br>
 * <br>Revision 1.13  2008-02-12 15:12:44  hermen
 * <br>added setting to use small icons on buttons
 * <br>
 * <br>Revision 1.12  2008-01-15 10:18:39  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class UserSettingsPlugin extends SettingsPlugin {
	private static final long serialVersionUID = 1L;
	
	/**
	 * static logger for this class
	 */
	private static Logger logger = Logger.getLogger(UserSettingsPlugin.class
			.getName());
	
	/**
	 * enable system beeps
	 */
	public static final String SMALL_ICONS = "SMALL_ICONS";
	
	/**
	 * set the iconsize of the toolbar
	 */
	public static final String TOOLBAR_SIZE = "toolbarsize";
	
	/**
	 * enable system beeps
	 */
	public static final String SOUND_ENABLED = "soundEnabled";
	
	/**
	 * enable OSD
	 */
	public static final String OSD_ENABLED = "osdEnabled";
	
	/**
	 * set OSD display time
	 */
	public static final String OSD_TIME = "osdTime";
	
	/**
	 * enable toaster messages
	 */
	public static final String TOASTER_ENABLED = "toasterEnabled";
	
	/**
	 * show printer dialog
	 */
	public static final String PRINTER_DIALOG_ENABLED = "printerDialogEnabled";
	
	/**
	 * show printpreview
	 */
	public static final String PRINTER_PREVIEW_ENABLED = "printerPreviewEnabled";
	
	/**
	 * split batch printing
	 */
	public static final String PRINTER_BATCH_SPLIT_ENABLED = "PRINTER_BATCH_SPLIT_ENABLED";
	
	/**
	 * set size of split batch printing
	 */
	public static final String PRINTER_BATCH_SIZE = "PRINTER_BATCH_SIZE";
	
	public static final String GENERIC_SETTINGS = "GENERIC_SETTINGS";

	/**
	 * open new prescription after saving/printing a prescription
	 */
	public static final String OPEN_NEW_PRESCRIPTION_ENABLED = "OPEN_NEW_PRESCRIPTION_ENABLED";

	/**
	 * show the AccidentRelationDialog
	 */
	public static final String SHOW_ACCIDENT_RELATION_ENABLED = "SHOW_ACCIDENT_RELATION_ENABLED";
	
	
	private LocaleChooser userLocale;

	private GECAMedLocale[] localeArray;
	
	private GecamedUser currUser;

	private JButton passwordButton;

	private String passwordHash;
	
	private PasswordChangeDialog dialog = new PasswordChangeDialog(MainFrame.getInstance());

	private UserAdminInterface manager;

	private JPanel loginPanel;

	private JLabel localeLabel;

	private JLabel passLabel;

	private JPanel soundPanel;

	private JCheckBox soundCheckBox;

	private JCheckBox osdCheckBox;
	
	private JCheckBox toasterCheckBox;

	private JPanel printerPanel;

	private JCheckBox printDialogCheckBox;

	private JCheckBox printPreviewCheckBox;

	private JSlider osdTime;

	private JCheckBox printBatchsplitCheckBox;

	private IntegerField printBatchSize;

	private JPanel guiLook;

	private JCheckBox smallIcons;

	private JComboBox toolbarSize;

	private HashMap<String, String> genericSettings = new HashMap<String, String>();

	protected String password = null;
	
	private JPanel prescriptionPanel;

	private JCheckBox prescriptionCheckBox;
	
	private JPanel accidentRelationDialogPanel;

	private JCheckBox accidentRelationDialogCheckBox;
	
	
	
	
	
	/**
	 * create a new UserSettingsPlugin
	 * @param p_Name
	 */
	public UserSettingsPlugin(String p_Name) {
		super(p_Name);
		this.setStationaryValues();
		buildPanel();
	}
	
	/**
	 * create the GUI 
	 */
	@SuppressWarnings("unchecked")
	private void buildPanel() {
		
		Translatrix.addBundle("lu.tudor.santec.gecamed.usermanagement.gui.resources.WidgetResources");
        		
		this.currUser = GECAMedModule.getCurrentUser();
		
		setLabel(Translatrix.getTranslationString("um.User"));
		
		try {
			this.localeArray = 
				((ListManagerInterface) ManagerFactory.getRemote(ListManagerBean.class)).getGECAMedLocales();
			this.manager = 
				(UserAdminInterface) ManagerFactory.getRemote(UserAdminBean.class);
	    } catch (Exception e) {
	    	logger.log(Level.WARN, "lookup for UserAdminBean/remote failed", e);
		}
	    
		// #####################################################
	    
	    this.loginPanel = createSubPanel(Translatrix
				.getTranslationString("um.login"));
		this.loginPanel.setLayout(new FormLayout("pref:grow, 3dlu, 100dlu", "top:pref, 3dlu, pref"));
		CellConstraints cc = new CellConstraints();
	    
		this.localeLabel = new JLabel(Translatrix.getTranslationString("um.language"));
		this.loginPanel.add(this.localeLabel, cc.xy(1,1));
		
		this.userLocale = new LocaleChooser(localeArray);
		this.loginPanel.add(this.userLocale, cc.xy(3,1));

		this.passLabel = new JLabel(Translatrix.getTranslationString("um.password"));
		this.loginPanel.add(this.passLabel, cc.xy(1,3));
		
		this.passwordButton = new JButton(Translatrix.getTranslationString("um.change"));
		this.passwordButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				String pw = dialog.getPassword();
				if (pw != null) {
					passwordHash = pw;
					password = dialog.getUnencodedPassword();
				}
			}
		});
		
		if (GECAMedUtils.isDemo())
			this.passwordButton.setEnabled(false);
		
		this.loginPanel.add(this.passwordButton, cc.xy(3,3));
		addSubPanel(this.loginPanel);
		
		// #####################################################
		
		this.guiLook = createSubPanel(Translatrix
				.getTranslationString("um.guilook"));
		this.guiLook.setLayout(new GridLayout(0,1));
		
		Integer[] sizes = {
			Integer.valueOf(16),
			Integer.valueOf(24),
			Integer.valueOf(32),
			Integer.valueOf(48)
		};
		this.toolbarSize = new JComboBox(sizes);
		JPanel jp = new JPanel(new BorderLayout());
		jp.add(new JLabel(Translatrix.getTranslationString("um.guilook.ToolbarIconSize")), BorderLayout.CENTER);
		jp.add(toolbarSize, BorderLayout.EAST);
		this.guiLook.add(jp);
		
		this.smallIcons = new JCheckBox("<html><p>" + Translatrix.getTranslationString("um.guilook.useSmallIcons"));
		this.guiLook.add(this.smallIcons);
			
		addSubPanel(this.guiLook);
		
		// #####################################################
		
		this.soundPanel = createSubPanel(Translatrix
				.getTranslationString("um.messaging"));
		this.soundPanel.setLayout(new GridLayout(0,1));
		
		this.soundCheckBox = new JCheckBox("<html><p>" + Translatrix.getTranslationString("um.enableSound"));
		this.soundPanel.add(this.soundCheckBox);
		
		JPanel p = new JPanel(new BorderLayout());
		this.osdCheckBox = new JCheckBox("<html><p>" + Translatrix.getTranslationString("um.enableOSD"));		
		p.add(osdCheckBox, BorderLayout.WEST);
		this.osdTime = new JSlider(JSlider.HORIZONTAL, 0, 3000, 800);
		this.osdTime.setMajorTickSpacing(1000);
		this.osdTime.setMinorTickSpacing(200);
		//	Create the label table
		Hashtable labelTable = new Hashtable();
		labelTable.put( Integer.valueOf( 0 ), new JLabel("0s") );
		labelTable.put( Integer.valueOf( 1000 ), new JLabel("1s") );
		labelTable.put( Integer.valueOf( 2000 ), new JLabel("2s") );
		labelTable.put( Integer.valueOf( 3000 ), new JLabel("3s") );
		this.osdTime.setLabelTable( labelTable );
		this.osdTime.setPaintTicks(true);
		this.osdTime.setPaintLabels(true);
		p.add(osdTime);
		this.soundPanel.add(p);
		
		this.toasterCheckBox = new JCheckBox("<html><p>" + Translatrix.getTranslationString("um.enableToaster"));
		this.soundPanel.add(this.toasterCheckBox);
		
		addSubPanel(this.soundPanel);
		
		// #####################################################
		
		this.printerPanel = createSubPanel(Translatrix
				.getTranslationString("um.printer"));
		this.printerPanel.setLayout(new FormLayout("pref:grow, 3dlu, 20dlu , 3dlu", "top:pref, 3dlu, top:pref, 3dlu, top:pref, 3dlu"));
		
		this.printDialogCheckBox = new JCheckBox("<html><p>" + Translatrix.getTranslationString("um.showPrintDialog"));
		this.printerPanel.add(this.printDialogCheckBox, cc.xyw(1,1,3));
		
		this.printPreviewCheckBox = new JCheckBox("<html><p>" + Translatrix.getTranslationString("um.showPrintPreview"));
		this.printerPanel.add(this.printPreviewCheckBox, cc.xyw(1,3,3));
		
		this.printBatchsplitCheckBox = new JCheckBox("<html><p>" + Translatrix.getTranslationString("um.splitBatchPrinting"));
		this.printerPanel.add(this.printBatchsplitCheckBox, cc.xyw(1,5,1));
		
		this.printBatchSize = new IntegerField();
		this.printerPanel.add(this.printBatchSize, cc.xyw(3,5,1));
		
		addSubPanel(this.printerPanel);
		
		// #####################################################
		// option open new prescription
		this.prescriptionPanel = createSubPanel(Translatrix
				.getTranslationString("um.prescription"));
		this.prescriptionPanel.setLayout(new BorderLayout());
		this.prescriptionCheckBox = new JCheckBox(Translatrix.getTranslationString("um.openNewPrescription"));
		this.prescriptionPanel.add(this.prescriptionCheckBox,BorderLayout.WEST);
		
		addSubPanel(this.prescriptionPanel);
		
		// #####################################################
		// option show/not show the AccidentRelationDialog
		this.accidentRelationDialogPanel = createSubPanel(Translatrix
				.getTranslationString("um.accidentRelationDialogPanelTitle"));
		this.accidentRelationDialogPanel.setLayout(new BorderLayout());
		this.accidentRelationDialogCheckBox = new JCheckBox(Translatrix.getTranslationString("um.accidentRelationDialogText"));
		this.accidentRelationDialogPanel.add(this.accidentRelationDialogCheckBox,BorderLayout.WEST);
		
		addSubPanel(this.accidentRelationDialogPanel);
		
		
		
		// #####################################################
		
		setIcon(AdminModule.getIcon(AdminModule.ICON_USERMANAGEMENT));
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.settings.SettingsPlugin#revertToDefaults()
	 */
	public void revertToDefaults() {
		
		this.toolbarSize.setSelectedItem((Integer) getDefault(TOOLBAR_SIZE));
		this.smallIcons.setSelected((Boolean) getDefault(SMALL_ICONS));
		this.soundCheckBox.setSelected((Boolean) getDefault(SOUND_ENABLED));
		this.osdCheckBox.setSelected((Boolean) getDefault(OSD_ENABLED));
		this.osdTime.setValue((Integer) getDefault(OSD_TIME));
		this.toasterCheckBox.setSelected((Boolean) getDefault(TOASTER_ENABLED));
		this.printDialogCheckBox.setSelected((Boolean) getDefault(PRINTER_DIALOG_ENABLED));
		this.printPreviewCheckBox.setSelected((Boolean) getDefault(PRINTER_PREVIEW_ENABLED));
		try {
			this.printBatchsplitCheckBox.setSelected((Boolean) getDefault(PRINTER_BATCH_SPLIT_ENABLED));
			this.printBatchSize.setValue((Integer) getDefault(PRINTER_BATCH_SIZE));			
		} catch (Exception e) {}
		this.prescriptionCheckBox.setSelected((Boolean) getDefault(OPEN_NEW_PRESCRIPTION_ENABLED));
		this.accidentRelationDialogCheckBox.setSelected((Boolean) getDefault(SHOW_ACCIDENT_RELATION_ENABLED));
		
		reflectSettings();
		super.revertToDefaults();
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.settings.SettingsPlugin#updateSettings()
	 */
	public void updateSettings() {
		/* ================================================== */
		currUser.setGecamedLocale((GECAMedLocale)userLocale.getSelectedItem());
		currUser.setPasswordhash(this.passwordHash);
		try {
			manager.saveUser(currUser);
			if (password != null)
			    MainFrame.loginScreen.reLogin(password);
		} catch (Exception e) {
			logger.log(Level.WARN, "save UserSetting failed", e);
		}
		
		setValue(TOOLBAR_SIZE, (Integer) this.toolbarSize.getSelectedItem());
		setValue(SMALL_ICONS, (Boolean) this.smallIcons.isSelected());
		setValue(SOUND_ENABLED, (Boolean) this.soundCheckBox.isSelected());
		setValue(OSD_ENABLED, (Boolean) this.osdCheckBox.isSelected());
		setValue(OSD_TIME, (Integer) this.osdTime.getValue());
		setValue(TOASTER_ENABLED, (Boolean) this.toasterCheckBox.isSelected());
		setValue(PRINTER_DIALOG_ENABLED, (Boolean) this.printDialogCheckBox.isSelected());
		setValue(PRINTER_PREVIEW_ENABLED, (Boolean) this.printPreviewCheckBox.isSelected());
		setValue(PRINTER_BATCH_SPLIT_ENABLED, (Boolean) this.printBatchsplitCheckBox.isSelected());
		setValue(PRINTER_BATCH_SIZE, printBatchSize.getValue());
		setValue(OPEN_NEW_PRESCRIPTION_ENABLED, (Boolean) this.prescriptionCheckBox.isSelected());
		setValue(SHOW_ACCIDENT_RELATION_ENABLED, (Boolean) this.accidentRelationDialogCheckBox.isSelected());
		
		setValue(GENERIC_SETTINGS, this.genericSettings);
		super.updateSettings();
		/* ================================================== */
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.settings.SettingsPlugin#reflectSettings()
	 */
	public void reflectSettings() {
		/* ================================================== */
		if (currUser.getGecamedLocale() != null) {
			this.userLocale.setSelectedItem(currUser.getGecamedLocale());
		} else {
			try {
				this.userLocale.setSelectedItem(new GECAMedLocale(Locale.getDefault()));				
			} catch (Exception e) {}
		}
		
		this.passwordHash = currUser.getPasswordhash();
		this.password = null;
		
		super.reflectSettings();
		
		this.toolbarSize.setSelectedItem((Integer) getValue(TOOLBAR_SIZE));
		this.smallIcons.setSelected((Boolean) getValue(SMALL_ICONS));
		
		
		
		this.soundCheckBox.setSelected((Boolean) getValue(SOUND_ENABLED));
		this.osdCheckBox.setSelected((Boolean) getValue(OSD_ENABLED));
		this.osdTime.setValue((Integer) getValue(OSD_TIME));
		this.toasterCheckBox.setSelected((Boolean) getValue(TOASTER_ENABLED));
		this.printDialogCheckBox.setSelected((Boolean) getValue(PRINTER_DIALOG_ENABLED));
		this.printPreviewCheckBox.setSelected((Boolean) getValue(PRINTER_PREVIEW_ENABLED));
		try {
			this.printBatchsplitCheckBox.setSelected((Boolean) getValue(PRINTER_BATCH_SPLIT_ENABLED));
			this.printBatchSize.setValue((Integer) getValue(PRINTER_BATCH_SIZE));			
		} catch (Exception e) {}
		
		this.prescriptionCheckBox.setSelected((Boolean) getValue(OPEN_NEW_PRESCRIPTION_ENABLED));
		this.accidentRelationDialogCheckBox.setSelected((Boolean) getValue(SHOW_ACCIDENT_RELATION_ENABLED));
		
		/* ------------------------------------------------------- */
		this.genericSettings.clear();
		HashMap<String, String> tmp = (HashMap<String, String>) getValue(GENERIC_SETTINGS);
		if (tmp != null)
			this.genericSettings.putAll(tmp);
		/* ================================================== */
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.settings.SettingsPlugin#relocalize()
	 */
	public void relocalize() {
		/* ================================================== */
		super.relocalize();
		
		setLabel(Translatrix.getTranslationString("um.User"));
		this.localeLabel.setText(Translatrix
				.getTranslationString("um.language"));
		this.passLabel.setText(Translatrix
				.getTranslationString("um.password"));
		
		this.smallIcons.setText("<html><p>" + Translatrix
				.getTranslationString("um.guilook.useSmallIcons"));
		
		this.soundCheckBox.setText("<html><p>" + Translatrix
				.getTranslationString("um.enableSound"));
		this.soundCheckBox.setText("<html><p>" + Translatrix
				.getTranslationString("um.enableOSD"));
		this.soundCheckBox.setText("<html><p>" + Translatrix
				.getTranslationString("um.enableToaster"));
		this.printDialogCheckBox.setText("<html><p>" + Translatrix
				.getTranslationString("um.showPrintDialog"));
		this.printPreviewCheckBox.setText("<html><p>" + Translatrix
				.getTranslationString("um.showPrintPreview"));
		this.printBatchsplitCheckBox.setText("<html><p>" + Translatrix
				.getTranslationString("um.splitBatchPrinting"));
		
		this.prescriptionCheckBox.setText(Translatrix.getTranslationString("um.openNewPrescription"));
		
		this.accidentRelationDialogCheckBox.setText(Translatrix.getTranslationString("um.accidentRelationDialogText"));
		
		/* ================================================== */
	}
	
	/**
	 * set some static defaults
	 */
	public void setStationaryValues() {
		setStationary(TOOLBAR_SIZE, 32);
		setStationary(SMALL_ICONS, false);
		setStationary(SOUND_ENABLED, true);
		setStationary(OSD_ENABLED, true);
		setStationary(OSD_TIME, 800);
		setStationary(TOASTER_ENABLED, true);
		setStationary(PRINTER_DIALOG_ENABLED, true);
		setStationary(PRINTER_PREVIEW_ENABLED, true);
		setStationary(PRINTER_BATCH_SPLIT_ENABLED, false);
		setStationary(PRINTER_BATCH_SIZE, 100);
		setStationary(OPEN_NEW_PRESCRIPTION_ENABLED, false);
		setStationary(SHOW_ACCIDENT_RELATION_ENABLED, true);
	}
	
	
	/**
	 * Saves a value in the generic settings
	 * 
	 * @param key
	 * @param value
	 */
	public void addGenericSetting(String key, String value) {
		/* ================================================== */
		this.genericSettings.put(key, value);
		MainFrame.getInstance().saveSettings();
		/* ================================================== */
	}
	
	/**
	 * reads a value from the generic settings
	 * 
	 * @param key
	 * @return
	 */
	public String readGenericSetting(String key) {
		/* ================================================== */
		return this.genericSettings.get(key);
		/* ================================================== */
	}
	
	/**
	 * Removes a generic setting
	 * 
	 * @param key
	 */
	public void deleteGenericSettings(String key) {
		/* ================================================== */
		this.genericSettings.remove(key);
		MainFrame.getInstance().saveSettings();
		/* ================================================== */
	}
	
	/**
	 * Set/save a key and boolean value
	 * 
	 * @param key
	 * @param value
	 */
	public void setValueAndSave(String key, Object value)
	{
		setValue(key, (Boolean) value);
		reflectSettings();
		updateSettings();
		MainFrame.getInstance().saveSettings();
	}
}
