/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.reporting.gui;

import java.util.ArrayList;
import java.util.Enumeration;

import lu.tudor.santec.gecamed.reporting.ejb.entity.beans.ReportParameter;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.hp.hpl.sparta.Document;
import com.hp.hpl.sparta.Element;
import com.hp.hpl.sparta.Node;
import com.hp.hpl.sparta.Parser;


/**
 * This class is a parser for the JRXML report files from JasperReports
 * is has methods to access the report name and the reports parameters.
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 */
public class JRXMLParser {

	public static final String TAG_JASPER = "jasperReport";
	public static final String TAG_PARAMETER = "parameter";
	public static final String REPORT_NAME = "name";
	public static final String PARAMETER_NAME = "name";
	public static final String PARAMETER_CLASS = "class";
	public static final String PARAMETER_DESCRIPTION = "parameterDescription";
	public static final String PARAMETER_DEFAULT = "defaultValueExpression";
	
    private static Logger m_Logger = Logger.getLogger ("lu.tudor.santec.gecamed.reporting.gui.JRXMLParser");
    
    private Document m_Document;
	
	public JRXMLParser() {

	}
	     
//	 ***************************************************************************
//	 * Class Body                                                              *
//	 ***************************************************************************

//	 ---------------------------------------------------------------------------

	 /**
	  * Parses the JRXML report
	 * @param xmlBytes the report file as byte[]
	 * @throws Exception
	 */
	public void parseFile (byte[] xmlBytes) throws Exception
	     {     
	     try {
	         m_Document = Parser.parse (xmlBytes);
	         }
	     catch (Exception p_Exception)
	         {
	         m_Logger.log (Level.WARN,"Failed to parse file",p_Exception);
	         throw p_Exception;
	         }
	     }

//	 ---------------------------------------------------------------------------

	 /**
	  * returns the value of the xml Tag/Attribute specified by the XPath
	 * @param p_XPath the Path to the Tag
	 * @param p_Attribute the Attributre to retrieve, if null the Tagvalue is retrieved
	 * @return
	 */
	private String getValue (String p_XPath, String p_Attribute) {
	     Element             l_Match;
	     Node                l_Node;
	     String              l_Value;
	    
	     l_Value = ""; 
	     
	     if (m_Document == null) return l_Value;
	     
	     try {
	         l_Match = m_Document.xpathSelectElement (p_XPath);
	         
	         if (l_Match != null)
	             {
	             if (p_Attribute == null)    
	                 {
	                 l_Node = l_Match.getFirstChild();
	                 if (l_Node != null) l_Value = l_Node.toString();
	                 }
	             else
	                 {
	                 l_Value = l_Match.getAttribute (p_Attribute);
	                 }
	             }
	         }
	     catch (Exception p_Exception)
	         {
	         m_Logger.log (Level.WARN,"Error while retrieving value " + p_XPath,p_Exception);
	         }
	    
	     return l_Value;
	     }  

	 @SuppressWarnings("rawtypes")
	private Enumeration getChilds (String p_XPath) 
	 {
	 	try {
	 	    return m_Document.xpathSelectElements(p_XPath);
	 	}catch (Exception p_Exception) {
	 		return null;
	 	}
	 }
	 
	 /**
	  * returns the Reports name
	 * @return the Reports name
	 */
	public String getReportName() {
		return getValue(TAG_JASPER, REPORT_NAME);
	}
	 
	 /**
	  * returns an Aray of all ReportParameters
	 * @return an Aray of all ReportParameters
	 */
	public ReportParameter[] getReportParameters() {
		ArrayList<ReportParameter> parameters = new ArrayList<ReportParameter>();
		@SuppressWarnings("rawtypes")
		Enumeration en = getChilds(TAG_JASPER + "/" + TAG_PARAMETER);
		while (en.hasMoreElements()) {
			Element element = (Element) en.nextElement();
			ReportParameter param = new ReportParameter();
			param.setName(element.getAttribute(PARAMETER_NAME));
			param.setType(element.getAttribute(PARAMETER_CLASS));
			try {
				param.setComment(element.xpathSelectElement(PARAMETER_DESCRIPTION).toString());
			} catch (Exception e) {
			}
			try {
				param.setDefaultValue(element.xpathSelectElement(PARAMETER_DEFAULT).toString());
			} catch (Exception e) {
			}
			parameters.add(param);
		}
		return parameters.toArray(new ReportParameter[0]);
	 }
	 
	 
//	 ***************************************************************************
//	 * End of Class                                                            *
//	 ***************************************************************************


	
}
