/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/**
 * @author Martin Heinemann martin.heinemann@tudor.lu
 *
 *
 *
 * @version
 * <br>$Log: MemoControler.java,v $
 * <br>Revision 1.6  2013-12-27 18:09:27  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.5  2013-07-15 06:18:39  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.4  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.3  2008-01-03 14:44:46  heinemann
 * <br>cleaned imports
 * <br>
 * <br>Revision 1.2  2007-11-20 08:58:54  hermen
 * <br>moved Managerfactory to core.utils and refactured code to use ManagerFactory instead of context.lookup
 * <br>
 * <br>Revision 1.1  2007-10-22 09:58:14  heinemann
 * <br>*** empty log message ***
 * <br>
 *   
 */
package lu.tudor.santec.gecamed.patient.gui.memo;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Vector;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.Icon;

import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.PatientMemo;
import lu.tudor.santec.gecamed.patient.ejb.session.beans.PatientMemoManagerBean;
import lu.tudor.santec.gecamed.patient.ejb.session.interfaces.PatientMemoInterface;
import lu.tudor.santec.gecamed.patient.gui.PatientManagerModule;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * @author martin.heinemann@tudor.lu
 * 22.10.2007
 * 10:01:29
 *
 *
 * @version
 * <br>$Log: MemoControler.java,v $
 * <br>Revision 1.6  2013-12-27 18:09:27  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.5  2013-07-15 06:18:39  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.4  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.3  2008-01-03 14:44:46  heinemann
 * <br>cleaned imports
 * <br>
 * <br>Revision 1.2  2007-11-20 08:58:54  hermen
 * <br>moved Managerfactory to core.utils and refactured code to use ManagerFactory instead of context.lookup
 * <br>
 * <br>Revision 1.1  2007-10-22 09:58:14  heinemann
 * <br>*** empty log message ***
 * <br>
 *   
 */
public class MemoControler {
	
	/** the logger Object for this class */
	private static Logger logger = Logger.getLogger(MemoControler.class.getName());
	
	public static final String MEMOS_CHANGED = "memos_changed";
	
	
	private Patient patient;
	private Vector<PatientMemo> memos = new Vector<PatientMemo>();
	
//	private Vector<PropertyChangeListener> propListener = new Vector<PropertyChangeListener>();
	private PropertyChangeSupport changeSupport;
	
	private static Pattern 
		c_InnerMostTagPattern = Pattern.compile ("^.*<([A-Z][A-Z0-9]*)[^>]*>(.*?)</\\1>.*$",Pattern.CASE_INSENSITIVE | Pattern.DOTALL);

	private static Pattern
		c_TagContentPattern = Pattern.compile ("^.*?(\\w+).*?$",Pattern.CASE_INSENSITIVE | Pattern.DOTALL);
	
	
	
	public MemoControler() {
		/* ================================================== */

		/* ================================================== */
	}
	
	
	/**
	 * @param patient
	 */
	public void setPatient(Patient patient) {
		/* ================================================== */
		if (this.patient == null || !this.patient.equals (patient)) {
			/* ------------------------------------------------------- */
				this.patient = patient;
				
				Thread t = new Thread() {
					public void run() {
						/* ================================================== */
						loadMemos();
						firePropertyChange();
						/* ================================================== */
					}
				};
				t.start();
			/* ------------------------------------------------------- */
			}
		/* ================================================== */
	}
	
	
	/**
	 * @return
	 */
	@SuppressWarnings("unchecked")
	private void loadMemos() {
		/* ================================================== */
		if (this.patient == null) {
			/* ------------------------------------------------------- */
			this.memos.clear();
			return ;
			/* ------------------------------------------------------- */
		}
		/* ------------------------------------------------------- */
		PatientMemoInterface mManager = (PatientMemoInterface) 
				ManagerFactory.getRemote(PatientMemoManagerBean.class);
		/* ------------------------------------------------------- */
		try {
			List<PatientMemo> col = (ArrayList) mManager.getMemosByPatientID(patient.getId());
			if (col != null && col.size() > 0) {
				/* ------------------------------------------------------- */
				// reverse list
//				Collections.reverse(col);
				this.memos.clear();
				this.memos.addAll(col);
				sortMemos();
				/* ------------------------------------------------------- */
			}
			else {
				this.memos.clear();
			}
		} catch (Exception e1) {
			e1.printStackTrace();
		}
		/* ================================================== */
	}
	
	
	public Vector<PatientMemo> getMemos() {
		/* ================================================== */
		return this.memos;
		/* ================================================== */
	}
	
	/**
	 * @return
	 */
	public Long getMemoCount() {
		/* ================================================== */
		Long count = (Long) PatientManagerModule.getCache(this.patient)
								.get(PatientManagerModule.CACHE_PATIENT_MEMO_COUNT);
		if (count == null)
			count = 0L;
		
		return count;
		/* ================================================== */
	}
	
	
	/**
	 * @param m
	 * @return
	 */
	public boolean addMemo(PatientMemo m) {
		/* ================================================== */
		if (m != null && hasContent(m.getNote())) {
			// save the memo
			try {
				/* ------------------------------------------------------- */
				PatientMemo savedMemo = getMemoInterface().saveMemo(m);
				this.memos.add(savedMemo);
				/* ------------------------------------------------------- */
				// update cache
				updateMemoCache(savedMemo);
				if (savedMemo.getImportant())
					updateWarningCache(savedMemo);
				// =========================================================
				sortMemos();
				firePropertyChange();
				return true;
			} catch (Exception e) {
				logger.log(Level.WARN, "saving memo failed", e);
				firePropertyChange();
				return false;
			}
		}
		return false;
		/* ================================================== */
	}
	
	/**
	 * @param m
	 */
	public void deleteMemo(PatientMemo m) {
		/* ================================================== */
		try {
			/* ------------------------------------------------------- */
			getMemoInterface().deleteMemo(m);
			
			this.memos.remove(m);
			/* ------------------------------------------------------- */
			// update cache
			if (m.getImportant()) {
				m.setImportant(false);
				updateWarningCache(m);
			}
			updateMemoCache(null);
			sortMemos();
			firePropertyChange();
			/* ------------------------------------------------------- */
		} catch (Exception e) {
			logger.log(Level.WARN, "deleting memo failed", e);
		}
		/* ================================================== */
	}
	
	
	/**
	 * @param memo
	 * @param oldwasimportant
	 */
	public void saveMemo(PatientMemo memo, boolean oldwasimportant) {
		/* ================================================== */

		if (memo != null && hasContent(memo.getNote())) {
			// save the memo
			try {
				/* ------------------------------------------------------- */
				PatientMemo savedMemo = getMemoInterface().saveMemo(memo);
				// try to find the existing memo
				if (this.memos.contains(memo)) {
					this.memos.setElementAt(memo, this.memos.indexOf(memo));
				} else
					this.memos.add(savedMemo);
				/* ------------------------------------------------------- */
				// update the cache
				if (oldwasimportant ^ savedMemo.getImportant())
					updateWarningCache(savedMemo);
				sortMemos();
				firePropertyChange();
				/* ------------------------------------------------------- */
			} catch (Exception e) {
				logger.log(Level.WARN, "saving memo failed", e);
			}
		}
		/* ================================================== */
	}
	
	
	/**
	 * returns the icon according to the count and state of the memo list
	 * 
	 * @return
	 */
	public Icon getIcon() {
		/* ================================================== */
		if (getMemoCount() > 0) {
			return GECAMedModule.getIcon(GECAMedIconNames.MEMO);
		} else {
			return GECAMedModule.getIcon(GECAMedIconNames.MEMO_EMPTY);
		}
		/* ================================================== */
	}
	
	
	public void addPropertyChangeListener(PropertyChangeListener l) {
		/* ================================================== */
		if (this.changeSupport == null)
			this.changeSupport = new PropertyChangeSupport(this);
		
		changeSupport.addPropertyChangeListener(l);
		/* ================================================== */
	}
	
	private void firePropertyChange() {
		/* ================================================== */
		if (this.changeSupport != null)
			this.changeSupport.firePropertyChange(MEMOS_CHANGED, true, false);
		/* ================================================== */
	}
	
	
	
	/**
	 * @param memo
	 */
	@SuppressWarnings("unchecked")
	private void updateWarningCache(PatientMemo memo) {
		/* ================================================== */
		List<PatientMemo> wmemos = (ArrayList<PatientMemo>) PatientManagerModule.getCache(this.patient)
					.get(PatientManagerModule.CACHE_PATIENT_WARNING_MEMOS);
		/* ------------------------------------------------------- */
		
		if (wmemos.contains(memo)) {
			/* ------------------------------------------------------- */
			if (!memo.getImportant())
				wmemos.remove(memo);
			/* ------------------------------------------------------- */
		} else {
			/* ------------------------------------------------------- */
			if (memo.getImportant())
				wmemos.add(memo);
			/* ------------------------------------------------------- */
		}
//		PatientManagerModule.getCache(this.m_Patient)
//			.put(PatientManagerModule.CACHE_PATIENT_WARNING_MEMOS, wcount);
		/* ================================================== */
	}
	
	
	/**
	 * @param memo
	 */
	private void updateMemoCache(PatientMemo memo) {
		/* ================================================== */
		Long count = (Long) PatientManagerModule.getCache(this.patient)
		 .get(PatientManagerModule.CACHE_PATIENT_MEMO_COUNT);
		
		if (memo != null) {
			/* ================================================== */
			// increase the cache count
			// =========================================================
			if (count == null)
				count = 1L;
			else
				count++;
			/* ------------------------------------------------------- */
		} else {
			// ======================================================
			// decrease cache count
			// ======================================================
			if (count != null)
				count--;
			else
				count = 0L;
			/* ------------------------------------------------------- */
		}
		PatientManagerModule.getCache(this.patient)
			.put(PatientManagerModule.CACHE_PATIENT_MEMO_COUNT, count);
		/* ================================================== */
	}
	
	
	/**
	 * @return
	 */
	private PatientMemoInterface getMemoInterface() {
		/* ================================================== */
		PatientMemoInterface mManager = (PatientMemoInterface) 
			ManagerFactory.getRemote(ManagerFactory.MEMOMANAGER);
		return mManager;
		/* ================================================== */
	}
	
	
	/**
	 * @param p_HTML
	 * @return
	 */
	public static boolean hasContent(String p_HTML) {
		Matcher l_InnerMostTagMatcher;
		Matcher l_ContentMatcher;
		boolean l_HasContent = false;

		if ((p_HTML == null) || (p_HTML.length() == 0))
			return false;

		l_InnerMostTagMatcher = c_InnerMostTagPattern.matcher(p_HTML);
		if (l_InnerMostTagMatcher.matches()) {
			if (l_InnerMostTagMatcher.group(2) != null) {
				l_ContentMatcher = c_TagContentPattern
						.matcher(l_InnerMostTagMatcher.group(2));
				if (l_ContentMatcher.matches())
					l_HasContent = true;
			}
		}
		return l_HasContent;
	}
	
	
	
	/**
	 * sorts the memos by date
	 */
	private void sortMemos() {
		
		Collections.sort(this.memos, new Comparator<PatientMemo>() {

			public int compare(PatientMemo o1, PatientMemo o2) {
				if (o1 == null || o1.getCreationDate() == null)
					return 1;
				if (o2 == null || o2.getCreationDate() == null)
					return 1;
				if (o1.getCreationDate().before(o2.getCreationDate()))
					return 1;
				if (o1.getCreationDate().after(o2.getCreationDate()))
					return -1;
				
				return 0;
			}
			
		});
	}
	
}
