/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 2010 by SANTEC/TUDOR www.santec.tudor.lu                   *
 *                                                                           *
 *                                                                           *
 *  This library is free software; you can redistribute it and/or modify it  *
 *  under the terms of the GNU Lesser General Public License as published    *
 *  by the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                      *
 *                                                                           *
 *  This software is distributed in the hope that it will be useful, but     *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of               *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU        *
 *  Lesser General Public License for more details.                          *
 *                                                                           *
 *  You should have received a copy of the GNU Lesser General Public         *
 *  License along with this library; if not, write to the Free Software      *
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA  *
 *                                                                           *
 *****************************************************************************/
package lu.tudor.santec.gecamed.patient.gui.history.antecedents;

import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JToggleButton;
import javax.swing.ScrollPaneConstants;

import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.widgets.GECAMedBaseDialog;
import lu.tudor.santec.gecamed.core.gui.widgets.NamedComponent;
import lu.tudor.santec.gecamed.core.gui.widgets.htmleditor.HTMLEditor;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Antecedents;
import lu.tudor.santec.gecamed.usermanagement.ejb.session.beans.LoginBean;
import lu.tudor.santec.gecamed.usermanagement.ejb.session.interfaces.LoginInterface;
import lu.tudor.santec.i18n.Translatrix;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * Dialog to edited antecedents. It is opened by the AntecedentsViewDialog when
 * the use double-click on a antecedent of the antecedents list. The edit is
 * based on the HTMLEditor.
 * 
 * @author Thorsten Roth thorsten.roth(at)tudor.lu
 * 
 * @version <br>
 *          $Log: AntecedentsEditDialog.java,v $
 *          Revision 1.2  2013-03-22 10:29:15  ferring
 *          store position and size of the dialog by user
 *
 *          Revision 1.1  2010-03-23 15:13:02  troth
 *          Redesign of the patient view history -
 *          Create AntecedentsEditDialog to edited antecedents
 *
 */
public class AntecedentsEditDialog extends GECAMedBaseDialog implements ActionListener {

	private static final long serialVersionUID = 1L;

	private JTextField textField;

	private HTMLEditor htmlEditor;

	private JToggleButton m_ImportantToggleButton;

	private LoginInterface loginManager;

	private String description = "";

	private Antecedents antecedents;

	private AntecedentsListModel antecedentsModel;

	private boolean newAntecedents;

	// ***************************************************************************
	// * Class Constants *
	// ***************************************************************************

	private final static Dimension c_PreferredSize = new Dimension(600, 400);

	protected static final String EDIT = "edit";

	private final static String c_Columns = "3dlu,fill:pref:grow,"
			+ "3dlu,fill:max(30dlu;pref)," + "3dlu,fill:max(30dlu;pref),3dlu";

	private final static String c_Rows = "3dlu,fill:pref,3dlu,"
			+ "fill:pref:grow," + "3dlu,fill:pref,3dlu";

	// ***************************************************************************
	// * Constructor(s) *
	// ***************************************************************************

	/**
	 * Create a new AntecedentsEditDialog
	 * @param title the title of the dialog
	 */
	public AntecedentsEditDialog(String title) {
		
		super(title, true, true);
		
		setResizingOptions(RESIZING_ALL);
		
		this.setTitle(Translatrix.getTranslationString("patient.antecedents.editorTitle"));
		loginManager = (LoginInterface) ManagerFactory
				.getRemote(LoginBean.class);
		CellConstraints l_Constraints;
		FormLayout l_Layout;
		JScrollPane l_Scroller;
		JPanel panel = new JPanel();

		l_Scroller = new JScrollPane();
		l_Scroller.setHorizontalScrollBarPolicy(ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		l_Constraints = new CellConstraints();
		l_Layout = new FormLayout(c_Columns, c_Rows);

		htmlEditor = new HTMLEditor();
		l_Scroller.setViewportView(htmlEditor);

		panel.setLayout(l_Layout);
		this.textField = new JTextField();

		panel.add(new NamedComponent(Translatrix
				.getTranslationString("patient.antecedents.shortcut"),
				textField, true, JLabel.HORIZONTAL), l_Constraints.xy(2, 2));
		panel.add(l_Scroller, l_Constraints.xywh(2, 4, 5, 1));
		
		okButton.addActionListener(this);

		cancelButton.addActionListener(this);

		m_ImportantToggleButton = new JToggleButton(Translatrix
				.getTranslationString("patient.antecedents.public"),
				GECAMedModule.getSmallIcon(GECAMedIconNames.ENTRY_PUBLIC));
		m_ImportantToggleButton.addActionListener(this);

		this.addButton(m_ImportantToggleButton);
		this.addMainPanel(panel);
		this.setSize(c_PreferredSize);
	}

	// ---------------------------------------------------------------------------

	/**
	 * Shows the Dialog and load the antecedents to edit
	 * @param antecedents specified the antecedents to edit
	 * @param antecedentsModel the antecedents model
	 * @param newAntecedents if newAntecedents is true the antecedents is add else the antecedents is update 
	 */
	public void showDialog(Antecedents antecedents,
			AntecedentsListModel antecedentsModel, boolean newAntecedents) {

		this.antecedents = antecedents;
		this.antecedentsModel = antecedentsModel;
		this.newAntecedents = newAntecedents;

		this.textField.setText(this.antecedents.getShortcut());

		// set antecedents description
		try {
			description = this.antecedents.getDescription();
		} catch (Exception e) {
		}
		this.htmlEditor.setText(description);

		// set antecedents flag private/public
		if (this.antecedents.getLockedBy() != null) {

			this.m_ImportantToggleButton.setText(Translatrix
					.getTranslationString("patient.antecedents.private"));
			this.m_ImportantToggleButton.setIcon(GECAMedModule.getSmallIcon(GECAMedIconNames.ENTRY_PRIVATE));
		} else {
			this.m_ImportantToggleButton.setText(Translatrix
					.getTranslationString("patient.antecedents.public"));
			this.m_ImportantToggleButton.setIcon(GECAMedModule.getSmallIcon(GECAMedIconNames.ENTRY_PUBLIC));
		}

		this.htmlEditor.requestFocus();
		MainFrame.showDialogCentered(this);
	}

	// ---------------------------------------------------------------------------

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent p_ActionEvent) {

		
		// save antecedents
		if (p_ActionEvent.getSource() == okButton) {
			this.setVisible(false);
			this.antecedents.setDescription(htmlEditor.getText());
			this.antecedents.setShortcut(textField.getText());
			
			if (newAntecedents)
				antecedentsModel.addAntecedents(this.antecedents, 0);
			else
				antecedentsModel.saveAntecedents(this.antecedents, 0);

			// close antecedents editdialog
		} else if (p_ActionEvent.getSource() == cancelButton) {
			this.setVisible(false);
		
			// set antecedents private/public
		} else if (p_ActionEvent.getSource().equals(m_ImportantToggleButton)) {

			if (this.m_ImportantToggleButton
					.getText()
					.equals(
							Translatrix
									.getTranslationString("patient.antecedents.public"))) {
				this.m_ImportantToggleButton.setText(Translatrix
						.getTranslationString("patient.antecedents.private"));
				this.m_ImportantToggleButton.setIcon(GECAMedModule.getSmallIcon(GECAMedIconNames.ENTRY_PRIVATE));
				this.antecedents.setLockedBy(loginManager.getCurrentUserID());
			} else {
				this.m_ImportantToggleButton.setText(Translatrix
						.getTranslationString("patient.antecedents.public"));
				this.m_ImportantToggleButton.setIcon(GECAMedModule.getSmallIcon(GECAMedIconNames.ENTRY_PUBLIC));
				this.antecedents.setLockedBy(null);
			}
		}
	}
	// ***************************************************************************
	// * End of Class *
	// ***************************************************************************
}
