/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 2010 by SANTEC/TUDOR www.santec.tudor.lu                   *
 *                                                                           *
 *                                                                           *
 *  This library is free software; you can redistribute it and/or modify it  *
 *  under the terms of the GNU Lesser General Public License as published    *
 *  by the Free Software Foundation; either version 2 of the License, or     *
 *  (at your option) any later version.                                      *
 *                                                                           *
 *  This software is distributed in the hope that it will be useful, but     *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of               *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU        *
 *  Lesser General Public License for more details.                          *
 *                                                                           *
 *  You should have received a copy of the GNU Lesser General Public         *
 *  License along with this library; if not, write to the Free Software      *
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA  *
 *                                                                           *
 *****************************************************************************/
package lu.tudor.santec.gecamed.patient.gui.history;

import java.awt.BorderLayout;
import java.awt.Component;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import javax.swing.Action;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.table.TableCellRenderer;

import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.IconFetcher;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.listener.IEntryPrintRenderer;
import lu.tudor.santec.gecamed.core.gui.listener.IEntryTypeHandler;
import lu.tudor.santec.gecamed.core.gui.plugin.filehandler.DefaultEntryDeleteAction;
import lu.tudor.santec.gecamed.core.gui.utils.LineColorCellRenderer;
import lu.tudor.santec.gecamed.core.gui.utils.StringHelper;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.esante.ejb.entity.beans.Dsp;
import lu.tudor.santec.gecamed.esante.ejb.session.beans.CDAManagerBean;
import lu.tudor.santec.gecamed.esante.gui.IconNames;
import lu.tudor.santec.gecamed.esante.gui.actions.UploadToDspAction;
import lu.tudor.santec.gecamed.esante.gui.utils.ESanteGuiUtils;
import lu.tudor.santec.gecamed.esante.gui.webservice.DspPrivileges;
import lu.tudor.santec.gecamed.esante.gui.webservice.Security;
import lu.tudor.santec.gecamed.esante.gui.webservice.WebserviceConstants;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.IncidentEntry;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.patient.gui.PatientManagerModule;
import lu.tudor.santec.gecamed.patient.gui.history.table.HistoryTableModel;
import lu.tudor.santec.gecamed.prescription.ejb.entity.beans.Prescription;
import lu.tudor.santec.gecamed.prescription.ejb.session.beans.PrescriptionManagerBean;
import lu.tudor.santec.gecamed.prescription.ejb.session.interfaces.PrescriptionManager;
import lu.tudor.santec.gecamed.prescription.gui.actionhandler.PrescriptionEntryCopyTextAction;
import lu.tudor.santec.gecamed.prescription.gui.actionhandler.PrescriptionEntryModifyAction;
import lu.tudor.santec.gecamed.prescription.gui.actionhandler.PrescriptionEntryReprescribeAction;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;

import furbelow.AnimatedIcon;

/**
 * Handler for the Prescription IncidentEntryTypes
 * 
 * @author martin.heinemann@tudor.lu
 * 04.03.2008
 * 10:36:20
 *
 *
 * @version
 * <br>$Log: PrescriptionHandler.java,v $
 * <br>Revision 1.26  2014-02-13 15:35:06  donak
 * <br>Fixed bug in routine detecting if the user is already logged into a patient dap but needs a presence password for uploading a document
 * <br>Fixed isUploadAllowed() routine and also its usage. This routine now indicates if the user is allowed to upload a document with his current permissions. If you want to determine if the user might be allowed to upload a document by providing a presence password, please use isPresencePasswordAllowed()
 * <br>
 * <br>Revision 1.25  2014-01-31 16:29:43  donak
 * <br>Added error dialog when document cannot be uploaded due to invalid access privileges
 * <br>Fixed bug that prevented saml assertions from being renewed after they exceeded in cache
 * <br>Fixed bug that prevented documents from being uploaded (gecamed id has not been written to template due to renaming of placeholder)
 * <br>SMART UPLOAD (TM) feature: Upload option is added to context menu dependent on dsp access permissions and upload success probability calculations
 * <br>Upload support for images added
 * <br>
 * <br>Revision 1.24  2014-01-30 18:14:12  donak
 * <br>upload option will only be offered, if the user possesses the appropriate mandate and access rights to the dsp.
 * <br>
 * <br>Revision 1.23  2013-10-29 10:04:08  donak
 * <br>Fixed: Unlinked patient records possessed the possibility to upload documents to the DSP, which is impossible
 * <br>Fixed: User will now be informed if client time is out of sync with the server and thus the saml assertion is denied
 * <br>
 * <br>Revision 1.22  2013-10-16 14:36:46  donak
 * <br>Finished document uploading.
 * <br>Increased performance, more intuitive upload process including progress indicator
 * <br>Created a separate library-like class for IncidentEntry to pdf conversion
 * <br>
 * <br>Revision 1.21  2013-10-11 12:18:09  donak
 * <br>Display "my dsp" icon for files and prescriptions that have been uploaded to dsp
 * <br>Resize of "my dsp" icons
 * <br>
 * <br>Revision 1.20  2013-10-10 16:31:35  donak
 * <br>Started to add eSanté "uploaded" indicators to incident entries
 * <br>
 * <br>Revision 1.19  2013-09-30 11:44:09  donak
 * <br>eSant� upload integration
 * <br>
 * <br>Revision 1.18  2013-07-15 06:18:37  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.17  2013-02-14 09:54:54  kutscheid
 * <br>first commit of the new/remodelled importer implementation
 * <br>fix some nullpointer exceptions
 * <br>
 * <br>Revision 1.16  2013-02-05 13:15:18  ferring
 * <br>letter status added
 * <br>
 * <br>Revision 1.15  2010-11-26 16:50:41  troth
 * <br>Complete - # 682: Mehrfache Wiederverschreibungen nicht mehr m�glich in Release 1.1.01
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/682
 * <br>
 * <br>Revision 1.14  2010-10-04 13:37:04  troth
 * <br>Complete - # 563: Verwirrung bei der Gruppierung von einzelnen Verschreibungen
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/563
 * <br>
 * <br>Revision 1.13  2010-09-06 14:28:55  troth
 * <br>Incomplete - # 624: Probleme mit  History-Display von '<'-Zeichen in S.O.A.P. Eintr�gen
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/624
 * <br>
 * <br>Revision 1.12  2010-07-28 12:39:52  troth
 * <br>fix bug now line of the open prescription entry in the history table of every open prescriptionpanel is colored not only the current.
 * <br>
 * <br>Revision 1.11  2010-05-19 12:45:08  troth
 * <br>updated Javadoc and refactored code
 * <br>
 * <br>Revision 1.10  2010-05-19 09:50:09  troth
 * <br>fix bug in the prescription renderer to draw entries in the history table  Incomplete - # 505: Scheinbar willk�rliche Ver�nderungen beim Anklicken von Historie Eintr�gen
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/505
 * <br>
 * <br>Revision 1.9  2010-04-26 16:42:58  troth
 * <br>Redesign of the prescription view
 * <br>
 * <br>Revision 1.8  2009-05-06 15:58:51  heinemann
 * <br>added method to generate printable output of the incident entrys
 * <br>
 * <br>Revision 1.7  2008-12-17 15:23:06  heinemann
 * <br>initial commit after merging of branch new prescription
 * <br>
 * <br>Revision 1.6.2.1  2008-12-15 14:08:21  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.6  2008-09-25 09:43:07  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.5  2008-07-03 11:59:57  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.4  2008-04-09 09:39:43  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.3  2008-03-20 16:08:14  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.2  2008-03-10 12:37:52  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.1  2008-03-06 15:34:27  heinemann
 * <br>*** empty log message ***
 * <br>
 *   
 */
public class PrescriptionHandler implements IEntryTypeHandler, IEntryPrintRenderer {

	/** the logger Object for this class */
	private static Logger logger = Logger.getLogger(PrescriptionHandler.class.getName());

	private PrescriptionRenderer renderer = new PrescriptionRenderer();

	// action handler for uploading prescription to eSanté DSP
	private UploadToDspAction uploadToDspAction;
	// action handler for updating a document (and it's metadata) that has already been uploaded to eSanté DSP
	private UploadToDspAction updateDocumentAction;


	private PrescriptionEntryReprescribeAction represcribeAction;

	private PrescriptionEntryCopyTextAction copyTextAction;

	private PrescriptionEntryModifyAction modifyAction;

	private DefaultEntryDeleteAction deleteAction;
	
	private boolean patientIsLinkedToDsp = false;

	public PrescriptionHandler() {
		
		// add the upload action to the prescription action handler
		this.uploadToDspAction = new UploadToDspAction();
		this.updateDocumentAction = new UploadToDspAction(Translatrix.getTranslationString("esante.actions.updateDocument"));
		this.represcribeAction = new PrescriptionEntryReprescribeAction();
		this.copyTextAction = new PrescriptionEntryCopyTextAction();
		this.modifyAction = new PrescriptionEntryModifyAction();

		this.deleteAction = new DefaultEntryDeleteAction(Translatrix.getTranslationString("prescription.buttons.delete"),
				Translatrix.getTranslationString("prescription.deleteDialog.title"), Translatrix.getTranslationString("prescription.deleteDialog.text"));
		
		Patient patient = GECAMedModule.getCurrentPatient();
		this.patientIsLinkedToDsp = CDAManagerBean.getInstance().getLinkedDsp(patient.getId())!=null;
	}

	/**
	 * Indicates if the local Gecamed patient is currently linked to an electronic health record (dsp) 
	 * @return True, if the patient is linked to a dsp
	 */
	private boolean isPatientLinkedToDsp() {
		
		Patient patient = GECAMedModule.getCurrentPatient();
		return CDAManagerBean.getInstance().getLinkedDsp(patient.getId()) != null;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see lu.tudor.santec.gecamed.core.gui.listener.IEntryTypeHandler#getRenderer(java.lang.String)
	 */
	public TableCellRenderer getRenderer(String entryType) {
		/* ====================================================== */
		return renderer;
		/* ====================================================== */
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see lu.tudor.santec.gecamed.core.gui.listener.IEntryTypeHandler#getSpecialEntries(java.lang.String, java.lang.Integer)
	 */
	public HashMap<Integer, Object> getSpecialEntries(String entryType, Integer patientId) {
		/* ====================================================== */
		// get all values from the server
		/* ------------------------------------------------------- */
		try {
			/* --------------------------------------------- */
			List<Prescription> list = getPrescriptions(patientId);
			/* ------------------------------------------------------- */
			if (list == null)
				return null;
			/* ------------------------------------------------------- */
			HashMap<Integer, Object> elements = new HashMap<Integer, Object>();
			for (Prescription p : list) {
				/* ------------------------------------------------------- */
				elements.put(p.getIncidentEntryId(), p);
				/* ------------------------------------------------------- */
			}
			return elements;
			/* --------------------------------------------- */
		} catch (Exception e) {
			/* --------------------------------------------- */
			e.printStackTrace();
			/* --------------------------------------------- */
		}
		return null;
		/* ====================================================== */
	}

	/**
	 * get all prescriptions from one patient selected by the patient id
	 * 
	 * @param patientId
	 *            the patient id
	 * @return List<Prescription> return a list of all prescriptions of one patient
	 */
	public List<Prescription> getPrescriptions(Integer patientId) {
		List<Prescription> list = null;
		/* ------------------------------------------------------- */
		// get all values from the server
		/* ------------------------------------------------------- */
		try {
			logger.info("Reloading prescriptions for patient " + patientId + " from server");
			PrescriptionManager manager = (PrescriptionManager) ManagerFactory.getRemote(PrescriptionManagerBean.class);

			list = manager.getPrescriptions(patientId);
			logger.info("prescriptions for patient " + patientId + " loaded");
		} catch (Exception e) {
			e.printStackTrace();
		}
		return list;
	}

/**
	 * Renderer for the Prescription IncidentEntrys
	 * 
	 * @author martin.heinemann@tudor.lu
	 * 04.03.2008
	 * 10:46:21
	 *
	 *
	 * @version
	 * <br>$Log: PrescriptionHandler.java,v $
	 * <br>Revision 1.26  2014-02-13 15:35:06  donak
	 * <br>Fixed bug in routine detecting if the user is already logged into a patient dap but needs a presence password for uploading a document
	 * <br>Fixed isUploadAllowed() routine and also its usage. This routine now indicates if the user is allowed to upload a document with his current permissions. If you want to determine if the user might be allowed to upload a document by providing a presence password, please use isPresencePasswordAllowed()
	 * <br>
	 * <br>Revision 1.25  2014-01-31 16:29:43  donak
	 * <br>Added error dialog when document cannot be uploaded due to invalid access privileges
	 * <br>Fixed bug that prevented saml assertions from being renewed after they exceeded in cache
	 * <br>Fixed bug that prevented documents from being uploaded (gecamed id has not been written to template due to renaming of placeholder)
	 * <br>SMART UPLOAD (TM) feature: Upload option is added to context menu dependent on dsp access permissions and upload success probability calculations
	 * <br>Upload support for images added
	 * <br>
	 * <br>Revision 1.24  2014-01-30 18:14:12  donak
	 * <br>upload option will only be offered, if the user possesses the appropriate mandate and access rights to the dsp.
	 * <br>
	 * <br>Revision 1.23  2013-10-29 10:04:08  donak
	 * <br>Fixed: Unlinked patient records possessed the possibility to upload documents to the DSP, which is impossible
	 * <br>Fixed: User will now be informed if client time is out of sync with the server and thus the saml assertion is denied
	 * <br>
	 * <br>Revision 1.22  2013-10-16 14:36:46  donak
	 * <br>Finished document uploading.
	 * <br>Increased performance, more intuitive upload process including progress indicator
	 * <br>Created a separate library-like class for IncidentEntry to pdf conversion
	 * <br>
	 * <br>Revision 1.21  2013-10-11 12:18:09  donak
	 * <br>Display "my dsp" icon for files and prescriptions that have been uploaded to dsp
	 * <br>Resize of "my dsp" icons
	 * <br>
	 * <br>Revision 1.20  2013-10-10 16:31:35  donak
	 * <br>Started to add eSanté "uploaded" indicators to incident entries
	 * <br>
	 * <br>Revision 1.19  2013-09-30 11:44:09  donak
	 * <br>eSant� upload integration
	 * <br>
	 * <br>Revision 1.18  2013-07-15 06:18:37  ferring
	 * <br>logging changed
	 * <br>
	 * <br>Revision 1.17  2013-02-14 09:54:54  kutscheid
	 * <br>first commit of the new/remodelled importer implementation
	 * <br>fix some nullpointer exceptions
	 * <br>
	 * <br>Revision 1.16  2013-02-05 13:15:18  ferring
	 * <br>letter status added
	 * <br>
	 * <br>Revision 1.15  2010-11-26 16:50:41  troth
	 * <br>Complete - # 682: Mehrfache Wiederverschreibungen nicht mehr m�glich in Release 1.1.01
	 * <br>http://santec.tudor.lu/trac/gecamed/ticket/682
	 * <br>
	 * <br>Revision 1.14  2010-10-04 13:37:04  troth
	 * <br>Complete - # 563: Verwirrung bei der Gruppierung von einzelnen Verschreibungen
	 * <br>http://santec.tudor.lu/trac/gecamed/ticket/563
	 * <br>
	 * <br>Revision 1.13  2010-09-06 14:28:55  troth
	 * <br>Incomplete - # 624: Probleme mit  History-Display von '<'-Zeichen in S.O.A.P. Eintr�gen
	 * <br>http://santec.tudor.lu/trac/gecamed/ticket/624
	 * <br>
	 * <br>Revision 1.12  2010-07-28 12:39:52  troth
	 * <br>fix bug now line of the open prescription entry in the history table of every open prescriptionpanel is colored not only the current.
	 * <br>
	 * <br>Revision 1.11  2010-05-19 12:45:08  troth
	 * <br>updated Javadoc and refactored code
	 * <br>
	 * <br>Revision 1.10  2010-05-19 09:50:09  troth
	 * <br>fix bug in the prescription renderer to draw entries in the history table  Incomplete - # 505: Scheinbar willk�rliche Ver�nderungen beim Anklicken von Historie Eintr�gen
	 * <br>http://santec.tudor.lu/trac/gecamed/ticket/505
	 * <br>
	 * <br>Revision 1.9  2010-04-26 16:42:58  troth
	 * <br>Redesign of the prescription view
	 * <br>
	 * <br>Revision 1.8  2009-05-06 15:58:51  heinemann
	 * <br>added method to generate printable output of the incident entrys
	 * <br>
	 * <br>Revision 1.7  2008-12-17 15:23:06  heinemann
	 * <br>initial commit after merging of branch new prescription
	 * <br>
	 * <br>Revision 1.6.2.1  2008-12-15 14:08:21  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.6  2008-09-25 09:43:07  heinemann
	 * <br>fixed copyrights
	 * <br>
	 * <br>Revision 1.5  2008-07-03 11:59:57  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.4  2008-04-09 09:39:43  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.3  2008-03-20 16:08:14  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.2  2008-03-10 12:37:52  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.1  2008-03-06 15:34:27  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 *   
	 */
	class PrescriptionRenderer extends LineColorCellRenderer {

		/**
		 * 
		 */
		private static final long serialVersionUID = 1L;

		private final static int fontSize = 9;

		private DateFormat df = DateFormat.getDateInstance(DateFormat.SHORT);

		private int smallFontSize = fontSize - 2;

		private Date beginDate;

		private Date endDate;

		private Date accidentDate;

		private String accidentNr = "";

		private String beginDateStr = "";

		private String endDateStr = "";

		private String accidentDateStr = "";

		private String physicianShortName = "";

		private String dateTextline = "";

		private JLabel uploadedIconLabel;
		private JPanel cellPanel;

		private ImageIcon eSanteIcon;
		private AnimatedIcon uploadingIcon;

		public PrescriptionRenderer() {

			this.uploadedIconLabel = new JLabel();
			this.uploadedIconLabel.setOpaque(false);
			this.cellPanel = new JPanel(new BorderLayout());
			this.cellPanel.setOpaque(true);
			this.cellPanel.add(uploadedIconLabel, BorderLayout.EAST);
			this.uploadingIcon = new AnimatedIcon(IconFetcher.getIcon(IconNames.class, IconNames.LOADING_ANIMATED));
			this.eSanteIcon = ESanteGuiUtils.getESanteIcon(64);
		}

		/**
		 * Indicates if the local Gecamed patient is currently linked to an electronic health record (dsp) 
		 * @return True, if the patient is linked to a dsp
		 */
		private boolean isPatientLinkedToDsp() {
			
			Patient patient = GECAMedModule.getCurrentPatient();
			return CDAManagerBean.getInstance().getLinkedDsp(patient.getId()) != null;
		}
		
		/*
		 * (non-Javadoc)
		 * 
		 * @see lu.tudor.santec.gecamed.core.gui.utils.LineColorCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean,
		 * boolean, int, int)
		 */
		@Override
		public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
			/* ====================================================== */
			if (value instanceof IncidentEntry) {
				/* ------------------------------------------------------- */
				if (column == 2) {
					/* ------------------------------------------------------- */
					return super.getTableCellRendererComponent(table, "<html><b>M</b>", isSelected, hasFocus, row, column);
					/* ------------------------------------------------------- */
				} else if (column == 3) {
					/* ------------------------------------------------------- */
					HistoryTableModel model = (HistoryTableModel) table.getModel();
					IncidentEntry entry = (IncidentEntry) value;
					Prescription prescription = (Prescription) model.getSpecialEntry(entry.getId());
					StringBuffer buff = new StringBuffer();

					/* ------------------------------------------------------- */
					if (prescription != null) {

						// get short name of the physician of the prescription
						Physician physician = MainFrame.getPhysicianById(prescription.getPhysicianId());
						if (physician != null) {
							physicianShortName = physician.getMnemonic();
						}
						// createDate = presc.getCreationDate();
						beginDate = prescription.getPrescriptionDate();
						endDate = prescription.getExpiryDate();
						accidentDate = prescription.getAccidentDate();
						accidentNr = prescription.getAccidentNr();

						// if(createDate != null) createDateStr = df.format(createDate);
						if (physicianShortName != null) {
							dateTextline = physicianShortName;
						}

						if (beginDate != null) {
							beginDateStr = df.format(beginDate);
							dateTextline = dateTextline + " | " + Translatrix.getTranslationString("prescription.name.date") + ": " + beginDateStr;
						}

						if (endDate != null) {
							endDateStr = df.format(endDate);
							dateTextline = dateTextline + " - " + Translatrix.getTranslationString("prescription.name.expirydate") + ": " + endDateStr;
						}

						if (accidentDate != null) {
							accidentDateStr = df.format(accidentDate);
							dateTextline = dateTextline + " | Accident > Nr: " + accidentNr;
							accidentDateStr = df.format(accidentDate);
							dateTextline = dateTextline + " - Date: " + accidentDateStr;
						}

						buff.append("<html><span style=\"font-family:Arial;font-size:").append(fontSize).append("px;\">").append(prescription.toHTMLString())
								.append("</span><br><br><span style=\"font-family:Arial;font-size:").append(smallFontSize).append("px;\">")
								.append(dateTextline).append("</span></html>");

					}

					if (entry.getCdaUniqueId() == null) {
						// document hat not been uploaded to DSP - display no icon at all
						this.uploadedIconLabel.setIcon(null);
					} else if (entry.getCdaUniqueId().equals(IconNames.UPLOAD_INDICATOR_STRING)) {
						// if the document is currently in the upload process, display an animated gif to indicate activity
						this.uploadedIconLabel.setIcon(this.uploadingIcon);
					} else {
						// if the document has already been uploaded to the dsp indicate this by a "my dsp" icon
						this.uploadedIconLabel.setIcon(this.eSanteIcon);
					}
					// render the cell
					super.getTableCellRendererComponent(table, buff.toString(), isSelected, hasFocus, row, column);
					// and add it to the table
					this.cellPanel.add(this, BorderLayout.CENTER);
					// set the alignment
					this.setHorizontalAlignment(JLabel.LEFT);
					// and also the background color
					this.cellPanel.setBackground(this.getBackground());

					return this.cellPanel;
					/* ------------------------------------------------------- */
				}
				/* ------------------------------------------------------- */
			}
			return super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
			/* ====================================================== */
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see lu.tudor.santec.gecamed.core.gui.listener.IEntryTypeHandler#getActions(lu.tudor.santec.gecamed.patient.ejb.entity.beans.IncidentEntry)
	 */
	public Collection<Action> getActions(IncidentEntry entry) {

		List<Action> actions = new ArrayList<Action>();

		this.represcribeAction.setEntry(entry);
		this.copyTextAction.setEntry(entry);
		this.modifyAction.setEntry(entry);
		this.deleteAction.setEntry(entry);

		actions.add(this.represcribeAction);
		actions.add(this.copyTextAction);
		actions.add(this.modifyAction);
		actions.add(this.deleteAction);

		// obtain user privileges from cache
		DspPrivileges privileges = Security.getCachedDspPrivileges(GECAMedModule.getCurrentPatient().getIdLuxembourg());
		// only add an upload option if the patient has been linked to a dsp and the user possesses sufficient permissions
		// However if the user is not logged into the eSanté DSP thoses permissions cannot be determined.
		// Thus show the upload option also in this case
		if (isPatientLinkedToDsp() && ((privileges == null) || privileges.isUploadAllowed() || privileges.isPresencePasswordAllowed())) {

			if (entry.getCdaUniqueId() == null) {
				// this prescription is not yet available in the eSanté DSP, thus add the upload item to the context menu
				this.uploadToDspAction.setEntry(entry);
				actions.add(this.uploadToDspAction);
			} else {
				// add item for replacing the document
				this.updateDocumentAction.setUploadMode(WebserviceConstants.UPLOAD_REPLACE_DOCUMENT);
				this.updateDocumentAction.setEntry(entry);
				actions.add(this.updateDocumentAction);
			}
		}

		return actions;
	}

	public JPopupMenu getPopup(IncidentEntry entry) {
		return null;
	}

	public boolean openEntry(IncidentEntry entry) {
		/* ====================================================== */
		PatientManagerModule.getInstance().getPatientPanel().openConsultation(entry);
		return true;
		/* ====================================================== */
	}

	public IEntryPrintRenderer getPrintRenderer(String entryType) {
		/* ====================================================== */
		return this;
		/* ====================================================== */
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see lu.tudor.santec.gecamed.core.gui.listener.IEntryPrintRenderer#renderPrintEntry(lu.tudor.santec.gecamed.patient.ejb.entity.beans.IncidentEntry)
	 */
	public String renderPrintEntry(IncidentEntry entry) {
		/* ====================================================== */
		if (entry == null)
			return null;
		/* ------------------------------------------------------- */
		try {
			/* --------------------------------------------- */
			PrescriptionManager manager = (PrescriptionManager) ManagerFactory.getRemote(PrescriptionManagerBean.class);

			List<Prescription> presc = manager.getPrescriptionByIncidentEntry(entry.getId());
			if (presc != null) {
				/* ------------------------------------------------------- */
				StringBuffer buff = new StringBuffer();

				buff.append(StringHelper.NEWLINE);
				buff.append(Translatrix.getTranslationString("prescription") + ": ");
				buff.append(StringHelper.NEWLINE);

				for (Prescription p : presc) {
					buff.append(StringHelper.checkNull(p.getTextContent()));
					buff.append(StringHelper.NEWLINE);
				}
				return buff.toString();
				/* ------------------------------------------------------- */
			}
			/* --------------------------------------------- */
		} catch (Exception e) {
			/* --------------------------------------------- */
			logger.error("Error while rendering prescription print.", e);
			/* --------------------------------------------- */
		}
		return null;
		/* ====================================================== */
	}
}
