/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.gui.fileimport;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;

import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.prescription.gui.util.VLabel;
import lu.tudor.santec.i18n.Translatrix;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * @author martin.heinemann@tudor.lu
 * 10.03.2008
 * 15:55:29
 *
 *
 * @version
 * <br>$Log: FileEditPanel.java,v $
 * <br>Revision 1.3  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.2  2008-03-12 10:14:07  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.1  2008-03-11 17:17:11  heinemann
 * <br>*** empty log message ***
 * <br>
 *   
 */
public class FileEditPanel extends JPanel implements ActionListener, MouseListener {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	private JFileChooser fileChooser;
	private VLabel fileNameLabel;
	private JLabel hintLabel;
	private NiceTextArea textField;
	private JButton chooseButton;

	private int state = 0;
	private JScrollPane textScroll;
	private File currentFile;
	private JButton deleteButton;
	
	private Cursor handCursor = new Cursor(Cursor.HAND_CURSOR);
	private Cursor defaultCursor = new Cursor(Cursor.DEFAULT_CURSOR);
	
	public static final int STATE_DELETED = -1;
	
	/**
	 * @param fileChooser
	 */
	public FileEditPanel(JFileChooser fileChooser) {
		/* ================================================== */
		this.fileChooser = fileChooser;
		
		initComponent();
		initActions();
		/* ================================================== */
	}


	/**
	 * 
	 */
	private void initActions() {
		/* ====================================================== */
		this.chooseButton.addActionListener(this);
		this.deleteButton.addActionListener(this);
		this.fileNameLabel.addMouseListener(this);
		/* ====================================================== */
	}
	
	
	/**
	 * @return the state. See FileImporter for states
	 */
	public int getState() {
		/* ================================================== */
		return this.state;
		/* ================================================== */
	}
	
	/**
	 * @return the selected file.
	 */
	public File getFile() {
		/* ================================================== */
		return this.currentFile;
		/* ================================================== */
	}
	
	/**
	 * @return
	 */
	public String getDescription() {
		/* ================================================== */
		return this.textField.getText();
		/* ================================================== */
	}
	
	/**
	 * 
	 */
	private void initComponent() {
		/* ====================================================== */
		this.setLayout(new FormLayout(
				// cols
				"3dlu," +
				"fill:100dlu:grow," +
				"4dlu," +
				"fill:pref," +
				"3dlu," +
				"fill:pref," +
				"3dlu",
				// rows
				"3dlu," +
				"fill:pref," +
				"4dlu," +
				"fill:40dlu," +
				"3dlu"));
		this.setOpaque(false);
		/* ------------------------------------------------------- */
		this.fileNameLabel = new VLabel("");
		this.hintLabel = new JLabel();
		this.textField = new NiceTextArea(Translatrix.getTranslationString("fileimport.dialog.enterDescription"));
		this.chooseButton = new JButton("...");
		this.deleteButton = new JButton("X");
		this.textScroll = new JScrollPane(this.textField);
		/* ------------------------------------------------------- */
		CellConstraints cc = new CellConstraints();
		this.add(fileNameLabel, cc.xy(2, 2));
		this.add(hintLabel, cc.xy(2, 3));
		this.add(chooseButton, cc.xy(4, 2));
		this.add(deleteButton, cc.xy(6, 2));
		/* ------------------------------------------------------- */
		this.add(textScroll, cc.xy(2, 4));
		
		/* ------------------------------------------------------- */
		chooseButton.setToolTipText(Translatrix.getTranslationString("fileimport.dialog.tooltip.changefile"));
		deleteButton.setToolTipText(Translatrix.getTranslationString("fileimport.dialog.tooltip.deletefile"));
		fileNameLabel.setToolTipText(Translatrix.getTranslationString("fileimport.dialog.tooltip.filenamelabel"));
		/* ====================================================== */
	}
	
	
	/**
	 * @param file
	 */
	public void setFile(File file) {
		/* ================================================== */
		int check = FileImporter.checkFile(file);
		/* ------------------------------------------------------- */
		this.currentFile = file;
		fileNameLabel.setText(file.getAbsolutePath());
		
		
		
		switchState(check);
		/* ================================================== */
	}
	
	/**
	 * @param state
	 */
	private void switchState(int state) {
		/* ================================================== */
		this.state = state;
		if (state == FileImporter.STATE_OK) {
			/* ------------------------------------------------------- */
			// green label
			/* ------------------------------------------------------- */
			fileNameLabel.setState(VLabel.STATE_TRUE);
			fileNameLabel.setWarning(null);
			/* ------------------------------------------------------- */
		} else 
			if (state == FileImporter.STATE_TOO_BIG) {
				/* ------------------------------------------------------- */
				// red label
				/* ------------------------------------------------------- */
				fileNameLabel.setState(VLabel.STATE_FALSE);
				fileNameLabel.setWarning(Translatrix.getTranslationString("fileimport.file2big"));
				/* ------------------------------------------------------- */
			} else
				if (state == FileImporter.STATE_ERROR) {
					/* ------------------------------------------------------- */
					// red label
					/* ------------------------------------------------------- */
					fileNameLabel.setState(VLabel.STATE_FALSE);
					fileNameLabel.setWarning(Translatrix.getTranslationString("fileimport.error"));
					/* ------------------------------------------------------- */
				}
				else
					if (state == STATE_DELETED) {
						/* ------------------------------------------------------- */
						this.setVisible(false);
						this.setSize(this.getWidth(), 0);
						/* ------------------------------------------------------- */
					}
		/* ================================================== */
	}


	public void actionPerformed(ActionEvent e) {
		/* ====================================================== */
		if (chooseButton.equals(e.getSource())) {
			/* ------------------------------------------------------- */
			// show the file chooser
			/* ------------------------------------------------------- */
			this.fileChooser.showOpenDialog(MainFrame.getInstance());
			
			fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
			fileChooser.setMultiSelectionEnabled(false);
			
			File f = fileChooser.getSelectedFile();
			if (f != null)
				setFile(f);
			/* ------------------------------------------------------- */
		}
		if (deleteButton.equals(e.getSource())) {
			/* ------------------------------------------------------- */
			switchState(STATE_DELETED);
			/* ------------------------------------------------------- */
		}
		/* ====================================================== */
	}


	public void mouseClicked(MouseEvent e) {}
	public void mouseEntered(MouseEvent e) {
		/* ================================================== */
		this.setCursor(handCursor);
		/* ================================================== */
	}
	public void mouseExited(MouseEvent e) {
		/* ================================================== */
		this.setCursor(defaultCursor);
		/* ================================================== */
	}
		public void mouseReleased(MouseEvent e) {}


	public void mousePressed(MouseEvent e) {
		/* ====================================================== */
		if (fileNameLabel.equals(e.getSource())) {
			/* ------------------------------------------------------- */
			setCollapsed(textScroll.isVisible());
			/* ------------------------------------------------------- */
		}
		/* ====================================================== */
	}

	/**
	 * Show the textarea or not
	 * 
	 * @param b
	 */
	public void setCollapsed(boolean b) {
		/* ================================================== */
		if (b) {
			textScroll.setVisible(false);
			this.setPreferredSize(new Dimension(this.getPreferredSize().width, fileNameLabel.getPreferredSize().height+10));
		} else {
			textScroll.setVisible(true);
			this.setPreferredSize(new Dimension(this.getPreferredSize().width, fileNameLabel.getPreferredSize().height+80));
			this.textField.requestFocus();
		}
		
		this.validate();
		this.updateUI();
		/* ================================================== */
	}
	
}
