/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.gui.administrative;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import javax.swing.DefaultListCellRenderer;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;

import lu.tudor.santec.gecamed.address.ejb.entity.beans.AddressType;
import lu.tudor.santec.gecamed.address.ejb.session.beans.AddressManagerBean;
import lu.tudor.santec.gecamed.address.ejb.session.interfaces.AddressManagerInterface;
import lu.tudor.santec.gecamed.address.gui.widgets.AddressPanel;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.GECAMedGuiUtils;
import lu.tudor.santec.gecamed.core.gui.widgets.NamedComponent;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.PatientAddress;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * Panel to display and edit the patients addresses
 * 
 *
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @version
 * <br>$Log: PatientAddressInfo.java,v $
 * <br>Revision 1.51  2013-12-27 18:08:00  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.50  2013-07-15 06:18:34  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.49  2009-05-29 13:51:21  hermen
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.48  2009-03-10 09:51:42  hermen
 * <br>small bugfixes
 * <br>
 * <br>Revision 1.47  2009-01-06 14:48:08  hermen
 * <br>small bugfixes
 * <br>
 * <br>Revision 1.46  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.45  2008-06-30 08:16:45  hermen
 * <br>added address autocompletion by locality
 * <br>
 * <br>Revision 1.44  2008-06-04 12:54:01  hermen
 * <br>fixed copy patient address to assure address
 * <br>
 * <br>Revision 1.43  2008-04-16 14:24:28  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.42  2008-04-15 16:01:03  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.41  2008-04-09 08:53:42  hermen
 * <br>fixed small possible bugs
 * <br>
 * <br>Revision 1.40  2008-03-14 08:13:23  hermen
 * <br>enhanced focus traversal
 * <br>
 * <br>Revision 1.39  2008-01-21 14:03:27  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class PatientAddressInfo extends JPanel implements KeyListener, ItemListener, PropertyChangeListener, IPatientDataPanel
{
    //~ Static fields/initializers =============================================

	/**
	 * static logger for this class
	 */
	private static Logger logger = Logger.getLogger(PatientAddressInfo.class.getName());
	
    private static final long serialVersionUID = 1L;
//    private static final String COLUMNSPACE = "2dlu";

    //~ Instance fields ========================================================

    private JComboBox type;

    private JTable addressTable;
    private JButton newButton;
    private JButton deleteButton;

	private boolean listenersActivated;

	private PatientAdminPanel patientEditPanel;
	private AddressManagerInterface manager;
	private String[] typeArray;
	private PatientAddressTableModel tableModel;

	private PatientAddress currentAddress;
	private AddressPanel addressPanel;


    //~ Constructors ===========================================================
    /**
     * builds a new PatientInfo to show/edit general Patient Data
     */
    public PatientAddressInfo(PatientAdminPanel patientEditPanel)
    {
    	this.setOpaque(false);
        this.patientEditPanel = patientEditPanel;
        this.buildPanel();
    }

    //~ Methods ================================================================
    /**
     * saves the Values of all fields to the passed PatientAddress object
     *
     * @param address the PatientAddress object to store the values in
     */
    public void getAddress(PatientAddress address)
    {
    	if (address == null) {
    		address = new PatientAddress();
    	}
    	try {
    		address.setType(type.getSelectedItem().toString());
    	} catch (Exception e) {
    		logger.log(Level.WARN, "setting address type failed", e);
		}
    	address.setZip(this.addressPanel.getZip());

    	address.setLocalityId(this.addressPanel.getLocalityID());
        address.setLocality(this.addressPanel.getTfLocality());

        address.setStreetName(this.addressPanel.getStreetName());
        address.setStreetNumber(this.addressPanel.getStreetNumber());

        address.setCountry(this.addressPanel.getCountry());

        if (address.getDate() == null) {
        	address.setDate(new Date());
        }
    }

    /**
     * fills the fields of the panel with data from the passed PatientAddress
     * object.
     *
     * @param address the PatientAddress object to load the values from
     */
    private void setAddress(PatientAddress address)
    {
    	listenersActivated = false;
    	
    	try
    	{
	    	if (address.getType() != null) {
		    	try {
		    		this.type.setSelectedItem(address.getType());
				} catch (Exception e) {
					try {
		    			this.type.setSelectedIndex(0);
					} catch (Exception ee) {
					}
				}
	    	} else {
	    		this.type.setSelectedIndex(0);
	    	}
	
			this.addressPanel.setZip(address.getZip());
	
			this.addressPanel.setLocalityID(address.getLocalityId());
			if (address.getLocalityId() == null) {
				this.addressPanel.setLocality(address.getLocality());
			}
			
			this.addressPanel.setStreet(address.getStreetName(), address.getStreetNumber());
			
			this.addressPanel.setCountry(address.getCountry());
    	}
    	finally
    	{
    		listenersActivated = true;
    	}
    }

    /**
     * adds the components to the panel
     */
    private void buildPanel() {
    	/* ================================================== */
//    	this.setHeader(Translatrix.getTranslationString("pm.address")+":");
//    	this.setHeaderIcon(PatientManagerModule.getMediumIcon(PatientManagerModule.INCIDENT_DEFAULT));
//    	this.enableAlpha(true);
        initComponents();
        // build Layout
        String col = "";
//        for (int i = 1; i <= 12; i++) {
//            col = col + "fill:25dlu:grow, " + COLUMNSPACE + ", ";
//        }
        col = "fill:22dlu,2dlu,fill:200dlu:grow,2dlu,fill:12dlu";
        /* ------------------------------------------------------- */
        FormLayout layout = new FormLayout(col,"pref, 3dlu, top:50dlu");
        this.setLayout(layout);
        CellConstraints cc = new CellConstraints();
        /* ------------------------------------------------------- */
        // add the type combobox
        /* ------------------------------------------------------- */
        this.add(new NamedComponent(Translatrix.getTranslationString(
									"pm.address.type"), this.type, false),
				cc.xyw(1, 1, 1));
        /* ------------------------------------------------------- */
        // add the address panel
        /* ------------------------------------------------------- */
        this.add(this.addressPanel, cc.xyw(3, 1, 3));
        /* ------------------------------------------------------- */
        // add the table containing all addresses
        /* ------------------------------------------------------- */
        JScrollPane pane = new JScrollPane(this.addressTable);
        pane.setBackground(GECAMedColors.c_ScrollPaneBackground);
        pane.getViewport().setOpaque(false);
        this.add(pane, cc.xyw(1, 3, 3));
        /* ------------------------------------------------------- */
        // create the buttons
        /* ------------------------------------------------------- */
        JPanel bb = new JPanel(new FormLayout("pref", "pref,1dlu,pref,pref:grow"));
        bb.setOpaque(false);
        bb.add(newButton, cc.xy(1, 1));
        bb.add(deleteButton, cc.xy(1, 3));
//        bsb.addFixed(this.newButton);
//        bsb.addFixed(this.deleteButton);
//        bsb.getPanel().setOpaque(false);
//        /* ------------------------------------------------------- */
//        this.add(bsb.getPanel(), cc.xyw(5, 3, 1));
        this.add(bb, cc.xy(5, 3));
        /* ================================================== */
    }

    /**
     * initializes the Components
     */
    private void initComponents() {
    	/* ================================================== */
        // Get the type manager
    	/* ------------------------------------------------------- */
    	listenersActivated = false;
    	
    	try
    	{
			try {
				manager = (AddressManagerInterface) ManagerFactory.getRemote(AddressManagerBean.class);
				typeArray = manager.getTypes();
		    } catch (Exception e) {
		    	MainFrame.reportServerError(e);
			}
		    /* ------------------------------------------------------- */
		    this.type = new JComboBox(typeArray);
		    this.type.setFocusable(false);
		    this.type.setEditable(false);
		    this.type.setRenderer(new AddressTypeListRenderer());
	    	this.type.addItemListener(this);
	    	/* ------------------------------------------------------- */
	    	this.addressPanel = new AddressPanel(false, false);
	    	this.addressPanel.addPropertyChangeListener(this);
	    	/* ------------------------------------------------------- */
	    	this.tableModel = new PatientAddressTableModel();
	    	this.addressTable = new JTable(this.tableModel);
	    	this.addressTable.setFocusable(false);
	    	this.addressTable.setOpaque(false);
	    	this.addressTable.setGridColor(this.addressTable.getBackground());
	    	this.addressTable.getColumnModel().getColumn(0).setMaxWidth(35);
	    	this.addressTable.getColumnModel().getColumn(3).setMaxWidth(100);
	    	this.addressTable.getColumnModel().getColumn(4).setMaxWidth(80);
	    	this.addressTable.setRowHeight(20);
	    	/* ------------------------------------------------------- */
	    	this.addressTable.getSelectionModel().addListSelectionListener(new ListSelectionListener(){
				public void valueChanged(ListSelectionEvent e) {
					if (e.getValueIsAdjusting())
						return;
					
					currentAddress = tableModel.getAddress(addressTable.getSelectedRow());
					if (currentAddress == null) {
						currentAddress = new PatientAddress();
					}
					setAddress(currentAddress);
				}
	    	});
	    	/* ------------------------------------------------------- */
	    	this.newButton = new JButton(GECAMedModule.getSmallIcon(GECAMedIconNames.ADD_LINE));
	//    	this.newButton = ButtonFactory.createEffectButton(GECAMedModule.getSmallIcon(GECAMedIconNames.ADD_LINE));
	    	
	    	this.newButton.setToolTipText(Translatrix.getTranslationString("pm.new"));
	    	this.newButton.setFocusable(false);
	    	this.newButton.setMargin(new java.awt.Insets(0, 0, 0, 0));
	    	
	    	/* ------------------------------------------------------- */
	    	this.newButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					currentAddress = new PatientAddress();
					setAddress(currentAddress);
					tableModel.getData().add(0,currentAddress);
					tableModel.fireTableDataChanged();
					addressTable.setRowSelectionInterval(0,0);
					patientEditPanel.setModified(true);
					listenersActivated = true;
				}
	    	});
	    	/* ------------------------------------------------------- */
	    	this.deleteButton = new JButton(GECAMedModule.getSmallIcon(GECAMedIconNames.REMOVE_LINE));
	//    	this.deleteButton = ButtonFactory.createEffectButton(GECAMedModule.getSmallIcon(GECAMedIconNames.REMOVE_LINE));
	    	this.deleteButton.setToolTipText(Translatrix.getTranslationString("pm.address.delete"));
	    	this.deleteButton.setMargin(new java.awt.Insets(0, 0, 0, 0));
	    	/* ------------------------------------------------------- */
	    	this.deleteButton.setFocusable(false);
	    	this.deleteButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					tableModel.getData().remove(addressTable.getSelectedRow());
					tableModel.fireTableDataChanged();
					try {
						addressTable.setRowSelectionInterval(0,0);
					} catch (Exception ee) {
					}
					patientEditPanel.setModified(true);
				}
	    	});
	    	/* ------------------------------------------------------- */
			this.newButton.setHorizontalTextPosition(JLabel.RIGHT);
			this.newButton.setHorizontalAlignment(JLabel.LEFT);
			this.deleteButton.setHorizontalTextPosition(JLabel.RIGHT);
			this.deleteButton.setHorizontalAlignment(JLabel.LEFT);
			/* ------------------------------------------------------- */
    	}
    	finally
    	{
    		listenersActivated = true;
    	}
		/* ================================================== */
    }

	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyTyped(java.awt.event.KeyEvent)
	 */
	public void keyTyped(KeyEvent e) {
		if (listenersActivated ) {
			this.patientEditPanel.setModified(true);
		}
	}
	public void keyPressed(KeyEvent e) {
	}
	public void keyReleased(KeyEvent e) {
	}

	public void itemStateChanged(ItemEvent e) {
		if (listenersActivated) {
			updateAddressInTable();
		}
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.gecamed.patient.gui.administrative.IPatientDataPanel#setPatient(lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient, boolean)
	 */
	public void setPatient(Patient patient, boolean editable) {

		setEditable(editable);

		Collection<PatientAddress> c = patient.getAddress();
		if (c == null) {
			c = new ArrayList<PatientAddress>();
		}
		this.tableModel.setData(c);
		try {
			if (addressTable.getRowCount() > 0)
				this.addressTable.setRowSelectionInterval(0,0);
			else
				this.currentAddress = new PatientAddress();
		} catch (Exception e) {
			logger.error("Error while setting patient address info", e);
			this.currentAddress = new PatientAddress();
		}
	}

	/* (non-Javadoc)
	 * @see lu.tudor.santec.gecamed.patient.gui.administrative.IPatientDataPanel#getPatient(lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient)
	 */
	@SuppressWarnings("unchecked")
	public void getPatient(Patient patient) {
		ArrayList<PatientAddress> al = new ArrayList<PatientAddress>();
		for (Iterator iter = tableModel.getData().iterator(); iter.hasNext();) {
			PatientAddress addr = (PatientAddress) iter.next();
			if ( addr.getType() == null && (addr.getLocality() == null || addr.getLocality() == "") )
				continue;
			al.add(addr);
		}
		patient.setAddress(new HashSet(al));
	}

	/**
	 * ListCellRenderer for the Addresses
	 * 
	 *
	 * @author Johannes Hermen johannes.hermen(at)tudor.lu
	 *
	 * @version
	 * <br>$Log: PatientAddressInfo.java,v $
	 * <br>Revision 1.51  2013-12-27 18:08:00  donak
	 * <br>Cleanup of imports
	 * <br>
	 * <br>Revision 1.50  2013-07-15 06:18:34  ferring
	 * <br>logging changed
	 * <br>
	 * <br>Revision 1.49  2009-05-29 13:51:21  hermen
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.48  2009-03-10 09:51:42  hermen
	 * <br>small bugfixes
	 * <br>
	 * <br>Revision 1.47  2009-01-06 14:48:08  hermen
	 * <br>small bugfixes
	 * <br>
	 * <br>Revision 1.46  2008-09-25 09:43:06  heinemann
	 * <br>fixed copyrights
	 * <br>
	 * <br>Revision 1.45  2008-06-30 08:16:45  hermen
	 * <br>added address autocompletion by locality
	 * <br>
	 * <br>Revision 1.44  2008-06-04 12:54:01  hermen
	 * <br>fixed copy patient address to assure address
	 * <br>
	 * <br>Revision 1.43  2008-04-16 14:24:28  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.42  2008-04-15 16:01:03  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.41  2008-04-09 08:53:42  hermen
	 * <br>fixed small possible bugs
	 * <br>
	 * <br>Revision 1.40  2008-03-14 08:13:23  hermen
	 * <br>enhanced focus traversal
	 * <br>
	 * <br>Revision 1.39  2008-01-21 14:03:27  hermen
	 * <br>updated Javadoc and refactured code
	 * <br>
	 *
	 */
	class AddressTypeListRenderer extends DefaultListCellRenderer {
		private static final long serialVersionUID = 1L;

		public Component getListCellRendererComponent(JList list, Object value,
				int index, boolean isSelected, boolean cellHasFocus) {
			try {
				if (value.toString().equals(AddressType.PRIVATE)) {
					this.setIcon(AddressTypeRenderer.PRIVAT);
				} else if (value.toString().equals(AddressType.WORK)) {
					this.setIcon(AddressTypeRenderer.WORK);
				} else	if (value.toString().equals(AddressType.BILLING)) {
					this.setIcon(AddressTypeRenderer.BILLING);
				} else	if (value.toString().equals(AddressType.HIS)) {
					this.setIcon(AddressTypeRenderer.HIS);
				}
			} catch (Exception e) {
				this.setIcon(null);
			}
			return this;
		}
	}

	/**
	 * TableModel for the patients addresses
	 * 
	 *
	 * @author Johannes Hermen johannes.hermen(at)tudor.lu
	 *
	 * @version
	 * <br>$Log: PatientAddressInfo.java,v $
	 * <br>Revision 1.51  2013-12-27 18:08:00  donak
	 * <br>Cleanup of imports
	 * <br>
	 * <br>Revision 1.50  2013-07-15 06:18:34  ferring
	 * <br>logging changed
	 * <br>
	 * <br>Revision 1.49  2009-05-29 13:51:21  hermen
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.48  2009-03-10 09:51:42  hermen
	 * <br>small bugfixes
	 * <br>
	 * <br>Revision 1.47  2009-01-06 14:48:08  hermen
	 * <br>small bugfixes
	 * <br>
	 * <br>Revision 1.46  2008-09-25 09:43:06  heinemann
	 * <br>fixed copyrights
	 * <br>
	 * <br>Revision 1.45  2008-06-30 08:16:45  hermen
	 * <br>added address autocompletion by locality
	 * <br>
	 * <br>Revision 1.44  2008-06-04 12:54:01  hermen
	 * <br>fixed copy patient address to assure address
	 * <br>
	 * <br>Revision 1.43  2008-04-16 14:24:28  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.42  2008-04-15 16:01:03  heinemann
	 * <br>*** empty log message ***
	 * <br>
	 * <br>Revision 1.41  2008-04-09 08:53:42  hermen
	 * <br>fixed small possible bugs
	 * <br>
	 * <br>Revision 1.40  2008-03-14 08:13:23  hermen
	 * <br>enhanced focus traversal
	 * <br>
	 * <br>Revision 1.39  2008-01-21 14:03:27  hermen
	 * <br>updated Javadoc and refactured code
	 * <br>
	 *
	 */
	class PatientAddressTableModel extends AbstractTableModel {

		private static final long serialVersionUID = 1L;
		private ArrayList<PatientAddress> data;
		
		private DateFormat df = GECAMedGuiUtils.getDateFormat(false);

		private String[] columnName = {
				Translatrix.getTranslationString("pm.address.type"),
				Translatrix.getTranslationString("pm.address.streetname"),
				Translatrix.getTranslationString("pm.address.locality"),
				Translatrix.getTranslationString("pm.address.country"),
				Translatrix.getTranslationString("pm.address.date")
		};
		@SuppressWarnings("unchecked")
		private Class[] columnsClasses = {ImageIcon.class, String.class, String.class, String.class, String.class};

		@SuppressWarnings("unchecked")
		public void setData(Collection<PatientAddress> data) {
			if (data != null) {
				this.data = new ArrayList(data);
			} else {
				this.data = null;
			}
			this.fireTableDataChanged();
		}

		public ArrayList<PatientAddress> getData() {
			return data;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.AbstractTableModel#getColumnName(int)
		 */
		public String getColumnName(int column) {
			return columnName[column];
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		public int getRowCount() {
			try {
				return data.size();
			} catch (Exception e) {
				return 0;
			}
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		public int getColumnCount() {
			return columnsClasses.length;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.AbstractTableModel#getColumnClass(int)
		 */
		public Class<?> getColumnClass(int column) {
			return columnsClasses[column];
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		public Object getValueAt(int rowIndex, int columnIndex) {
			PatientAddress addr = data.get(rowIndex);
			try {
				switch (columnIndex) {
				case 0:
					if (addr.getType().equals(AddressType.PRIVATE)) {
						return AddressTypeRenderer.PRIVAT;
					} else if (addr.getType().equals(AddressType.WORK)) {
						return AddressTypeRenderer.WORK;
					}	else if (addr.getType().equals(AddressType.BILLING)) {
						return AddressTypeRenderer.BILLING;
					} else	if (addr.getType().equals(AddressType.HIS)) {
						return AddressTypeRenderer.HIS;
					}
					return "";
				case 1:
					if (addr.getStreetNumber() == null || addr.getStreetNumber().equals("") )
						return addr.getStreetName();
					return addr.getStreetNumber() + ", " + addr.getStreetName();
				case 2:
					if (addr.getZip() == null && addr.getLocality() == null ) {
						return "";
					}
					return addr.getZip() + " " + addr.getLocality();
				case 3:
					if (Translatrix.getTranslationString("Country."  + addr.getCountry()).equals("Country."  + addr.getCountry())) {
						return addr.getCountry();
					}
					return Translatrix.getTranslationString("Country."  + addr.getCountry());
				case 4:
					return (addr.getDate() != null?df.format(addr.getDate()):"");
				default:
					return "";
				}
			} catch (Exception e) {
				return "";
			}
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.AbstractTableModel#isCellEditable(int, int)
		 */
		public boolean isCellEditable(int rowIndex, int columnIndex) {
			return false;
		}

		public PatientAddress getAddress(int row) {
			try {
				return data.get(row);
			} catch (Exception e) {
				return null;
			}

		}

		public void setAddress(int row, PatientAddress address) {
			try {
				this.data.set(row, address);
			} catch (Exception e) {
				logger.log(Level.WARN, "set Address failed", e);
			}
			fireTableRowsUpdated(row, row);
		}
	}

	/* (non-Javadoc)
	 * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
	 */
	public void propertyChange(PropertyChangeEvent evt) {
		if (listenersActivated && evt.getPropertyName().equals(AddressPanel.UNSAVED_CHANGES) ) {
			updateAddressInTable();
		}
	}


	private void updateAddressInTable() {
		getAddress(currentAddress);
		if (tableModel.getData().size() == 0) {
			tableModel.getData().add(currentAddress);
			tableModel.fireTableDataChanged();
			addressTable.setRowSelectionInterval(0,0);
		}
		tableModel.setAddress(addressTable.getSelectedRow(), currentAddress);
		this.patientEditPanel.setModified(true);
	}


	/**
	 * enables/disables the panel
	 * @param editable
	 */
	private void setEditable(boolean editable) {

		this.type.setEnabled(editable);
    	this.addressPanel.setEditable(editable);
    	this.newButton.setEnabled(editable);
    	this.deleteButton.setEnabled(editable);
	}

	public List<Component> getComponentsForFocus() {
		/* ====================================================== */
		List<Component> list = new ArrayList<Component>();
		/* ------------------------------------------------------- */
		list.addAll(addressPanel.getComponentsForFocus());
		/* ------------------------------------------------------- */
		return list;
		/* ====================================================== */
	}

	public boolean isFilledOK() {
	    return true;
	}


}
