/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JToggleButton;

import lu.tudor.santec.gecamed.core.gui.utils.GradientFactory;

/**
 * creates a buttonpanel to show on a PatientNaviBar. Thi s can be done in several days. Add a panel,
 * add single buttons, add actions etc
 * 
 * @author Johannes Hermen  johannes.hermen(at)tudor.lu
 * @author Martin Heinemann martin.heinemann(at)tudor.lu
 *
 */
public class PatientButtonPanel extends JPanel {

	private static final long serialVersionUID = 1L;
	protected JPanel content;
	private ButtonGroup buttonGroup = new ButtonGroup();
	private int columns; 
	private HashMap<Action, AbstractButton> map = new HashMap<Action, AbstractButton>();

	/**
	 * creates a new buttonpanel
	 * @param title the title of the buttonpanel
	 * @param color the background color of the title of the buttonpanel
	 * @param columns the number of button columns
	 * @param buttons the buttons to show on the buttonpanel
	 */
	public PatientButtonPanel(String title, Color color, int columns, Vector<AbstractButton> buttons) {
		createPanel(title, color, columns, buttons);
	}
	
	
	/**
	 * creates a new panel for the navibar.
	 * Be warned, if you use this constructor all the other methods
	 * of this class might not work! Just play on your panel!
	 * 
	 * @param title title of the button panel
	 * @param color background color for the title
	 * @param panel panel to display below the title
	 */
	public PatientButtonPanel(String title, Color color, JPanel panel) {
		/* ================================================== */
		this.setLayout(new BorderLayout(0,2));
		this.setOpaque(false);
		
		JLabel label = new JLabel(" " + title + ":");
		label.setOpaque(true);
		label.setPreferredSize(new Dimension(20,20));
		
		label.setBackground(color);
		this.add(GradientFactory.makeGradient(label, color), BorderLayout.NORTH);
		content = panel;
		content.setOpaque(false);
		
		this.add(content, BorderLayout.CENTER);
		/* ================================================== */
	}
	
	/**
	 * creates a new buttonpanel
	 * @param title the title of the buttonpanel
	 * @param color the background color of the title of the buttonpanel
	 * @param columns the number of button columns
	 * @param actions the actions to show on the buttonpanel
	 * @param toggleActions show actions as togglebuttons
	 */
	public PatientButtonPanel(String title, Color color, int columns, Vector<Action> actions, boolean toggleActions) {
		Vector<AbstractButton> buttons = new Vector<AbstractButton>();
		for (Iterator<Action> iter = actions.iterator(); iter.hasNext();) {
			Action action = iter.next();
			if (toggleActions) {
				buttons.add(new JToggleButton(action));
			} else {
				buttons.add(new JButton(action));
			}
		}
		createPanel(title, color, columns, buttons);
	}
	
	/**
	 * creates a new buttonpanel
	 * @param title the title of the buttonpanel
	 * @param color the background color of the title of the buttonpanel
	 * @param columns the number of button columns
	 * @param buttons the buttons to show on the buttonpanel
	 */
	private void createPanel (String title, Color color, int columns, Vector<AbstractButton> buttons) {
		/* ================================================== */
		this.columns = columns;
		this.setLayout(new BorderLayout(0,2));
		this.setOpaque(false);
		
		JLabel label = new JLabel(" " + title + ":");
		label.setOpaque(true);
		label.setPreferredSize(new Dimension(20,20));
		
		label.setBackground(color);
		this.add(GradientFactory.makeGradient(label, color), BorderLayout.NORTH);
		content = new JPanel(new GridLayout(0,columns, 2, 2));
		content.setOpaque(false);
		
		for (Iterator<AbstractButton> iter = buttons.iterator(); iter.hasNext();) {
			AbstractButton element = iter.next();
			addButton(element);
		}
		this.add(content, BorderLayout.CENTER);
		/* ================================================== */
	}
	
	public void setButtons (Vector<AbstractButton> buttons) {
		content.removeAll();
		for (Iterator<AbstractButton> iter = buttons.iterator(); iter.hasNext();) {
			AbstractButton element = iter.next();
			addButton(element);
		}
		content.validate();
	}
	
	/**
	 * adds a button to the panel
	 * @param jb the Button to add
	 */
	public void addButton(AbstractButton jb)  {
		if (columns == 1 ) jb.setHorizontalAlignment(JButton.LEFT);
		else {
			jb.setMargin(new java.awt.Insets(2, 2, 2, 2));
		}
		content.add(jb);
	}
	
	
	/**
	 * adds a button to the panel
	 * @param jb the Button to add
	 */
	public void addToggleButton(JToggleButton jb)  {
		if (columns == 1 ) jb.setHorizontalAlignment(JButton.LEFT);
		else {
			jb.setMargin(new java.awt.Insets(2, 2, 2, 2));
		}
		content.add(jb);
		buttonGroup.add(jb);
	}
	
	public void addComponent(JComponent component) {
		/* ====================================================== */
		content.add(component);
		/* ====================================================== */
	}
	
	
	/**
	 * removes a button from the panel
	 * @param jb the Button to remove
	 */
	public void removeButton(AbstractButton jb) {
		content.remove(jb);
		if (jb instanceof JToggleButton) {
			buttonGroup.remove(jb);
		}
	}
	
	/**
	 * adds an action to the panel as button
	 * @param a the action to add
	 */
	public void addAction(Action a) {
		JButton b = new JButton(a);
		map.put(a, b);
		addButton(b);
	}
	
	/**
	 * adds an action to the panel as togglebutton
	 * @param a the action to add
	 */
	public void addToggleAction(Action a) {
		JToggleButton b = new JToggleButton(a);
		map.put(a, b);
		addButton(b);
	}
	
	/**
	 * removes an action from the panel
	 * @param a
	 */
	public void removeAction(Action a) {
		AbstractButton b = map.get(a);
		map.remove(a);
		removeButton(b);
	}
	
	@Override
	public Component[] getComponents() {
		return content.getComponents();
	}
	
}
