/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.gui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;

import lu.tudor.santec.gecamed.agenda.ejb.entity.beans.AgendaCalendar;
import lu.tudor.santec.gecamed.agenda.ejb.entity.beans.Appointment;
import lu.tudor.santec.gecamed.agenda.ejb.entity.beans.AppointmentType;
import lu.tudor.santec.gecamed.agenda.ejb.session.beans.AppointmentManagerBean;
import lu.tudor.santec.gecamed.agenda.ejb.session.interfaces.AppointmentManager;
import lu.tudor.santec.gecamed.agenda.gui.AgendaAdminSettingsPlugin;
import lu.tudor.santec.gecamed.agenda.gui.AgendaModule;
import lu.tudor.santec.gecamed.agenda.gui.DefaultNamedCalendar;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.IconFetcher;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.LineColorCellRenderer;
import lu.tudor.santec.gecamed.core.gui.widgets.GECAMedBaseDialog;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.gecamed.office.ejb.session.beans.OfficeManagerBean;
import lu.tudor.santec.gecamed.office.ejb.session.interfaces.OfficeManagerInterface;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.waitingroom.ejb.entity.beans.Queue;
import lu.tudor.santec.gecamed.waitingroom.ejb.entity.beans.Room;
import lu.tudor.santec.gecamed.waitingroom.ejb.session.beans.WaitingroomManagerBean;
import lu.tudor.santec.gecamed.waitingroom.ejb.session.interfaces.WaitingroomManager;
import lu.tudor.santec.gecamed.waitingroom.gui.WaitingroomModule;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;

import bizcal.common.Event;
import bizcal.util.DateUtil;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * Dialog to show all visits of one patient. The past visits are selected from the table waitingroom.queue and
 * the future visits are selected from the table agenda.appointment in the database.
 * 
 * @author Thorsten Roth thorsten.roth(at)tudor.lu
 *
 * @version
 * <br>$Log: OverviewPatientVisitsDialog.java,v $
 * <br>Revision 1.5  2013-12-27 18:09:16  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.4  2013-07-15 06:18:35  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.3  2012-10-04 13:41:07  troth
 * <br>Ticket #1000
 * <br>
 * <br>Revision 1.2  2012-10-04 12:21:30  troth
 * <br>Ticket #1000 and parts of ticket #530.
 * <br>
 * <br>Revision 1.1  2012-10-03 12:24:33  troth
 * <br>Implement ticket #1000.
 * <br>
 */

public class OverviewPatientVisitsDialog extends GECAMedBaseDialog implements ActionListener, MouseListener, KeyListener, ListSelectionListener {

	private static final long serialVersionUID = 1L;
	
	/**
	 * static logger for this class
	 */
	private static Logger logger = Logger
			.getLogger(OverviewPatientVisitsDialog.class.getName());
	
	private static final int ROWHEIGHT = 30;
	
	private JTable 					table;
	private PatientVisitsTableModel tableModel;
	private JPanel 					mainPanel;
	private JLabel 					titleLabel;
	private JScrollPane 			jsp;
	
	
	/**
	 * Constructor
	 */
	public OverviewPatientVisitsDialog()
	{
		super(Translatrix.getTranslationString("overviewPatientVisitsDialog.title"));

		this.mainPanel = new JPanel();
		this.mainPanel.setBackground(GECAMedColors.c_GECAMedBackground);
		
		CellConstraints cc = new CellConstraints();
		this.mainPanel.setLayout(new FormLayout(
				"3dlu, pref:grow, 3dlu",
				"3dlu, fill:pref, 3dlu, fill:200dlu:grow, 3dlu"));

		titleLabel = new JLabel(Translatrix.getTranslationString("overviewPatientVisitsDialog.tapTitle"));
		this.mainPanel.add(titleLabel, cc.xy(2,2));
		
		this.tableModel = new PatientVisitsTableModel();
		this.table = new JTable(tableModel);
		this.table.addMouseListener(this);
		PatientVisitsRenderer renderer = new PatientVisitsRenderer(this.tableModel);
		this.table.getColumnModel().getColumn(0).setCellRenderer(renderer);
		this.table.getColumnModel().getColumn(1).setCellRenderer(renderer);
		this.table.getColumnModel().getColumn(2).setCellRenderer(renderer);
		this.table.getColumnModel().getColumn(3).setCellRenderer(renderer);
		this.table.getColumnModel().getColumn(4).setCellRenderer(renderer);
		this.table.getColumnModel().getColumn(0).setMinWidth(50);
		this.table.getColumnModel().getColumn(0).setMaxWidth(50);
		this.table.getColumnModel().getColumn(1).setMinWidth(50);
		this.table.getColumnModel().getColumn(1).setMaxWidth(50);
		this.table.getColumnModel().getColumn(2).setMinWidth(90);
		this.table.getColumnModel().getColumn(2).setMaxWidth(90);
		this.table.getColumnModel().getColumn(3).setMinWidth(70);
		this.table.getColumnModel().getColumn(3).setMaxWidth(70);
//		this.table.getColumnModel().getColumn(3).setMinWidth(130);
//		this.table.getColumnModel().getColumn(3).setMaxWidth(130);
		this.table.setRowHeight(ROWHEIGHT);
		this.table.setAutoscrolls(true);
		this.table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		this.table.addKeyListener(this);
		// selection listener to switch the state
		this.table.getSelectionModel().addListSelectionListener(this);
		jsp = new JScrollPane(this.table);
		jsp.setOpaque(false);
		jsp.getViewport().setOpaque(false);

		this.mainPanel.add(jsp, cc.xy(2,4));
		this.add(mainPanel);
		
		this.okButton.addActionListener(this);
		this.cancelButton.addActionListener(this);
		//this.setSize(800,600);
		this.pack();
	}
	
	/**
	 * Shows the Dialog and init the table module
	 */
	public void showDialog()
	{		
		
		Patient currentPatient = GECAMedModule.getCurrentPatient();
		
		String titleTxt = Translatrix.getTranslationString("overviewPatientVisitsDialog.tapTitle")
			+ ": "
			+ currentPatient.getFirstName()
			+ " "
			+ currentPatient.getSurName()
			+ " ("
			+ currentPatient.getSocialSecurityNumber()
			+ ")";
		
		this.titleLabel.setText(titleTxt);
		
		// fill data table
		// get past visits of patient
		WaitingroomManager manager = (WaitingroomManager) ManagerFactory.getRemote(WaitingroomManagerBean.class);
		List<Queue> queues = manager.getQueues4Patient(currentPatient.getId());
		ArrayList<PatientVisit> data = new ArrayList<PatientVisit>();
		
		if (queues != null)
		{
			for (Queue queue : queues)
		    {	
		    	if (queue != null)
		    	{
		    		PatientVisit pateintVisit = new PatientVisit();
		    		Room room = manager.getRoomById(queue.getRoomId());
		    		
		    		pateintVisit.setQueueID(queue.getId());
		    		
		    		if(queue.getIsPresent() || queue.getTreated()) pateintVisit.setDone(true);
		    		else pateintVisit.setDone(false);
		    		
		    		pateintVisit.setDate(queue.getStartDate());
		    		pateintVisit.setWhoWhat(room.getName());
		    		
		    		if(queue.getDescription() != null)
		    			pateintVisit.setDescription(queue.getDescription().replace("\n", " "));
		    		else 
		    			pateintVisit.setDescription("");
		    		
		    		if(room != null)
		    		{
		    			OfficeManagerInterface officeManagerInterface = (OfficeManagerInterface) ManagerFactory.getRemote(OfficeManagerBean.class);
		    			Physician physician = null;
						try {
							if (room.getPhysicianId() != null) {
								physician = officeManagerInterface.getPhysician(room.getPhysicianId());								
								if(physician != null)
									pateintVisit.setWhoWhat(physician.getFirstName() + " " + physician.getName());
							}
						} catch (Exception e) {
							logger.warn("Error getting Phsician", e);
						}
		    			
		    		}
		    		data.add(pateintVisit);
		    		
		    	}
		    }
			
		}
		
		// get future visits of patient
		AppointmentManager aManager = (AppointmentManager) 
		ManagerFactory.getRemote(AppointmentManagerBean.class);
		List<Appointment> appointments = aManager.getAppointments(currentPatient.getId(), 20);
		if (appointments != null)
		{
			// convert to Event objects
			List<Event> futureApps = new ArrayList<Event>();
			// create a limit data, lets say from to day plus 400 days
			Date destination = DateUtil.getDiffDay(new Date(), 400);
			
			for (Appointment app : appointments) {
				/* ------------------------------------------------------- */
				List<Event> tList = DefaultNamedCalendar.createRecurrEvents(app, new Date(), destination,(Locale)AgendaModule.getInstance().getAgendaAdminSettingsPlugin().getValue(AgendaAdminSettingsPlugin.CALENDAR_LOCALE));
				// set the color to the events
				for (Event tE : tList)
				{
					if(tE.getStart().getTime() >= new Date().getTime())
					{
					  tE.setColor(new Color(AgendaModule.getCalendarForId(app.getCalendarId()).getColor()));
					  // set Calendar id to Event
					  tE.set(Event.CALENDAR_ID, app.getCalendarId());
					  // set Calendar isBackground to Event
					  tE.set(Event.CALENDAR_IS_BACKGROUND, AgendaModule.getCalendarForId(app.getCalendarId()).isBackground());
					  futureApps.add(tE);
					}
				}
				
				/* ------------------------------------------------------- */
				//futureApps.addAll(tList);
				/* ------------------------------------------------------- */
			}
			/* ------------------------------------------------------- */
			Collections.sort(futureApps);
			
			for (Event appointment : futureApps)
		    {	
		    	if (appointment != null)
		    	{
		    		PatientVisit pateintVisit = new PatientVisit();
		    		
		    		Appointment app = (Appointment) appointment.getId();

		    		Integer calendarId = (Integer) appointment.get(Event.CALENDAR_ID);
		    		AgendaCalendar agendaCalendar = aManager.getCalendar(calendarId);
		    		
		    		pateintVisit.setAppointmentID(app.getId());
		    		pateintVisit.setDone(false);
		    		pateintVisit.setDate(appointment.getStart());
		    		pateintVisit.setWhoWhat(agendaCalendar.getTitle());
		    		
		    		if(appointment.getDescription() != null)
		    			pateintVisit.setDescription(appointment.getDescription().replace("\n", " "));
		    		else
		    			pateintVisit.setDescription("");
		    		
		    		OfficeManagerInterface officeManagerInterface = (OfficeManagerInterface) ManagerFactory.getRemote(OfficeManagerBean.class);
	    			Physician physician;
					try {
						if (agendaCalendar.getPhysicianId() != null) {
							physician = officeManagerInterface.getPhysician(agendaCalendar.getPhysicianId());
							if(physician != null)
								pateintVisit.setWhoWhat(physician.getFirstName() + " " + physician.getName());							
						}
					} catch (Exception e) {
						logger.warn("Error getting Phsician", e);
					}
		    		data.add(pateintVisit);
		    	}
		    }
		}
		
		tableModel.setPateintVisits(data);
		
		// set scollbar position
		int scrollvalue = jsp.getVerticalScrollBar().getMinimum();
		if(queues != null) scrollvalue = scrollvalue + (queues.size()-2) * ROWHEIGHT;
		jsp.getViewport().setViewPosition(new java.awt.Point(0, scrollvalue)); 
		
		MainFrame.showDialogCentered(this);
//		this.setVisible(true);
		/* ================================================== */
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		/* ================================================== */
		if (e.getSource().equals(this.cancelButton))
		{
			this.setVisible(false);
		} else if (e.getSource().equals(this.okButton))
		{
			this.setVisible(false);
		}
		/* ================================================== */
	}
	
	private PatientVisit getSelectedVisit() {
		int row = this.table.getSelectedRow();
		return this.tableModel.getPateintVisit(row);
	}

	public void mouseClicked(MouseEvent e) {
		if (e.getClickCount() >= 2) {
			try {
				new Thread() {
					public void run() {
						PatientVisit visit = getSelectedVisit();
						if (visit != null) {
							if (visit.getAppointmentID() != null) {
								setVisible(false);
								AgendaModule.editAppointment(visit.getAppointmentID());
							} else if (visit.getQueueID() != null) {
								if (DateUtil.isSameDay(visit.getDate(), new Date())) {
									setVisible(false);
									WaitingroomModule.editQueue(visit.getQueueID());							
								}
							}
						}						
					}
				}.start();
			} catch (Exception ee) {}
		}
	}

	public void mouseEntered(MouseEvent e) {}

	public void mouseExited(MouseEvent e) {}

	public void mousePressed(MouseEvent e) {}

	public void mouseReleased(MouseEvent e) {}
	
	public void keyPressed(KeyEvent e) {}

	public void keyReleased(KeyEvent e) {}
	
	public void keyTyped(KeyEvent e) {}
	
	// ========================================================================================
	// selection listener method for the JTable
	// the state will be switched
	// ========================================================================================

	public void valueChanged(ListSelectionEvent e) {
		/* ====================================================== */
		table.scrollRectToVisible(table.getCellRect(table.getSelectedRow(),0,true));
		/* ====================================================== */
	}
	
	
	/**
	 * @author Thorsten Roth thorsten.roth(at)tudor.lu
	 *
	 * @version
	 * <br>$Log: OverviewPatientVisitsDialog.java,v $
	 * <br>Revision 1.5  2013-12-27 18:09:16  donak
	 * <br>Cleanup of imports
	 * <br>
	 * <br>Revision 1.4  2013-07-15 06:18:35  ferring
	 * <br>logging changed
	 * <br>
	 * <br>Revision 1.3  2012-10-04 13:41:07  troth
	 * <br>Ticket #1000
	 * <br>
	 * <br>Revision 1.2  2012-10-04 12:21:30  troth
	 * <br>Ticket #1000 and parts of ticket #530.
	 * <br>
	 * <br>Revision 1.1  2012-10-03 12:24:33  troth
	 * <br>Implement ticket #1000.
	 * <br>
	 */
	public class PatientVisitsTableModel extends AbstractTableModel  {

		private static final long serialVersionUID = 1L;
		
		private ArrayList<PatientVisit> data = new ArrayList<PatientVisit>();
		
		private String[] columnNames = {
		    Translatrix.getTranslationString("overviewPatientVisitsDialog.columnNames.typ"),
			Translatrix.getTranslationString("overviewPatientVisitsDialog.columnNames.done"),
			Translatrix.getTranslationString("overviewPatientVisitsDialog.columnNames.date"),
			Translatrix.getTranslationString("overviewPatientVisitsDialog.columnNames.time"),
			Translatrix.getTranslationString("overviewPatientVisitsDialog.columnNames.whoWhat"),
			Translatrix.getTranslationString("overviewPatientVisitsDialog.columnNames.description")
		};
		
		private Class<?>[] columnClasses = {
		    Boolean.class,
			Boolean.class,
			String.class,
			String.class,
			String.class,
			String.class
		};
		

		public int getColumnCount() {
			return columnNames.length;
		}
		
		@Override
		public Class<?> getColumnClass(int columnIndex) {
			return columnClasses[columnIndex];
		}

		public int getRowCount() {
			return data.size();
		}

		@Override
		public String getColumnName(int column) {
			return columnNames[column];
		}

		public Object getValueAt(int rowIndex, int columnIndex) {
			
			PatientVisit e = data.get(rowIndex);
			if (e == null) 
				return null;
			
			switch (columnIndex) {
			case 0:
				return (e.getAppointmentID() != null);
			case 1:
				return e.getDone();
				
			case 2:
				SimpleDateFormat sdf = new SimpleDateFormat("dd.MM.yyyy");
				return sdf.format(e.getDate());
				
			case 3:
				SimpleDateFormat stf = new SimpleDateFormat("HH:mm");
				return stf.format(e.getDate());
				
			case 4:
				return e.getWhoWhat();
				
			case 5:
				return e.getDescription();
				
			default:
				break;
			}
			return null;
		}
		
		public void setPateintVisits(Collection<PatientVisit> app) {
			if (app != null)
				this.data = new ArrayList<PatientVisit>(app);
			else
				this.data = new ArrayList<PatientVisit>();
			fireTableDataChanged();
		}

		public PatientVisit getPateintVisit(int row) {
			try {
				return data.get(row);				
			} catch (Exception e) {
			}
			return null;
		}


	}

	
	/**
	 * @author Thorsten Roth thorsten.roth(at)tudor.lu
	 *
	 * @version
	 * <br>$Log: OverviewPatientVisitsDialog.java,v $
	 * <br>Revision 1.5  2013-12-27 18:09:16  donak
	 * <br>Cleanup of imports
	 * <br>
	 * <br>Revision 1.4  2013-07-15 06:18:35  ferring
	 * <br>logging changed
	 * <br>
	 * <br>Revision 1.3  2012-10-04 13:41:07  troth
	 * <br>Ticket #1000
	 * <br>
	 * <br>Revision 1.2  2012-10-04 12:21:30  troth
	 * <br>Ticket #1000 and parts of ticket #530.
	 * <br>
	 * <br>Revision 1.1  2012-10-03 12:24:33  troth
	 * <br>Implement ticket #1000.
	 * <br>
	 */
	public class PatientVisitsRenderer extends LineColorCellRenderer {

		private static final long serialVersionUID = 1L;

		private PatientVisitsTableModel model;
		
		public PatientVisitsRenderer(PatientVisitsTableModel listModel) {
			this.model = listModel;
			this.setFont(new Font("Arial", Font.PLAIN, 12));
		}

		public Component getTableCellRendererComponent(JTable p_Table, Object value, boolean p_IsSelected, boolean hasFocus, int p_Row, int p_Column) {
			 
//			AgendaCalendar cal = AgendaModule.getCalendarForId(app.getCalendarId());
			Color c = Color.WHITE;
//			try {
//				c = new Color(cal.getColor());		
//				if (new Date().before(app.getEndDate()))
//					c = new Color(c.getRed(),c.getGreen(),c.getBlue(),120);
//				else
//					c = new Color(c.getRed(),c.getGreen(),c.getBlue(),20);
//
//			} catch (Exception ee) {
//			}

			
			switch (p_Column) {
			case 0:
				this.setVerticalAlignment(JLabel.CENTER);
				this.setHorizontalAlignment(JLabel.CENTER);
				this.setText("");
				if((Boolean) value)
				{
					this.setIcon(AgendaModule.getMiniIcon(AgendaModule.ICON_AGENDA));
				}else{
					this.setIcon(WaitingroomModule.getMiniIcon(WaitingroomModule.WAITINGROOM_ICON));
				}
				break;
			case 1:
				this.setVerticalAlignment(JLabel.CENTER);
				this.setHorizontalAlignment(JLabel.CENTER);
				this.setText("");
				if((Boolean) value)
				{
					this.setIcon(IconFetcher.getMiniIcon(
						PatientManagerIconNames.class,
						PatientManagerIconNames.CHECKBOX_SELECTED));
				}else{
					this.setIcon(IconFetcher.getMiniIcon(
							PatientManagerIconNames.class,
							PatientManagerIconNames.CHECKBOX_NOT_SELECTED));
				}
				break;
			case 2:
				this.setVerticalAlignment(JLabel.CENTER);
				this.setHorizontalAlignment(JLabel.CENTER);
				this.setText((String) value);
				this.setIcon(null);
				try {
					AppointmentType at = AgendaModule.getTypeForId((Integer)value);
					this.setIcon(AgendaModule.getMiniIcon(at.getIcon()));					
				} catch (Exception ee) {
				}
				break;
			case 3:
				this.setVerticalAlignment(JLabel.CENTER);
				this.setHorizontalAlignment(JLabel.CENTER);
				this.setText((String) value);
				try {
//					this.setText(cal.getTitle());					
				} catch (Exception ee) {}
				this.setIcon(null);
				break;
			default:
				this.setVerticalAlignment(JLabel.CENTER);
				this.setHorizontalAlignment(JLabel.CENTER);
				this.setText((String) value);
				this.setIcon(null);
			    if (p_Table.getRowHeight(p_Row) < this.getPreferredSize().height) {
			    	p_Table.setRowHeight(p_Row, this.getPreferredSize().height + 4);
			    }
			    break;
			}

			if (p_IsSelected)
				this.setBackground(p_Table.getSelectionBackground());
			else
				this.setBackground(c);
			return this;
		}

	}
	
	
	/**
	 * @author Thorsten Roth thorsten.roth(at)tudor.lu
	 *
	 * @version
	 * <br>$Log: OverviewPatientVisitsDialog.java,v $
	 * <br>Revision 1.5  2013-12-27 18:09:16  donak
	 * <br>Cleanup of imports
	 * <br>
	 * <br>Revision 1.4  2013-07-15 06:18:35  ferring
	 * <br>logging changed
	 * <br>
	 * <br>Revision 1.3  2012-10-04 13:41:07  troth
	 * <br>Ticket #1000
	 * <br>
	 * <br>Revision 1.2  2012-10-04 12:21:30  troth
	 * <br>Ticket #1000 and parts of ticket #530.
	 * <br>
	 * <br>Revision 1.1  2012-10-03 12:24:33  troth
	 * <br>Implement ticket #1000.
	 * <br>
	 */
	public class PatientVisit
	{
		private Boolean done;
		
		private Date date;
		
		private String whoWhat;
		
		private String description;
		
		private Integer appointmentID;
		
		private Integer queueID;
		
		public PatientVisit(){
			
		}
		
		public Boolean getDone()
		{
			return done;
		}
		
		public void setDone(Boolean done)
		{
			this.done = done;
		}
		
		public Date getDate()
		{
			return date;
		}
		
		public void setDate(Date date)
		{
			this.date = date;
		}
		
		public String getWhoWhat()
		{
			return whoWhat;
		}
		
		public void setWhoWhat(String whoWhat)
		{
			this.whoWhat = whoWhat;
		}
		
		public String getDescription()
		{
			return description;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}

		public Integer getAppointmentID() {
			return appointmentID;
		}

		public void setAppointmentID(Integer appointmentID) {
			this.appointmentID = appointmentID;
		}

		public Integer getQueueID() {
			return queueID;
		}

		public void setQueueID(Integer queueID) {
			this.queueID = queueID;
		}
	}
}
