/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.ejb.entity.beans;

import java.io.Serializable;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.NamedQuery;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;
import lu.tudor.santec.gecamed.core.ejb.entity.beans.Gender;
import lu.tudor.santec.gecamed.core.utils.SSNChecker;
import lu.tudor.santec.gecamed.esante.utils.InfoString;
import lu.tudor.santec.i18n.Translatrix;

/**
 * Entity Bean that maps the <b>patient.patient_stub</b> table.
 * used for the patient search
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @version
 * <br>$Log: PatientStub.java,v $
 * <br>Revision 1.10  2013-06-10 08:22:09  ferring
 * <br>eSante POC
 * <br>
 * <br>Revision 1.9  2013-05-15 12:56:26  ferring
 * <br>old search string won't be stored in DB any longer. This caused wrong results in the patient search.
 * <br>
 * <br>Revision 1.8  2013-04-09 07:37:28  ferring
 * <br>Patient search unaccented
 * <br>
 * <br>Revision 1.7  2013-02-14 16:28:48  troth
 * <br>Change the export search data for patient to patientStub
 * <br>
 * <br>Revision 1.6  2012-11-28 16:30:11  troth
 * <br>New PatientAdminBean function to get patient by over his RIS-Id.
 * <br>
 * <br>Revision 1.5  2008-10-24 08:37:15  hermen
 * <br>Fixed Bug #230
 * <br>Sort Patients in the Patient List by Lastname, Firstname
 * <br>Lastname works, bus Firstname as 2nd sorting is not used....
 * <br>
 * <br>Revision 1.4  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.3  2008-04-28 08:05:48  hermen
 * <br>changed patient search to use patient_stub view and bean
 * <br>
 *
 */
@javax.persistence.NamedQueries( {
	@NamedQuery(name = "findAllPatientStub", query = "SELECT OBJECT(o) FROM PatientStub o ORDER BY o.surName, o.firstName"),
	@NamedQuery(name = "findPatientStubBySSN", query = "SELECT OBJECT(o) FROM PatientStub o WHERE socialSecurityNumber like :socialSecurityNumber ORDER BY o.surName, o.firstName")
 })
@Entity
@Table(name = "patient_stub", schema = "patient")
public class PatientStub extends GECAMedEntityBean implements Serializable {
	
	private static final long serialVersionUID = 1L;

	public static final String findAllPatientStub = "findAllPatientStub";
    public static final String findPatientStubBySSN = "findPatientStubBySSN";
    
	private String socialSecurityNumber;
	private String surName;
	private String maidenName;
	private String firstName;
	private String gender;
	private String idLuxembourg;
	private Integer insuranceID;
	private String address;
	private String phone;
	private Integer doctorID;
	private String searchHelper;
	private transient InfoString ssnInfo;
	private String storage;

	public PatientStub() {
	}

	/**
	 * @return Returns the firstName.
	 */
	@Column(name = "first_name")
	public String getFirstName() {
		return firstName;
	}
	/**
	 * @param firstName The firstName to set.
	 */
	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}
	/**
	 * @return Returns the gender.
	 */
	@Column(name = "gender")
	public String getGender() {
		return gender;
	}
	/**
	 * @param gender The gender to set.
	 */
	public void setGender(String gender) {
		this.gender = gender;
	}

	/**
	 * @return Returns the maidenName.
	 */
	@Column(name = "maiden_name")
	public String getMaidenName() {
		return maidenName;
	}
	/**
	 * @param maidenName The maidenName to set.
	 */
	public void setMaidenName(String maidenName) {
		this.maidenName = maidenName;
	}
	
	/**
	 * @return Returns the socialSecurityNumber.
	 */
	@Column(name = "social_security_number")
	public String getSocialSecurityNumber() {
		return socialSecurityNumber;
	}
	
	
	/**
	 * @param socialSecurityNumber The socialSecurityNumber to set.
	 */
	public void setSocialSecurityNumber(String socialSecurityNumber) {
		this.socialSecurityNumber = socialSecurityNumber;
	}
	
	
	@Transient
	public InfoString getSSNInfoString () {
		if (ssnInfo == null) {
			String matr = getSocialSecurityNumber();
			try {
				if (matr != null && SSNChecker.isValidSSN(matr) && matr.length() > 8)
				{
					matr = matr.substring(0, 8) + " " + matr.substring(8);
				}
				if (getIdLuxembourg() != null 
						&& getIdLuxembourg().trim().length() > 0)
					ssnInfo = new InfoString(matr, true);
				else
					ssnInfo = new InfoString(matr, false);
			} catch (Exception e) {
				ssnInfo = new InfoString(matr, false);
			}
		}
		return ssnInfo;
	}
	
	private transient String genderAcronym;
	@Transient
	public String getGenderAcronym ()
	{
		if (genderAcronym == null)
		{
			if (getGender() == null)
			{
				setGender(Gender.UNKNOWN);
			}
			genderAcronym= Translatrix.getTranslationString("Gender." + getGender()).substring(0, 1).toUpperCase();
		}
		
		return genderAcronym;
	}
	
	
	/**
	 * @return Returns the surName.
	 */
	@Column(name = "sur_name")
	public String getSurName() {
		return surName;
	}
	/**
	 * @param surName The surName to set.
	 */
	public void setSurName(String surName) {
		this.surName = surName;
	}
	
	/**
	 * @return Returns the insuranceID.
	 */
	@Column(name="insurance_id", updatable=false, insertable=false)
	public Integer getInsuranceID() {
		return insuranceID;
	}

	/**
	 * @param insuranceID The insuranceID to set.
	 */
	public void setInsuranceID(Integer insuranceID) {
		this.insuranceID = insuranceID;
	}
	
	@Column(name="id_luxembourg")
	public String getIdLuxembourg ()
	{
		return idLuxembourg;
	}
	
	public void setIdLuxembourg (String idLuxembourg)
	{
		this.idLuxembourg = idLuxembourg;
	}

	/**
	 * @return the phone
	 */
	@Column(name="phone")
	public String getPhone() {
		return phone;
	}

	/**
	 * @param phone the phone to set
	 */
	public void setPhone(String phone) {
		this.phone = phone;
	}

	/**
	 * @return the address
	 */
	@Column(name="address")
	public String getAddress() {
		return address;
	}

	/**
	 * @param address the address to set
	 */
	public void setAddress(String address) {
		this.address = address;
	}
	
	/**
	 * @return Returns the doctorID.
	 */
	@Column(name = "doctor_id")
	public Integer getDoctorID() {
		return doctorID;
	}
	/**
	 * @param doctorID The doctorID to set.
	 */
	public void setDoctorID(Integer doctorID) {
		this.doctorID = doctorID;
	}
	

	@Column (insertable = false, updatable = false, name = "search")
	public String getSearchHelper ()
	{
		return searchHelper;
	}
	
	/**
	 * @return Returns the storage.
	 */
	@Column(name = "storage")
	public String getStorage() {
		return storage;
	}
	/**
	 * @param storage The Storage to set.
	 */
	public void setStorage(String storage) {
		this.storage = storage;
	}
	
	/**
	 * <b>ONLY TO BE USED BY HIBERNATE</b>
	 * @param searchHelper
	 */
	@Deprecated
	public void setSearchHelper (String searchHelper)
	{
		this.searchHelper = searchHelper;
	}
	
	
	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		if (this.getSurName() == null && this.getFirstName() == null) {
			return Translatrix.getTranslationString("pm.newPatient");
		}
		try {
			String ssn = "";
			try {
				if (SSNChecker.isValidSSN(ssn))
				{
					ssn = this.getSocialSecurityNumber().substring(0, 8) + " " 
							+ this.getSocialSecurityNumber().substring(8);
				}
			} catch (Exception e) {
			}
			return this.getSurName() + ", " + this.getFirstName() + " (" +ssn + ")";
		} catch (Exception e) {
			return "Patient has no name yet!";
		}
	}

	public String toShortString() {
		if (this.getSurName() == null && this.getFirstName() == null) {
			return Translatrix.getTranslationString("pm.newPatient");
		}
		return this.getSurName() + ", " + this.getFirstName();
	}


}
