/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.patient.ejb.entity.beans;

import java.io.Serializable;

import javax.persistence.Entity;
import javax.persistence.NamedQuery;
import javax.persistence.Table;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The Insurance class represents the available health insurances.
 * @author nmac
 * @created 06/04/26
 */

@Entity
//@Cache(usage = CacheConcurrencyStrategy.READ_ONLY) 
@Table(name = "insurance", schema = "patient")

@javax.persistence.NamedQueries 
	({
	@NamedQuery(name = Insurance.c_AllInsurances, 			 	query = "SELECT OBJECT(o) FROM Insurance o WHERE o.disbanded = false ORDER BY o.acronym ASC"),
	@NamedQuery(name = Insurance.c_PrimaryInsurances, 		 	query = "SELECT OBJECT(o) FROM Insurance o WHERE o.disbanded = false AND o.primary = true ORDER BY o.id ASC"),
	@NamedQuery(name = Insurance.c_ThirdPartyPayingInsurances,  query = "SELECT OBJECT(o) FROM Insurance o WHERE o.disbanded = false AND o.thirdParty = true ORDER BY o.id ASC"),
	@NamedQuery(name = Insurance.c_PrivateInsurances,           query = "SELECT OBJECT(o) FROM Insurance o WHERE o.disbanded = false AND o.private = true ORDER BY o.id ASC"),
	@NamedQuery(name = Insurance.c_UCMAffiliatedInsurances,     query = "SELECT OBJECT(o) FROM Insurance o WHERE o.disbanded = false AND o.ucmAffiliated = true ORDER BY o.id ASC"),
	@NamedQuery(name = Insurance.c_DisbandedInsurances, 		query = "SELECT OBJECT(o) FROM Insurance o WHERE o.disbanded = true ORDER BY o.acronym ASC"),
	@NamedQuery(name = Insurance.c_InsurancesByAcronym,    	    query = "SELECT OBJECT(o) FROM Insurance o WHERE upper(o.acronym) LIKE upper(:acronym)  ORDER BY o.id ASC")
	})

public class Insurance extends GECAMedEntityBean implements Serializable 
	{
	private static final long serialVersionUID = 1L;
	
	public static final String INSURANCE_CNS 	= "CNS";
	public static final String INSURANCE_AAA 	= "AAA";
	public static final String INSURANCE_PRIV	= "PRIV";
	public static final String INSURANCE_MAMMO	= "MAMMO";
	
	private String	m_Acronym;
	private String 	m_Hl7Acronym;
	private String	m_Name;
	private Boolean	m_UCMAffiliated;
	private Boolean	m_Primary;
	private Boolean	m_ThirdParty;
	private Boolean m_Private;
	private Double	m_Majoration;
	private Boolean m_Disbanded;

//***************************************************************************
//* Constants		                                                        *
//***************************************************************************	

	public static final transient String c_AllInsurances 				= "getAllInsurances";
	public static final transient String c_PrimaryInsurances 			= "getPrimaryInsurances";
	public static final transient String c_ThirdPartyPayingInsurances 	= "getThirdPartyPayingInsurances";
	public static final transient String c_PrivateInsurances 			= "getPrivateInsurances";
	public static final transient String c_UCMAffiliatedInsurances 		= "getUCMAffiliatedInsurances";
	public static final transient String c_DisbandedInsurances 			= "getDisbandedInsurances";
	public static final transient String c_InsurancesByAcronym 			= "findInsurancesByAcronym";
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	
	
//***************************************************************************
//* Getter and Setter Methods	                                            *
//***************************************************************************	

//---------------------------------------------------------------------------
/**
 * Returns the acronym corresponding to this insurance
 * @return this insurances' acronym
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="acronym")

public String getAcronym() 
	{
	return this.m_Acronym;
	}

//---------------------------------------------------------------------------
/**
 * Sets this inscurances' acronym to the specified value
 * @param p_Acronym specifies the value to set this inscurances' acronym to.
 */
//---------------------------------------------------------------------------

public void setAcronym (String p_Acronym) 
	{
	this.m_Acronym = p_Acronym;
	}

//---------------------------------------------------------------------------
/**
 * Returns the hl7_acronym corresponding to this insurance
 * @return this insurances' hl7_acronym
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="hl7_acronym")

public String getHl7Acronym() 
	{
	return this.m_Hl7Acronym;
	}

//---------------------------------------------------------------------------
/**
 * Sets this inscurances' hl7_acronym to the specified value
 * @param p_Hl7Acronym specifies the value to set this inscurances' hl7_acronym to.
 */
//---------------------------------------------------------------------------

public void setHl7Acronym (String p_Hl7Acronym) 
	{
	this.m_Hl7Acronym = p_Hl7Acronym;
	}
//---------------------------------------------------------------------------
/**
 * @return Returns the full name of this insurance
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="name")

public String getName() 
	{
	return this.m_Name;
	}

//---------------------------------------------------------------------------
/**
 * Sets the full name of this insurance
 * @param p_Name specifies the new name for this insurance.
 */
//---------------------------------------------------------------------------

public void setName (String p_Name) 
	{
	this.m_Name = p_Name;
	}

//---------------------------------------------------------------------------
/**
 * tells whether this health insurance is affiliated to the UCM
 * (Union des Caisse de Maladie) or not. 
 * @return Returns <b>true</b> if insurance is affiliated
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="ucm_affiliated")

public Boolean getUcmAffiliated() 
	{
	return this.m_UCMAffiliated;
	}

//---------------------------------------------------------------------------
/**
 * Sets the full name of this insurance
 * @param p_Name specifies the new name for this insurance.
 */
//---------------------------------------------------------------------------

public void setUcmAffiliated (Boolean p_UCMAffiliated) 
	{
	this.m_UCMAffiliated = p_UCMAffiliated;
	}

//---------------------------------------------------------------------------
/**
  * Tells whether this health insurance covers third party payment or not.
  * @return Returns <b>true</b> if insurance covers third party payment.
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="is_primary")

public Boolean getPrimary() 
	{
	return this.m_Primary;
	}

//---------------------------------------------------------------------------
/**
 * Sets or resets this insurances third party payment flag.
 * @param p_ThirdParty specifies whether to set or reset this insurances'
 * third party payment flag.
 */
//---------------------------------------------------------------------------

public void setPrimary (Boolean p_Primary) 
	{
	this.m_Primary = p_Primary;
	}

//---------------------------------------------------------------------------
/**
  * Tells whether this health insurance covers third party payment or not.
  * @return Returns <b>true</b> if insurance covers third party payment.
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="is_third_party")

public Boolean getThirdParty() 
	{
	return this.m_ThirdParty;
	}

//---------------------------------------------------------------------------
/**
 * Sets or resets this insurances third party payment flag.
 * @param p_ThirdParty specifies whether to set or reset this insurances'
 * third party payment flag.
 */
//---------------------------------------------------------------------------

public void setThirdParty (Boolean p_ThirdParty) 
	{
	this.m_ThirdParty = p_ThirdParty;
	}

//---------------------------------------------------------------------------
/**
 * Tells whether this health insurance is private or not.
 * @return Returns <b>true</b> if insurance is private.
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="is_private")

public Boolean getPrivate() 
	{
	return this.m_Private;
	}

//---------------------------------------------------------------------------
/**
 * Sets or resets this insurances private flag.
 * @param p_Private specifies whether this insurance is a private insurance
 * or not
 */
//---------------------------------------------------------------------------

public void setPrivate (Boolean p_Private) 
	{
	this.m_Private = p_Private;
	}

//---------------------------------------------------------------------------
/**
 * Some health insurances, for instance health insurances of the european
 * parliament, allow a mojoration of the rates defined by the UCM. The
 * getMajoration method returns the majoration factor applicable for this
 * insurance.
 * @return Returns <b>true</b> if insurance is private.
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="majoration")

public Double getMajoration() 
	{
	return m_Majoration;
	}

//---------------------------------------------------------------------------
/**
 * Sets the applicable majoration factor for this insurance
 * @param p_Majoration specifies the majoration factor for this insurance.
 * Example: If a majoration of 15% is applicable, then majoration factor should 
 * be 1.15 => 115%
 */
//---------------------------------------------------------------------------

public void setMajoration (Double p_Majoration) 
	{
	m_Majoration = p_Majoration;
	}

//---------------------------------------------------------------------------
/**
 * Sets whether this insurance was disbanded or still exists.
 * @param p_Dispanded specifies whether this insurance still exists or 
 * was disbanded and belongs to the CNS now.
 */
//---------------------------------------------------------------------------

public void setDisbanded (Boolean p_Disbanded)
	{
	m_Disbanded = p_Disbanded;
	}

//---------------------------------------------------------------------------
/**
 * Specifies whether this insurance still exists or was disbanded and belongs 
 * to the CNS now.
 * @return <code>true</code> if the insurance is now part of the CNS, else 
 * <code>false</code>
 */
//---------------------------------------------------------------------------

public Boolean getDisbanded ()
	{
	return m_Disbanded;
	}

public String toString() {
	return this.getAcronym();
}


@Override
public int compareTo (GECAMedEntityBean o)
{
	if (o instanceof Insurance)
	{
		Insurance insurance = (Insurance) o;
		
		if (getAcronym() != null && insurance.getAcronym() != null)
		{
			int i = getAcronym().compareTo(insurance.getAcronym());
			if (i != 0)
				return i;
		}
	}
	return super.compareTo(o);
}
//***************************************************************************
//* End of Class															*
//***************************************************************************
}
