/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/*
 * Author: Johannes Hermen Tudor/Santec
 * Mail: johannes.hermen@tudor.lu
 * Created: Jan 17, 2005
 *
 */
package lu.tudor.santec.gecamed.office.gui.physicianmanagement;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Date;

import javax.swing.JButton;

import lu.tudor.santec.gecamed.address.ejb.entity.beans.AddressType;
import lu.tudor.santec.gecamed.address.gui.widgets.AddressPanel;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.widgets.HeaderedPanel;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.PhysicianAddress;
import lu.tudor.santec.i18n.Translatrix;
import lu.tudor.santec.widgets.gui.ButtonFactory;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;


/**
 * represents a JPanel to show/edit the physicians address
 *
 * @author Johannes Hermen johannes.hermen
 * 
 * @Version
 * <br>$Log: PhysicianAddressPanel.java,v $
 * <br>Revision 1.8  2013-12-27 18:09:26  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.7  2013-07-15 06:18:37  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.6  2010-03-12 14:17:42  hermen
 * <br>cleanup of panel layouts and icons
 * <br>
 * <br>Revision 1.5  2008-12-05 17:34:23  heinemann
 * <br>moved some basic classes to lu.tudor.santec.widgets
 * <br>
 * <br>Revision 1.4  2008-09-25 09:43:08  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.3  2008-06-30 08:16:45  hermen
 * <br>added address autocompletion by locality
 * <br>
 * <br>Revision 1.2  2008-05-29 10:07:51  hermen
 * <br>added delete physician address
 * <br>
 * <br>Revision 1.1  2008-05-09 13:40:45  hermen
 * <br>added physician address
 * <br>
 * <br>Revision 1.26  2008-04-16 14:24:28  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.25  2008-01-15 13:08:51  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 */
public class PhysicianAddressPanel extends HeaderedPanel implements KeyListener, ItemListener, PropertyChangeListener, ActionListener 
{
    //~ Static fields/initializers =============================================

	/**
	 * static logger for this class
	 */
	private static Logger logger = Logger.getLogger(PhysicianAddressPanel.class
			.getName());
	
    private static final long serialVersionUID = 1L;
    private static final String COLUMNSPACE = "2dlu"; 
    
	private boolean listenersActivated = false;
	
	private PhysicianDataDialog physicianDataPanel;

	private PhysicianAddress currentAddress;
	private AddressPanel addressPanel;

	private boolean haschanges;

	private JButton deleteAddress; 


    //~ Constructors ===========================================================
    /**
     */
    public PhysicianAddressPanel(PhysicianDataDialog physicianDataPanel)
    {
    	this.setOpaque(false);
        this.physicianDataPanel = physicianDataPanel;
        this.buildPanel();
    }

    //~ Methods ================================================================
    /**
     * saves the Values of all fields to the passed PhysicianAddress object
     *
     * @param address the PhysicianAddress object to store the values in
     */
    private void getAddress(PhysicianAddress address)
    {
    	if (address == null) {
    		address = new PhysicianAddress();
    	}
    	try {
    		address.setType(AddressType.PRIVATE);
    	} catch (Exception e) {
    		logger.log(Level.WARN, "setting address type failed" ,e);
	}
    	
    	address.setZip(this.addressPanel.getZip());
    	
    	address.setLocalityId(this.addressPanel.getLocalityID());
        address.setLocality(this.addressPanel.getTfLocality());
    	
        address.setStreetName(this.addressPanel.getStreetName());
        address.setStreetNumber(this.addressPanel.getStreetNumber());
        
        address.setCountry(this.addressPanel.getCountry());
    	
        if (address.getDate() == null) {
        	address.setDate(new Date());
        }
    }

    /**
     * fills the fields of the panel with data from the passed PhysicianAddress
     * object.
     *
     * @param address the PhysicianAddress object to load the values from
     */
    private void setAddress(PhysicianAddress address)
    {
    	listenersActivated = false;
		this.addressPanel.setZip(address.getZip());
		
		this.addressPanel.setLocalityID(address.getLocalityId());	
		if (address.getLocalityId() == null) {
			this.addressPanel.setLocality(address.getLocality());
		}
		
		this.addressPanel.setStreet(address.getStreetName(), address.getStreetNumber());
    	
		this.addressPanel.setCountry(address.getCountry());

        listenersActivated = true;
    }

    /**
     * adds the components to the panel
     */
    private void buildPanel()
    {
	
	setHeader(Translatrix.getTranslationString("pm.address"));
	
        initComponents();
        // build Layout
        FormLayout layout = new FormLayout("pref:grow, 2dlu, pref","bottom:pref"); 
        this.setLayout(layout); 
        CellConstraints cc = new CellConstraints();
        this.add(this.addressPanel, cc.xy(1, 1));
        
        this.add(this.deleteAddress, cc.xy(3, 1));
    }

    /**
     * initializes the Components
     */
    private void initComponents()
    {    	
    	this.addressPanel = new AddressPanel(false);
    	this.addressPanel.addPropertyChangeListener(this);
    	
    	this.deleteAddress = ButtonFactory.createEffectButton("", GECAMedModule.getSmallIcon(GECAMedModule.EDIT_CLEAR_RTL));
    	this.deleteAddress.addActionListener(this);
    	
    }

	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyTyped(java.awt.event.KeyEvent)
	 */
	public void keyTyped(KeyEvent e) {
		this.physicianDataPanel.setUnsavedChanges(true);
		this.haschanges = true;
	}
	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyPressed(java.awt.event.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) {
	}
	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyReleased(java.awt.event.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) {
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ItemListener#itemStateChanged(java.awt.event.ItemEvent)
	 */
	public void itemStateChanged(ItemEvent e) {
		if (listenersActivated ) {
			this.physicianDataPanel.setUnsavedChanges(true);
			this.haschanges = true;
			getAddress(currentAddress);
		}
	}
	
	/**
	 * sets the physician to the panel
	 * @param physician
	 */
	public void setPhysician(Physician physician) {
	    this.haschanges = false;
	    if (physician.getPhysicianAddress() != null) {
		this.currentAddress = physician.getPhysicianAddress();
	    } else {
		this.currentAddress = new PhysicianAddress();
	    }
	    this.setAddress(this.currentAddress);
	 
	}
	
	/**
	 * gets the physician from the panel
	 * @param physician
	 */
	public void getData(Physician physician) {
		getAddress(currentAddress);
		if (haschanges)
		    physician.setPhysicianAddress(currentAddress);
	}
	
	/* (non-Javadoc)
	 * @see java.beans.PropertyChangeListener#propertyChange(java.beans.PropertyChangeEvent)
	 */
	public void propertyChange(PropertyChangeEvent evt) {
		if (listenersActivated && evt.getPropertyName().equals(AddressPanel.UNSAVED_CHANGES)) {
//			this.physicianManagementPanel.setUnsavedChanges(true);
			this.haschanges = true;
			getAddress(currentAddress);
		}
	}
	
	/* (non-Javadoc)
	 * @see javax.swing.JComponent#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		listenersActivated = enabled;
		this.addressPanel.setEditable(enabled);
		super.setEnabled(enabled);
	}

	public void actionPerformed(ActionEvent e) {
	    if (e.getSource().equals(this.deleteAddress) ) {
//		getData(physicianManagementPanel.currentPhysician);
//		physicianManagementPanel.currentPhysician.setPhysicianAddress(null);
//		physicianManagementPanel.savePhysician();
	    }
	}
	
}
