/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.office.gui.officemanagement;

import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;

import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.PhoneType;
import lu.tudor.santec.gecamed.core.gui.widgets.EMailField;
import lu.tudor.santec.gecamed.core.gui.widgets.HeaderedPanel;
import lu.tudor.santec.gecamed.core.gui.widgets.LimitTextField;
import lu.tudor.santec.gecamed.core.gui.widgets.NamedComponent;
import lu.tudor.santec.gecamed.core.gui.widgets.PhoneField;
import lu.tudor.santec.gecamed.core.gui.widgets.UCMCodeField;
import lu.tudor.santec.gecamed.core.utils.UCMCodeFormatter;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Office;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.OfficePhone;
import lu.tudor.santec.i18n.Translatrix;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

/**
 * JPanel to edit the general office data such as name, ucm, etc.
 *
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @version
 * <br>$Log: OfficeGeneralPanel.java,v $
 * <br>Revision 1.17  2013-12-27 18:09:26  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.16  2010-03-12 14:17:42  hermen
 * <br>cleanup of panel layouts and icons
 * <br>
 * <br>Revision 1.15  2009-02-11 14:05:53  hermen
 * <br>added management of sites to the office admin tab
 * <br>
 * <br>Revision 1.14  2008-09-25 09:43:10  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.13  2008-03-17 16:07:59  mack
 * <br>Changes required by moving SSNFormatter into core.utils package
 * <br>
 * <br>Revision 1.12  2008-01-15 13:08:50  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class OfficeGeneralPanel extends HeaderedPanel implements KeyListener
{
    //~ Static fields/initializers =============================================
    
    private static final long serialVersionUID = 1L;

    //~ Instance fields ========================================================

    public JTextField name;
    public UCMCodeField ucmField;
    public JTextField website;
    private OfficeManagementPanel officeManagementPanel;
    private JTextArea information;
    private boolean isEnabled;
    private EMailField emailAddress;
    private JTextField faxNumber;
    private JTextField phoneNumber;
    private HashMap<String, OfficePhone> phones = new HashMap<String, OfficePhone>();


    //~ Constructors ===========================================================

    public OfficeGeneralPanel(OfficeManagementPanel officeManagementPanel)
    {
    	this.setOpaque(false);
    	this.officeManagementPanel = officeManagementPanel;
        this.buildPanel();
    }

    //~ Methods ================================================================
    /**
     * gets the office from the panel
     */
    public void getData(Office office)
    {
        office.setName(name.getText());
        office.setUcmCode((String) ucmField.getValue());
        office.setWebsite(website.getText());
        office.setInformation(information.getText());
        
    	List<OfficePhone> phoneList = new ArrayList<OfficePhone>();
    	if (!this.phoneNumber.getText().equals("") )  {
    		OfficePhone p = new OfficePhone();
    		p.setType(PhoneType.HOME);
    		p.setNumber(this.phoneNumber.getText());
        	phoneList.add(p);	
    	}
    	
    	if (!this.faxNumber.getText().equals("") )  {
    		OfficePhone p = new OfficePhone();
    		p.setType(PhoneType.FAX);
    		p.setNumber(this.faxNumber.getText());
        	phoneList.add(p);	
    	}
    	  	
    	office.setOfficePhones(new HashSet(phoneList));
        office.setEmail(this.emailAddress.getText());
    }

    /**
     * sets the office to the panel
     */
    public void setOffice(Office office)
    {
        this.name.setText(office.getName());
        this.ucmField.setValue(office.getUcmCode());
        this.website.setText(office.getWebsite());
        this.information.setText(office.getInformation());
        	
            
        phones.clear();
        Collection<OfficePhone> phoneList = office.getOfficePhones();
        if (phoneList != null) {
        	for (OfficePhone p : phoneList)
        		phones.put(p.getType(), p);
        }
        		
         this.phoneNumber.setText(phones.get(PhoneType.HOME) != null ? phones.get(PhoneType.HOME).getNumber() : "");
         this.faxNumber.setText(phones.get(PhoneType.FAX) != null ? phones.get(PhoneType.FAX).getNumber() : "");
         this.emailAddress.setText(office.getEmail());
            
    }

    /**
     * adds the components to the panel
     */
    private void buildPanel()
    {
	setHeader(Translatrix.getTranslationString("office.office"));
	
        // initialize components
        initComponents();

        FormLayout layout = new FormLayout(
        	"150dlu, 3dlu, 150dlu, 3dlu, 100dlu:grow",
        	"3dlu, top:pref, 3dlu, pref, 3dlu, pref");
        
        this.setLayout(layout);
//        this.setBorder(new GECAMedTitledBorder(Translatrix.getTranslationString(
//                    "office.generalinfo"))); 
        CellConstraints cc = new CellConstraints();

        // add Labels
        // row 1
        this.add(new NamedComponent(Translatrix.getTranslationString(
                    "office.name"), this.name), cc.xy(1, 2)); 
        this.add(new NamedComponent(Translatrix.getTranslationString(
        			"phy.ucm"), this.ucmField), cc.xy(1,4)); 
        this.add(new NamedComponent(Translatrix.getTranslationString(
                    "office.website"), this.website), cc.xy(1,6));
        
        this.add(new NamedComponent(Translatrix.getTranslationString("pm.phone"), this.phoneNumber), cc.xy(3,2)); 
        this.add(new NamedComponent(Translatrix.getTranslationString("pm.fax"), this.faxNumber), cc.xy(3,4)); 
        this.add(new NamedComponent(Translatrix.getTranslationString("pm.mail"), this.emailAddress), cc.xy(3,6)); 
        
        JScrollPane jsp = new JScrollPane(this.information);
        jsp.setOpaque(false);
        jsp.getViewport().setOpaque(false);
        this.add(new NamedComponent(Translatrix.getTranslationString(
	"office.information"),jsp), cc.xywh(5,2,1,5));
//        this.add(jsp, cc.xywh(5,4,1,3));
    }

    /**
     * initializes the Components
     */
    private void initComponents()
    {
        this.name = new LimitTextField(90);
        this.name.addKeyListener(this);
        this.ucmField = new UCMCodeField(new UCMCodeFormatter());
        this.ucmField.addKeyListener(this);
        this.website = new LimitTextField(90);
        this.website.addKeyListener(this);
        this.information = new JTextArea(4, 30);
        this.information.addKeyListener(this);
        
        this.phoneNumber = new PhoneField(30);
        this.phoneNumber.addKeyListener(this);
        this.faxNumber = new PhoneField(30);
        this.faxNumber.addKeyListener(this);
        this.emailAddress = new EMailField();
        this.emailAddress.addKeyListener(this);
    }
    
	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyTyped(java.awt.event.KeyEvent)
	 */
	public void keyTyped(KeyEvent e) {
		if (isEnabled)
			this.officeManagementPanel.setUnsavedChanges(true);
	}
	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyPressed(java.awt.event.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) {
	}
	/* (non-Javadoc)
	 * @see java.awt.event.KeyListener#keyReleased(java.awt.event.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) {
	}

	/* (non-Javadoc)
	 * @see javax.swing.JComponent#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		this.isEnabled = enabled;
		this.name.setEditable(enabled);
		this.ucmField.setEditable(enabled);
		this.website.setEditable(enabled);
		this.information.setEditable(enabled);
		super.setEnabled(enabled);
	}
	
}
