/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.office.ejb.entity.beans;

import java.awt.Color;
import java.io.Serializable;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.JoinColumn;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;
import lu.tudor.santec.gecamed.core.utils.GECAMedUtils;

import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;

/**
 * Entity Bean that maps the <b>office.physician</b> table
 * which stores a information about the systems physicians
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 *@Version
 * <br>$Log: Physician.java,v $
 * <br>Revision 1.28  2013-03-19 16:09:44  troth
 * <br>1. Set default color for Physician.
 * <br>2. Fix Bug that show the wrong physician if patients physician is null in DB.
 * <br>
 * <br>Revision 1.27  2012-08-06 14:04:04  troth
 * <br>Now the MedLogin service stores the tokens of the users in the new table external password store.
 * <br>
 * <br>Revision 1.26  2012-08-01 15:38:02  troth
 * <br>Remove token and password saving in the database for MedLogin and CyberLab. Now the token and password is only save for one GECAMed session.
 * <br>
 * <br>Revision 1.25  2012-07-12 15:50:32  troth
 * <br>Implementation of Ticket #1003.
 * <br>
 * <br>Revision 1.24  2012-03-05 16:44:25  troth
 * <br>First version of the MedLogin Button.
 * <br>
 * <br>Revision 1.23  2011-02-01 13:14:51  ferring
 * <br>Error messages added for the labo module
 * <br>
 * <br>Revision 1.22  2010-07-06 13:10:21  troth
 * <br>Bug fix - # 542: Patient name no longer printed in the certificate address area
 * <br>http://santec.tudor.lu/trac/gecamed/ticket/542
 * <br>
 * <br>Revision 1.21  2009-06-30 08:56:05  hermen
 * <br>created getName4Printing in physician for printing the name
 * <br>
 * <br>Revision 1.20  2009-06-03 09:23:40  hermen
 * <br>fixed npe on empty physician name
 * <br>
 * <br>Revision 1.19  2009-03-20 10:20:05  hermen
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.18  2008-12-15 13:46:26  hermen
 * <br>made physician chooser bigger
 * <br>
 * <br>Revision 1.17  2008-12-15 11:02:49  hermen
 * <br>added additional billing ucmcode to physician
 * <br>
 * <br>Revision 1.16  2008-11-18 11:07:05  hermen
 * <br>changed PhysicianChooser for Maisons médicales
 * <br>
 * <br>Revision 1.15  2008-09-25 09:43:08  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.14  2008-05-09 13:40:45  hermen
 * <br>added physician address
 * <br>
 * <br>Revision 1.13  2008-04-25 07:38:15  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.12  2008-01-15 13:08:51  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * 
 */
@javax.persistence.NamedQueries({

    // findAll NamedQuery
	@javax.persistence.NamedQuery(name="findAllPhysician",
		    query="SELECT OBJECT(o) FROM Physician o ORDER BY o.name, o.firstName ASC"),
	@javax.persistence.NamedQuery(name="getPhysiciansByName",
			query="SELECT OBJECT(o) FROM Physician o WHERE lower(o.name) = lower(:name) AND lower (o.firstName) = lower(:firstName)"),
	@javax.persistence.NamedQuery(name="getPhysicianByUCMCode",
			query="SELECT OBJECT(o) FROM Physician o WHERE o.ucmCode = :ucmCode")
})

@Entity
@Cache(usage = CacheConcurrencyStrategy.NONSTRICT_READ_WRITE) 
@Table(name = "physician", schema = "office")
public class Physician extends GECAMedEntityBean implements Serializable {

	private static final long serialVersionUID = 1L;
	
	public static final String UCM_DEFAULT = "000000-00";
	
	private Integer userId;
	private String name;
	private String firstName;
	private String ucmCode;
	private String ucmCodeTemp;
	private String speciality;
	private String email;
	private String title;
	private String phoneExtension;
	private String fax;	
	private String gsm;	
	private Integer color;
	private PhysicianAddress physicianAddress;
	private String ucmFacturation;
	
	public Physician() {
		this.ucmCode = UCM_DEFAULT;
	}
	
	public Physician(String name) {
		this.name = name;
		this.ucmCode = UCM_DEFAULT;
	}
	
	/**
	 * @return Returns the email.
	 */
	@javax.persistence.Column(name = "email")
	public String getEmail() {
		return email;
	}

	/**
	 * @param email The email to set.
	 */
	public void setEmail(String email) {
		this.email = email;
	}
	
	/**
	 * @return Returns the firstName.
	 */
	@javax.persistence.Column(name = "first_name")
	public String getFirstName() {
		return firstName;
	}

	/**
	 * @param firstName The firstName to set.
	 */
	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}

	/**
	 * @return Returns the name.
	 */
	@javax.persistence.Column(name = "name")
	public String getName() {
		return name;
	}

	/**
	 * @param name The name to set.
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return Returns the phoneExtension.
	 */
	@javax.persistence.Column(name = "phone_extension")
	public String getPhoneExtension() {
		return phoneExtension;
	}

	/**
	 * @param phoneExtension The phoneExtension to set.
	 */
	public void setPhoneExtension(String phoneExtension) {
		this.phoneExtension = phoneExtension;
	}
	
	
	/**
	 * @return Returns the fax.
	 */
	@javax.persistence.Column(name = "fax")
	public String getFax() {
		return fax;
	}

	/**
	 * @param fax The fax to set.
	 */
	public void setFax(String fax) {
		this.fax = fax;
	}
	
	/**
	 * @return Returns the gsm.
	 */
	@javax.persistence.Column(name = "gsm")
	public String getGsm() {
		return gsm;
	}

	/**
	 * @param gsm The gsm to set.
	 */
	public void setGsm(String gsm) {
		this.gsm = gsm;
	}

	/**
	 * @return Returns the speciality.
	 */
	@javax.persistence.Column(name = "speciality")
	public String getSpeciality() {
		return speciality;
	}

	/**
	 * @param speciality The speciality to set.
	 */
	public void setSpeciality(String speciality) {
		this.speciality = speciality;
	}

	/**
	 * @return Returns the title.
	 */
	@javax.persistence.Column(name = "title")
	public String getTitle() {
		return title;
	}

	/**
	 * @param title The title to set.
	 */
	public void setTitle(String title) {
		this.title = title;
	}

	/**
	 * @return Returns the ucmCode.
	 */
	@javax.persistence.Column(name = "ucm_code")
	public String getUcmCode() {
		return ucmCode;
	}

	/**
	 * @param ucmCode The ucmCode to set.
	 */
	public void setUcmCode(String ucmCode) {
		this.ucmCode = ucmCode;
	}
	
	@Transient
	public String getUcmCodeTemp () {
		return ucmCodeTemp;
	}
	@Transient
	public void setUcmCodeTemp (String code) {
		this.ucmCodeTemp = code;
	}
	
	/**
	 * Returns the Physicians UCM Code used for Billing/Invoices. 
	 * @return ucm_code_facturation if != null, else ucm_code
	 */
	@Transient
	public String getBillingCode () {
		if (getUcmFacturation() != null && getUcmFacturation().length() > 0) {
			return getUcmFacturation();			
		} else {
			return getUcmCode();
		}
	}
	
	/**
	 * @return the ucmFacturation
	 */
	@javax.persistence.Column(name = "ucm_code_facturation")
	public String getUcmFacturation() {
	    return ucmFacturation;
	}

	/**
	 * @param ucmFacturation the ucmFacturation to set
	 */
	public void setUcmFacturation(String ucmFacturation) {
	    this.ucmFacturation = ucmFacturation;
	}

	/**
	 * @return Returns the userId.
	 */
	@javax.persistence.Column(name = "user_id")
	public Integer getUserId() {
		return userId;
	}

	/**
	 * @param userId The userId to set.
	 */
	public void setUserId(Integer userId) {
		this.userId = userId;
	}
	

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer rendered;
		
		rendered = new StringBuffer();
		if (this.getName() != null)		
		    rendered.append(getName());
		if (this.getFirstName() != null)	
		    if ( this.getFirstName().length() > 5)
			rendered.append(", "+getFirstName().substring(0,4)+".");
		    else
			rendered.append(", "+getFirstName());
		return rendered.toString() + " ["+getMnemonic()+"]";
	}
	
	
	/**
	 * please use TemplateFormatter.formatPhysicianName instead!
	 * @return
	 */
	@Transient
	@Deprecated
	public String getName4Printing() {
		StringBuffer rendered;
		
		rendered = new StringBuffer();
		if (this.getName() != null)		
		    rendered.append(getName());
		if (this.getFirstName() != null)	
		    rendered.append(", "+getFirstName());
		return rendered.toString();
	}
	
	/**
	 * Returns a mnemonic made of the first letters of the name 
	 * 
	 * 
	 * @return
	 */
	@Transient
	public String getMnemonic() {
		try {
			/* --------------------------------------------- */
			String s = firstName.substring(0, 2)+name.substring(0, 1);
			return s;
			/* --------------------------------------------- */
		} catch (Exception e) {
			/* --------------------------------------------- */
		    	try {
		    	    return name.substring(0, 1);
			} catch (Exception e2) {
			    return "";
			}
			/* --------------------------------------------- */
		}
	}

	/**
	 * @return the color
	 */
	@javax.persistence.Column(name = "color")
	public Integer getColor() {
		if(color == null || color == -1) {
			color = new Color(254,254,254).getRGB();
		}
		return color;
	}

	/**
	 * @param color the color to set
	 */
	public void setColor(Integer color) {
		this.color = color;
	}
	
	/**
	 * returns all physicianAddress
	 * @return all physicianAddress
	 */
	@OneToOne(cascade = {CascadeType.ALL})
	@JoinColumn(name = "address_id")
	public PhysicianAddress getPhysicianAddress() {
		return this.physicianAddress;
	}

	/**
	 * sets the physicianAddress
	 * 
	 * @param physicianAddress
	 */
	public void setPhysicianAddress(
			PhysicianAddress physicianAddress) {
		this.physicianAddress = physicianAddress;
	}
	
	
	@Override
	public int compareTo (GECAMedEntityBean o)
	{
		if (o != null && o instanceof Physician)
		{
			int i = GECAMedUtils.compareTo(this.getName(), ((Physician)o).getName());
			if (i != 0)
				return i;
			i = GECAMedUtils.compareTo(this.getFirstName(), ((Physician)o).getFirstName());
			if (i != 0)
				return i;
		}
		return super.compareTo(o);
	}
}
