/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.office.ejb.entity.beans;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;

import javax.persistence.Entity;
import javax.persistence.JoinColumn;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.address.ejb.entity.beans.AddressType;
import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;

/**
 * Entity Bean that maps the <b>office.office</b> table
 * which stores a informations such as address, name, phones of the office
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 *@Version
 * <br>$Log: Office.java,v $
 * <br>Revision 1.13  2008-09-25 09:43:08  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.12  2008-08-21 15:45:12  heinemann
 * <br>removed the bogus behaviour when there is no office entry in the database. The fields on the print outs are now left blank. The user gets an information that the office address is missing.
 * <br>
 * <br>Revision 1.11  2008-01-15 13:08:51  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * 
 */
@javax.persistence.NamedQueries({

    // findAll NamedQuery
    @javax.persistence.NamedQuery(name="findAllOffice",
        query="SELECT OBJECT(o) FROM Office o")
})

@Entity
@Table(name = "office", schema = "office")
public class Office extends GECAMedEntityBean implements Serializable {

	private static final long serialVersionUID = 1L;
	public static final String UCM_DEFAULT = "000000-00";
	
	private String name;
	private String website;
	private String email;
	private String information;
	private String ucmCode;

	private Set<OfficeAddress> officeAddress;
	private Set<OfficePhone> officePhones;
	
	public Office() {
		this.ucmCode = UCM_DEFAULT;
	}
	
	/**
	 * @return Returns the email.
	 */
	@javax.persistence.Column(name = "email")
	public String getEmail() {
		return email;
	}
	/**
	 * @param email The email to set.
	 */
	public void setEmail(String email) {
		this.email = email;
	}
	/**
	 * @return Returns the name.
	 */
	@javax.persistence.Column(name = "name")
	public String getName() {
		return name;
	}
	/**
	 * @param name The name to set.
	 */
	public void setName(String name) {
		this.name = name;
	}
	/**
	 * @return Returns the website.
	 */
	@javax.persistence.Column(name = "website")
	public String getWebsite() {
		return website;
	}
	/**
	 * @param website The website to set.
	 */
	public void setWebsite(String website) {
		this.website = website;
	}


	/**
	 * returns all office addresses
	 * @return all office addresses
	 */
	@OneToMany(cascade = javax.persistence.CascadeType.ALL, fetch = javax.persistence.FetchType.EAGER)
	@JoinColumn(name = "office_id")
	public Set<OfficeAddress> getOfficeAddress() {
		return this.officeAddress;
	}

	/**
	 * sets the office addresses
	 * 
	 * @param officeAddress
	 */
	public void setOfficeAddress(
			Set<OfficeAddress> officeAddress) {
		this.officeAddress = officeAddress;
	}
	
	/**
	 * returns all office phones
	 * @return
	 */
	@OneToMany(cascade = javax.persistence.CascadeType.ALL, fetch = javax.persistence.FetchType.EAGER)
	@JoinColumn(name = "office_id")
	public Set<OfficePhone> getOfficePhones() {
		return this.officePhones;
	}

	/**
	 * sets all office phones
	 * @param officePhones
	 */
	public void setOfficePhones(
			Set<OfficePhone> officePhones) {
		this.officePhones = officePhones;
	}
	
	/**
	 * @return Returns the ucmCode.
	 */
	@javax.persistence.Column(name = "ucm_code")
	public String getUcmCode() {
		return ucmCode;
	}
	/**
	 * @param ucmCode The ucmCode to set.
	 */
	public void setUcmCode(String ucmCode) {
		this.ucmCode = ucmCode;
	}
	
	
	/**
	 * Returns the patient's home address
	 * 
	 * @return
	 */
	@Transient
	public OfficeAddress getLatestOfficeAddress() {
		/* ==================================== */
		Set<OfficeAddress> homeAddresses = new HashSet<OfficeAddress>();
		// --------------------------------------
		try {
			for (OfficeAddress adr : this.getOfficeAddress()) {
				/* --------------------------------------------- */
				if(adr.getType().equals(AddressType.PRIVATE)) 
					homeAddresses.add(adr);
				/* --------------------------------------------- */
			}
			// very bad. But there must be a home address.
			// The NullPointer can be catched somewhere else
			
			return homeAddresses.iterator().next();
		} catch (Exception e) {
			return null;
		}
		/* ==================================== */
	}
	
	/**
	 * Returns the normal phone of the office
	 * 
	 * @return
	 */
	@Transient
	public OfficePhone getOfficePhoneByType(String officeType) {
		/* ==================================== */
		if (this.getOfficePhones() != null) {
			/* ----------------------------------- */
			for (OfficePhone p : this.getOfficePhones()) {
				/* ----------------------------------- */
				if(p.getType().equals(officeType))
					return p;
				/* ----------------------------------- */
			}
			/* ----------------------------------- */
		}
		return null;
		/* ==================================== */
	}

	/**
	 * @return Returns the information.
	 */
	@javax.persistence.Column(name = "information")
	public String getInformation() {
		return information;
	}

	/**
	 * @param information The information to set.
	 */
	public void setInformation(String information) {
		this.information = information;
	}
	
}
