/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.labo.utils;

import lu.tudor.santec.i18n.Translatrix;

/**
 *
 * @author  santec
 */

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

public class LaboException extends Exception
{
	// ERROR CODES
	public static final int ERROR_UNKNOWN						= Integer.MIN_VALUE;
	public static final int	ERROR_NONE							= 0;
	public static final int	ERROR_ROOT_CA_INVALID				= 1<<0;
	public static final int	ERROR_KEY_INVALID					= 1<<1;
	public static final int ERROR_KEY_NOT_SET					= 1<<2;
	public static final int ERROR_PASSWORD_INVALID				= 1<<3;
	public static final int ERROR_PASSWORD_NOT_SET				= 1<<4;
	public static final int ERROR_READING_PASSWORD				= 1<<5;
	public static final int ERROR_CREATING_KEYSTORE_INSTANCE	= 1<<6;
	public static final int ERROR_STORING_KEYS					= 1<<7;
	public static final int ERROR_WRONG_SIGNATURE				= 1<<8;
	public static final int ERROR_DECRYPTION					= 1<<9;
	public static final int ERROR_CERTIFICATE_NOT_SET			= 1<<10;
	public static final int ERROR_PEMDATA_NOT_PARSED			= 1<<11;
	public static final int ERROR_VERIFYING_DATA				= 1<<12;
	public static final int ERROR_CREATING_CHAIN				= 1<<13;
	public static final int ERROR_LOADING_ROOT_CA				= 1<<14;
//	public static final int ERROR_	= 1>>;
	
	
	//***************************************************************************
	//* Class Constructor(s)                                                    *
	//***************************************************************************
	
	/**
	 * 
	 */
	private static final long	serialVersionUID	= 1L;
	
	private int					errorCode			= LaboException.ERROR_UNKNOWN;
	
	
	//---------------------------------------------------------------------------
	/**
	  * Creates a new instance of <code>SmimeException</code> without detail message.
	  */
	//---------------------------------------------------------------------------
	
	public LaboException(int errorCode)
	{
		this(null, errorCode, null);
	}
	
	
	//--------------------------------------------------------------------------- 
	/**
	  * Constructs an instance of <code>SmimeException</code> with the specified tag.
	  * @param exceptionTag specifies the exception tag.
	  */
	//---------------------------------------------------------------------------
	
	public LaboException(int errorCode, String exceptionTag)
	{
		this(null, errorCode, exceptionTag);
	}
	
	
	public LaboException(Exception e, int errorCode, String exceptionTag)
	{
		super("Following Labo Errors occured:" + getMessageForErrors(errorCode) + 
				(exceptionTag != null ? "\n" + exceptionTag : ""), 
				e);
		this.errorCode = errorCode;
	}
	
	
	//***************************************************************************
	//* Class Body                                                              *
	//***************************************************************************
	
	public String getLocalizedMessage()
	{
		return Translatrix.getTranslationString(getMessage());
	}
	
	
	public int getErrorCode()
	{
		return errorCode;
	}
	
	public static String getMessageForErrors (int errorCode)
	{
		StringBuilder b;
		
		
		if (errorCode == LaboException.ERROR_NONE)
			return "No errors";
		else
		{
			b	= new StringBuilder();
			
			if ((errorCode & LaboException.ERROR_ROOT_CA_INVALID) == LaboException.ERROR_ROOT_CA_INVALID)
				b.append("invalid Root CA, ");
			if ((errorCode & LaboException.ERROR_KEY_INVALID) == LaboException.ERROR_KEY_INVALID)
				b.append("invalid private key, ");
			if ((errorCode & LaboException.ERROR_KEY_NOT_SET) == LaboException.ERROR_KEY_NOT_SET)
				b.append("private key not set, ");
			if ((errorCode & LaboException.ERROR_PASSWORD_INVALID) == LaboException.ERROR_PASSWORD_INVALID)
				b.append("invalid password, ");
			if ((errorCode & LaboException.ERROR_PASSWORD_NOT_SET) == LaboException.ERROR_PASSWORD_NOT_SET)
				b.append("password not set, ");
			if ((errorCode & LaboException.ERROR_READING_PASSWORD) == LaboException.ERROR_READING_PASSWORD)
				b.append("error reading password, ");
			if ((errorCode & LaboException.ERROR_CREATING_KEYSTORE_INSTANCE) == LaboException.ERROR_CREATING_KEYSTORE_INSTANCE)
				b.append("error creating keystore instance, ");
			if ((errorCode & LaboException.ERROR_STORING_KEYS) == LaboException.ERROR_STORING_KEYS)
				b.append("error storing keys, ");
			if ((errorCode & LaboException.ERROR_WRONG_SIGNATURE) == LaboException.ERROR_WRONG_SIGNATURE)
				b.append("wrong signature, ");
			if ((errorCode & LaboException.ERROR_DECRYPTION) == LaboException.ERROR_DECRYPTION)
				b.append("error during decryption, ");
			if ((errorCode & LaboException.ERROR_CERTIFICATE_NOT_SET) == LaboException.ERROR_CERTIFICATE_NOT_SET)
				b.append("certificate not set, ");
			if ((errorCode & LaboException.ERROR_PEMDATA_NOT_PARSED) == LaboException.ERROR_PEMDATA_NOT_PARSED)
				b.append("error parsing PEM data, ");
			if ((errorCode & LaboException.ERROR_VERIFYING_DATA) == LaboException.ERROR_VERIFYING_DATA)
				b.append("error verifying data, ");
			if ((errorCode & LaboException.ERROR_CREATING_CHAIN) == LaboException.ERROR_CREATING_CHAIN)
				b.append("error creating certificate chain file");
			if ((errorCode & LaboException.ERROR_LOADING_ROOT_CA) == LaboException.ERROR_LOADING_ROOT_CA)
				b.append("error loading root CA from DB");
			if ((errorCode & LaboException.ERROR_UNKNOWN) == LaboException.ERROR_UNKNOWN)
				b.append("unknown error");
			
		}
		
		return b.toString().replaceAll("\\,\\s$", "");
	}
	
	//***************************************************************************
	//* End of Class                                                            *
	//***************************************************************************
}
