/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.labo.gui.admin;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.text.NumberFormat;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JProgressBar;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.border.TitledBorder;

import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.IconFetcher;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.SwingWorker;
import lu.tudor.santec.gecamed.core.gui.widgets.GECAMedBaseDialogImpl;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.labo.ejb.entity.beans.Connection;
import lu.tudor.santec.gecamed.labo.ejb.session.beans.LaboratoryBean;
import lu.tudor.santec.gecamed.labo.ejb.session.interfaces.LaboratoryInterface;
import lu.tudor.santec.gecamed.labo.gui.LaboModule;
import lu.tudor.santec.gecamed.labo.utils.LaboException;
import lu.tudor.santec.gecamed.labo.utils.PasswordEncrypter;
import lu.tudor.santec.gecamed.labo.utils.ResultDownloader;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

public class ConnectionDialog extends JDialog implements ItemListener,
														 ActionListener,
														 Relocalizable
	{
	private static final long	serialVersionUID	= -3448943402897014273L;
	
	/** the logger Object for this class */
	private static Logger logger = Logger.getLogger(ConnectionDialog.class.getName());
	
	private JPanel					m_ServerPanel;
	private TitledBorder			m_ServerBorder;
	private JLabel					m_ServerLabel;
	private JCheckBox				m_UseDefaultPort;
	private JLabel					m_PortLabel;
	private JTextField				m_Server;
	private JFormattedTextField		m_Port;
	private JCheckBox				m_PassiveMode;
	
	private JPanel					m_AuthenticationPanel;
	private TitledBorder			m_AuthenticationBorder;
	private JLabel					m_UsernameLabel;
	private JTextField				m_Username;
	private JCheckBox				m_ShowTyping;
	private JLabel					m_PasswordLabel;
	private JPasswordField			m_Password;
	
	private JPanel					m_SynchronizationPanel;
	private TitledBorder			m_SynchronizationBorder;
	private JCheckBox				m_Synchronize;

	private JPanel					m_TestPanel;
	private TitledBorder			m_TestBorder;
	private JButton					m_TestButton;
	private JProgressBar			m_TestProgress;
	
	private JButton					m_SaveButton;
	private JButton					m_CancelButton;
	private boolean					m_Canceled;

	private Connection				m_Connection;

	private NumberFormat			m_PortFormat;

	private SwingWorker				m_ConnectionWorker;
	ResultDownloader				m_TestDownloader;
	private PasswordEncrypter		m_PasswordEncrypter;
//	private boolean					m_Aborted;
	
	private LaboratoryInterface		m_LaboManager;
	
	private static ImageIcon m_ConnectionTestIcon   = IconFetcher.getIcon (LaboModule.class,LaboModule.c_i32_TestConnection);
	private static ImageIcon m_ConnectionAbortIcon  = IconFetcher.getIcon (LaboModule.class,LaboModule.c_i32_AbortConnection);
	private static ImageIcon m_ErrorIcon   			= IconFetcher.getIcon (LaboModule.class,LaboModule.c_i32_ConnectionError);
	private static ImageIcon m_SuccessIcon 			= IconFetcher.getIcon (LaboModule.class,LaboModule.c_i32_ConnectionSuccess);

//---------------------------------------------------------------------------
//***************************************************************************
//* Constants	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

	private final static int  c_IconTextGap   = 10;

	private final static char c_ClearText			 = '\u0000';
	private final static char c_EchoCharacter		 = '\u2022';
	
	private final static String c_Columns			  = "3dlu,fill:max(70dlu;pref):grow," +
														"3dlu,fill:max(35dlu;pref)," +
														"3dlu,fill:max(35dlu;pref),3dlu";
		
	private final static String c_Rows				  = "3dlu,fill:pref," +
														"3dlu,fill:pref," +
														"3dlu,fill:pref," +
														"3dlu,fill:pref," +
														"3dlu,fill:pref,3dlu";

	private final static String c_ServerColumns       = "3dlu,fill:max(40dlu;pref)," +
														"3dlu,fill:max(50dlu;pref)," +
														"3dlu,fill:max(50dlu;pref),3dlu";

	private final static String c_ServerRows 	      = "3dlu,fill:pref," +
														"3dlu,fill:pref," +
														"3dlu,fill:pref," +
														"3dlu,fill:pref,3dlu";
	
	private final static String c_AuthenticationColumns   = "3dlu,fill:max(40dlu;pref)," +
															"3dlu,fill:max(100dlu;pref),3dlu";

	private final static String c_AuthenticationRows 	  = "3dlu,fill:pref," +
															"3dlu,fill:pref," +
															"3dlu,fill:pref,3dlu";

	private final static String c_SynchronizationColumns  = "3dlu,fill:max(40dlu;pref)," +
															"3dlu,fill:max(100dlu;pref),3dlu";
	
	private final static String c_SynchronizationRows 	  = "3dlu,fill:pref,3dlu";

	private final static String c_TestColumns 			= 	"3dlu,fill:pref:grow,3dlu";

	private final static String c_TestRows 				= 	"3dlu,fill:pref," +
	  														"3dlu,fill:pref,3dlu";
	
	private final static String	c_Title				  = "ConnectionDialog.Title";
	
	private final static String	c_ServerBorder		  = "ConnectionDialog.ServerBorder";
	private final static String	c_ServerLabel		  = "ConnectionDialog.ServerLabel";
	private final static String	c_DefaultPortLabel	  = "ConnectionDialog.DefaultPortLabel";
	private final static String	c_PortLabel	  		  = "ConnectionDialog.PortLabel";
	private final static String	c_PassiveModeLabel	  = "ConnectionDialog.PassiveModeLabel";
	
	private final static String c_AuthenticationBorder	= "ConnectionDialog.AuthenticationBorder";
	private final static String c_UsernameLabel			= "ConnectionDialog.UsernameLabel";
	private final static String c_ShowTypingLabel		= "ConnectionDialog.ShowTypingLabel";
	private final static String c_PasswordLabel			= "ConnectionDialog.PasswordLabel";
	
	private final static String c_SynchronizationBorder	= "ConnectionDialog.SynchronizationBorder";
	private final static String c_SynchronizeLabel		= "ConnectionDialog.SynchronizeLabel";
	
	private final static String c_TestBorder			= "ConnectionDialog.TestBorder";
	private final static String c_TestButton			= "ConnectionDialog.TestButton";
	private final static String c_TestAbortButton		= "ConnectionDialog.TestAbortButton";
	
	private final static String c_TestProgressIdle			= "ConnectionDialog.TestProgressIdle";
	private final static String c_TestProgressConnecting	= "ConnectionDialog.TestProgressConnecting";
	private final static String c_TestSuccess				= "ConnectionDialog.TestSuccess";
	private final static String c_TestFailed				= "ConnectionDialog.TestFailed";
	
//	private final static String c_TestErrorTitle		= "ConnectionDialog.TestErrorTitle";
//	private final static String c_TestErrorMessage		= "ConnectionDialog.TestErrorMessage";

//---------------------------------------------------------------------------
//***************************************************************************
//* Constructor	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

public ConnectionDialog (String p_MasterPassword)
	{		
	super (MainFrame.getInstance(),Translatrix.getTranslationString(c_Title),true);	
	
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;	
	Color			l_DefaultColor;
	
	m_LaboManager		= (LaboratoryInterface) ManagerFactory.getRemote(LaboratoryBean.class);
	m_PasswordEncrypter = new PasswordEncrypter ();
	m_PasswordEncrypter.setEncryptionPassword (p_MasterPassword);

	this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
	this.getContentPane().setBackground (GECAMedColors.c_GECAMedBackground);
	
	l_Constraints  = new CellConstraints();
	
	l_DefaultColor = (Color) UIManager.getDefaults().get("TextField.inactiveBackground");   
	UIManager.getDefaults().put("TextField.inactiveBackground", Color.WHITE );

	m_PortFormat = NumberFormat.getIntegerInstance ();
	
	this.buildServerPanel();
	this.buildAuthenticationPanel();
	this.buildSynchronizationPanel();
	this.buildTestPanel();
	
	m_SaveButton    = new JButton ();
	m_SaveButton.setIcon(IconFetcher.getScaledIcon(IconFetcher.class,GECAMedIconNames.OK,16));
	m_SaveButton.addActionListener(this);
	
	m_CancelButton    = new JButton ();
	m_CancelButton.setIcon(IconFetcher.getScaledIcon(IconFetcher.class,GECAMedIconNames.CANCEL,16));
	m_CancelButton.addActionListener(this);

	this.relocalize();

	l_Layout = new FormLayout(c_Columns, c_Rows);
    this.setLayout (l_Layout);

	this.add (m_ServerPanel,  			l_Constraints.xywh(2, 2, 5, 1));
	this.add (m_AuthenticationPanel,  	l_Constraints.xywh(2, 4, 5, 1));
	this.add (m_SynchronizationPanel,   l_Constraints.xywh(2, 6, 5, 1));
	this.add (m_TestPanel,   			l_Constraints.xywh(2, 8, 5, 1));
	this.add (m_SaveButton,				l_Constraints.xywh(4, 10, 1, 1));
	this.add (m_CancelButton,			l_Constraints.xywh(6, 10, 1, 1));
	
	UIManager.getDefaults().put("TextField.inactiveBackground", l_DefaultColor );	
	
	this.getRootPane().setDefaultButton(m_SaveButton);
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

private void buildServerPanel ()
	{
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;	

	l_Layout = new FormLayout(c_ServerColumns, c_ServerRows);
	l_Constraints = new CellConstraints();
	
	m_ServerBorder = new TitledBorder ("");
	
	m_ServerPanel = new JPanel ();
	m_ServerPanel.setBorder(m_ServerBorder);
	m_ServerPanel.setLayout(l_Layout);
	m_ServerPanel.setOpaque (false);
	
	m_ServerLabel = new JLabel ();
	m_ServerLabel.setForeground(GECAMedColors.c_LabelColor);
	
	m_PortLabel   = new JLabel ();
	m_PortLabel.setForeground(GECAMedColors.c_LabelColor);
	
	m_Server	     = new JTextField ();
	m_UseDefaultPort = new JCheckBox ();
	m_UseDefaultPort.setBackground(GECAMedColors.c_GECAMedBackground);
	m_UseDefaultPort.setSelected(true);
	m_UseDefaultPort.addItemListener(this);
	
	m_Port			 = new JFormattedTextField (m_PortFormat);
	m_Port.setEnabled (false);
	
	m_PassiveMode    = new JCheckBox ();
	m_PassiveMode.setBackground(GECAMedColors.c_GECAMedBackground);
	
	m_ServerPanel.add (m_ServerLabel,  		l_Constraints.xywh(2, 2, 1, 1));
	m_ServerPanel.add (m_Server,  			l_Constraints.xywh(4, 2, 3, 1));
	m_ServerPanel.add (m_UseDefaultPort,  	l_Constraints.xywh(4, 4, 3, 1));
	m_ServerPanel.add (m_PortLabel,  		l_Constraints.xywh(2, 6, 1, 1));
	m_ServerPanel.add (m_Port,  			l_Constraints.xywh(4, 6, 1, 1));
	m_ServerPanel.add (m_PassiveMode,  		l_Constraints.xywh(4, 8, 3, 1));
	}

//---------------------------------------------------------------------------

private void buildAuthenticationPanel ()
	{
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;	

	l_Layout = new FormLayout(c_AuthenticationColumns, c_AuthenticationRows);
	l_Constraints = new CellConstraints();
	
	m_AuthenticationBorder = new TitledBorder ("");
	
	m_AuthenticationPanel = new JPanel ();
	m_AuthenticationPanel.setBorder(m_AuthenticationBorder);
	m_AuthenticationPanel.setLayout(l_Layout);
	m_AuthenticationPanel.setOpaque (false);
	
	m_UsernameLabel = new JLabel ();
	m_UsernameLabel.setForeground(GECAMedColors.c_LabelColor);
	
	m_PasswordLabel = new JLabel ();
	m_PasswordLabel.setForeground(GECAMedColors.c_LabelColor);
	
	m_Username 	    = new JTextField ();
	
	m_ShowTyping    = new JCheckBox ();
	m_ShowTyping.setSelected(false);
	m_ShowTyping.setBackground(GECAMedColors.c_GECAMedBackground);
	m_ShowTyping.addItemListener(this);
	
	m_Password		= new JPasswordField ();
	m_Password.setEchoChar(c_EchoCharacter);
	
	m_AuthenticationPanel.add (m_UsernameLabel,  l_Constraints.xywh(2, 2, 1, 1));
	m_AuthenticationPanel.add (m_Username,  	 l_Constraints.xywh(4, 2, 1, 1));
	m_AuthenticationPanel.add (m_ShowTyping,  	 l_Constraints.xywh(4, 4, 1, 1));
	m_AuthenticationPanel.add (m_PasswordLabel,  l_Constraints.xywh(2, 6, 1, 1));
	m_AuthenticationPanel.add (m_Password,  	 l_Constraints.xywh(4, 6, 1, 1));
	}

//---------------------------------------------------------------------------

private void buildSynchronizationPanel ()
	{
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;	

	l_Layout = new FormLayout(c_SynchronizationColumns, c_SynchronizationRows);
	l_Constraints = new CellConstraints();
	
	m_SynchronizationBorder = new TitledBorder ("");
	
	m_SynchronizationPanel = new JPanel ();
	m_SynchronizationPanel.setBorder(m_SynchronizationBorder);
	m_SynchronizationPanel.setLayout(l_Layout);
	m_SynchronizationPanel.setOpaque (false);
	
	m_Synchronize = new JCheckBox ();
	m_Synchronize.setSelected(false);
	m_Synchronize.setBackground(GECAMedColors.c_GECAMedBackground);
		
	m_SynchronizationPanel.add (m_Synchronize,  	 l_Constraints.xywh(4, 2, 1, 1));
	}

//---------------------------------------------------------------------------

private void buildTestPanel ()
	{
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;	

	l_Layout = new FormLayout(c_TestColumns, c_TestRows);
	l_Constraints = new CellConstraints();
	
	m_TestBorder = new TitledBorder ("");
	
	m_TestPanel = new JPanel ();
	m_TestPanel.setBorder(m_TestBorder);
	m_TestPanel.setLayout(l_Layout);
	m_TestPanel.setOpaque (false);

	m_TestButton = new JButton ();
	m_TestButton.setIcon(m_ConnectionTestIcon);
	m_TestButton.setText(Translatrix.getTranslationString(c_TestButton));
	m_TestButton.setIconTextGap(c_IconTextGap);
	m_TestButton.addActionListener(this);
	
	m_TestProgress = new JProgressBar ();
	m_TestProgress.setMinimum(0);
	m_TestProgress.setValue(0);
	m_TestProgress.setStringPainted(true);
	m_TestProgress.setString (Translatrix.getTranslationString(c_TestProgressIdle));	
	
	m_TestPanel.add (m_TestButton,  	 l_Constraints.xywh(2, 2, 1, 1));
	m_TestPanel.add (m_TestProgress,  	 l_Constraints.xywh(2, 4, 1, 1));
	}

//---------------------------------------------------------------------------

private void updateDialogFromConnection (Connection p_Connection)
	{
	if (p_Connection == null) return;
	
	m_Server.setText (p_Connection.getServer());
	if (p_Connection.getPort() != null)
		{
		m_UseDefaultPort.setSelected(false);
		m_Port.setText (p_Connection.getPort().toString());	
		}
	else
		{
		m_UseDefaultPort.setSelected(true);
		m_Port.setText ("");
		}
	
	m_PassiveMode.setSelected(p_Connection.getPassiveMode());
	
	m_Username.setText(p_Connection.getUsername());
	m_ShowTyping.setSelected(false);
	
	if (p_Connection.getPassword() != null)
	{	
		try
		{
			m_Password.setText(m_PasswordEncrypter.decryptPassword(p_Connection.getPassword ()));
		}
		catch (LaboException e)
		{
			logger.log(Level.ERROR, e.getMessage(), e);
			GECAMedBaseDialogImpl.showMessageDialog(this, 
					Translatrix.getTranslationString("title"), 
					Translatrix.getTranslationString("message"), 
					GECAMedBaseDialogImpl.OK_BUTTON_MODE, 
					GECAMedModule.getIcon(GECAMedIconNames.WARNING));
		}
	}
	
	m_Synchronize.setSelected(p_Connection.getSynchronize());
	}

//---------------------------------------------------------------------------

private void updateConnectionFromDialog (Connection p_Connection) 
	{
	String	l_EncryptedPassword;
	
	if (p_Connection == null) return;
	
	p_Connection.setServer(m_Server.getText());
	if (m_UseDefaultPort.isSelected())
		{
		p_Connection.setPort(null);	
		}
	else p_Connection.setPort((Integer)m_Port.getValue());
	
	p_Connection.setPassiveMode(m_PassiveMode.isSelected());
	
	p_Connection.setUsername(m_Username.getText());
	
	l_EncryptedPassword = m_PasswordEncrypter.encryptPassword(String.valueOf(m_Password.getPassword()));
	p_Connection.setPassword(l_EncryptedPassword);

	p_Connection.setSynchronize(m_Synchronize.isSelected());
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* SwingWorker	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

public synchronized void doConnect ()
	{
	Connection l_Connection;
		
	l_Connection = (Connection) m_Connection.clone();
	l_Connection.setPassword(String.valueOf(m_Password.getPassword()));

	m_TestButton.setIcon(m_ConnectionAbortIcon);
	m_TestButton.setText(Translatrix.getTranslationString(c_TestAbortButton));
	
	m_TestProgress.setIndeterminate(true);
	m_TestProgress.setString (Translatrix.getTranslationString(c_TestProgressConnecting));	
	
	this.closeConnection();
	
	if (m_TestDownloader == null) 
		m_TestDownloader = new ResultDownloader ();
	
	if (m_TestDownloader.openConnection(l_Connection, m_LaboManager))
		{
		try
			{
			m_TestButton.setIcon (m_SuccessIcon);	
			m_TestButton.setText (Translatrix.getTranslationString(c_TestButton));
			m_TestProgress.setString (Translatrix.getTranslationString(c_TestSuccess));
			}
		finally
			{
			m_TestDownloader.closeConnection();
			}
		}
	else
		{
		m_TestButton.setIcon (m_ErrorIcon);
		m_TestButton.setText (Translatrix.getTranslationString(c_TestButton));
		m_TestProgress.setString (Translatrix.getTranslationString(c_TestFailed));
		}
	
	m_TestProgress.setIndeterminate(false);
	}

//---------------------------------------------------------------------------

public synchronized void closeConnection ()
	{
	if ((m_TestDownloader != null) && m_TestDownloader.isConnected())
		m_TestDownloader.closeConnection();
	
	m_TestDownloader = null;
	}

//---------------------------------------------------------------------------

Object ConnectionTest ()
	{
	doConnect ();
	return "";
	}

//---------------------------------------------------------------------------

public void testConnection ()
	{
	this.updateConnectionFromDialog (m_Connection);
	
	m_ConnectionWorker = new SwingWorker() 
	{
	public Object construct() 
		{
		return ConnectionTest ();
		}
	public void start ()
		{
//		m_Aborted = false;
		super.start();
		}
	public void finished ()
		{
		closeConnection ();
		}
	public void interrupt ()
		{
//		m_Aborted = true;
		closeConnection ();
		super.interrupt();
		}
	};

	m_ConnectionWorker.start ();  	
	}

//---------------------------------------------------------------------------

public void abortConnectionTest ()
	{
//	m_Aborted = true;
	if (m_ConnectionWorker != null) m_ConnectionWorker.interrupt();
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

public void setConnection (Connection p_Connection)
	{
	m_Connection = p_Connection;
	this.updateDialogFromConnection(m_Connection);
	}

//---------------------------------------------------------------------------

public Connection getConnection ()
	{
	this.updateConnectionFromDialog(m_Connection);
	return m_Connection;
	}

//---------------------------------------------------------------------------

public boolean wasCanceled ()
	{
	return m_Canceled;	
	}

//---------------------------------------------------------------------------

public void relocalize()
	{
	this.setTitle(Translatrix.getTranslationString(c_Title));
		
	if (m_ServerBorder != null) 
		m_ServerBorder.setTitle(Translatrix.getTranslationString(c_ServerBorder));
	
	if (m_ServerLabel != null)
		m_ServerLabel.setText(Translatrix.getTranslationString(c_ServerLabel));
	
	if (m_UseDefaultPort != null)
		m_UseDefaultPort.setText(Translatrix.getTranslationString(c_DefaultPortLabel));
	
	if (m_PortLabel != null)
		m_PortLabel.setText(Translatrix.getTranslationString(c_PortLabel));
	
	if (m_PassiveMode != null)
		m_PassiveMode.setText(Translatrix.getTranslationString(c_PassiveModeLabel));
		
	if (m_AuthenticationBorder != null)
		m_AuthenticationBorder.setTitle(Translatrix.getTranslationString(c_AuthenticationBorder));
	
	if (m_UsernameLabel != null)
		m_UsernameLabel.setText(Translatrix.getTranslationString(c_UsernameLabel));
	
	if (m_ShowTyping != null)
		m_ShowTyping.setText(Translatrix.getTranslationString(c_ShowTypingLabel));
	
	if (m_PasswordLabel != null)
		m_PasswordLabel.setText(Translatrix.getTranslationString(c_PasswordLabel));	
	
	if (m_SynchronizationBorder != null)
		m_SynchronizationBorder.setTitle(Translatrix.getTranslationString(c_SynchronizationBorder));
	
	if (m_Synchronize != null)
		m_Synchronize.setText(Translatrix.getTranslationString(c_SynchronizeLabel));
	
	if (m_TestBorder != null)
		m_TestBorder.setTitle(Translatrix.getTranslationString(c_TestBorder));
	
	if (m_TestButton != null)
		m_TestButton.setText(Translatrix.getTranslationString(c_TestButton));
	
	if (m_SaveButton != null)
		m_SaveButton.setText (Translatrix.getTranslationString("core.save"));
	
	if (m_CancelButton != null)
		m_CancelButton.setText(Translatrix.getTranslationString("core.cancel"));

	}

//---------------------------------------------------------------------------

public void itemStateChanged (ItemEvent p_Event)
	{
    if (p_Event.getSource().equals(m_UseDefaultPort))
		{
		m_Port.setEnabled(!m_UseDefaultPort.isSelected());	
		}
	else if (p_Event.getSource().equals(m_ShowTyping))
		{
		if (m_ShowTyping.isSelected())
			 m_Password.setEchoChar(c_ClearText);
		else m_Password.setEchoChar(c_EchoCharacter);
		}
	}

//---------------------------------------------------------------------------

public void actionPerformed (ActionEvent p_ActionEvent)
	{
	if (p_ActionEvent.getSource().equals(m_TestButton))
		{
		if (	(m_TestButton.getText().equals(Translatrix.getTranslationString(c_TestButton)))
			 || (m_TestButton.getText().equals(Translatrix.getTranslationString(c_TestSuccess)))
			 || (m_TestButton.getText().equals(Translatrix.getTranslationString(c_TestFailed))))
			{
			this.testConnection();	
			}
		else if (m_TestButton.getText().equals(Translatrix.getTranslationString(c_TestAbortButton)))
			{
			this.abortConnectionTest();	
			}
		}
	else if (p_ActionEvent.getSource().equals(m_CancelButton))
		{
		this.abortConnectionTest();
		this.setVisible (false);
		m_Canceled = true;
		}
	else if (p_ActionEvent.getSource().equals(m_SaveButton))
		{				
		this.abortConnectionTest();
		m_Canceled = false;
		this.setVisible (false);
		}
	}
	
//---------------------------------------------------------------------------
//***************************************************************************
//* End of Class                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

	}
