/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.importexport.gui.importer;

import java.util.Collection;
import java.util.Set;

import javax.persistence.NoResultException;

import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.core.utils.UCMCodeFormatter;
import lu.tudor.santec.gecamed.core.utils.entitymapper.XML2EntityMapper;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.gecamed.office.ejb.session.beans.OfficeManagerBean;
import lu.tudor.santec.gecamed.office.ejb.session.interfaces.OfficeManagerInterface;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * Mapper class to create and save Physician Entities from the given mapper.
 * 
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 * 
 * @version
 * <br>
 *          $Log: PhysicianMapper.java,v $
 *          Revision 1.6  2013-12-27 18:09:25  donak
 *          Cleanup of imports
 *
 *          Revision 1.5  2013-07-15 06:18:36  ferring
 *          logging changed
 *
 *          Revision 1.4  2013-02-14 09:54:54  kutscheid
 *          first commit of the new/remodelled importer implementation
 *          fix some nullpointer exceptions
 * <br>
 *          Revision 1.3 2012-02-07 10:40:00 ferring <br>
 *          printStackTraces replaced by logging <br>
 * <br>
 *          Revision 1.2 2008-09-25 09:43:06 heinemann <br>
 *          fixed copyrights <br>
 * <br>
 *          Revision 1.1 2008-07-14 14:33:25 hermen <br>
 *          first checkin of the xml2physician mapper <br>
 * 
 */
public class PhysicianMapper {

	/** the logger Object for this class */
	private static Logger logger = Logger.getLogger(PhysicianMapper.class.getName());

	private static final String PHYSICIAN_MAPPING = "/lu/tudor/santec/gecamed/importexport/mapping/Physician.properties";
	private OfficeManagerInterface officeManager;

	public PhysicianMapper() {
		this.officeManager = (OfficeManagerInterface) ManagerFactory.getRemote(OfficeManagerBean.class);
	}

	/**
	 * creates Physician entities from the given mapper
	 * 
	 * @param mapper
	 * @return
	 * @throws Exception
	 */
	@SuppressWarnings("unchecked")
	public Collection<Physician> getPhysicians(XML2EntityMapper mapper) throws Exception {
		Set<Physician> physicians;
		physicians = (Set<Physician>) mapper.mapRepeating(Physician.class.getName(), PHYSICIAN_MAPPING);
		return physicians;
	}

	/**
	 * checks if the Physician is allready in the db by using the UCM code
	 * 
	 * @param phy
	 * @return
	 */
	public boolean checkPhysicianExists(Physician phy) {
		String uCMCode = UCMCodeFormatter.extractNumeric(phy.getUcmCode());
		uCMCode = UCMCodeFormatter.formatNumeric(uCMCode);
		Physician existingPhysician = null;
		try {
			existingPhysician = this.officeManager.getPhysicianByUCM(uCMCode);
		} catch (NoResultException nre) {
			return false;
		} catch (Exception e) {
			logger.log(Level.ERROR, e.getMessage(), e);
			return false;
		}

		if (existingPhysician == null)
			return false;
		return true;
	}

	/**
	 * saves the Physician to the Database
	 * 
	 * @param phy
	 * @return
	 */
	public Physician insertPhysician(Physician phy) {
		try {
			return this.officeManager.savePhysician(phy);
		} catch (Exception e) {
			logger.log(Level.ERROR, e.getMessage(), e);
		}
		return null;
	}

}
