/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.utils.querybuilder;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The Condition class implements a generic condition. A condition is defined
 * by a property to be compared against a value. Comparision is done by using
 * a specific operator.
 * 
 * @author nico.mack@tudor.lu
 */

public class Condition extends TreeNode
	{
	private static final long serialVersionUID = 1L;
	private String		m_Property;
	private Object		m_Value;
	private Operator	m_Operator;
	private Boolean		m_CaseSensitive;
	
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************
	
	public static final String c_Null = "NULL";
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Constructors                                       						*
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * Creates a new empty condition.
 */
//---------------------------------------------------------------------------

public Condition ()
	{
	m_Property = "";
	m_Value    = null;
	m_Operator = new Operator (Operator.c_None);
	m_CaseSensitive = true;
	}
		
//---------------------------------------------------------------------------
/**
 * Creates a new condition comparing the specified property with the
 * specified value and using the specified operator to compare them. The new
 * condition will be case sensitive by default.
 * @param p_Property specifies the property that should be compared.
 * @param p_Operator specifies the comparison operator to be used to compare
 * the property.
 * @param p_Value specifies the value to compare the porperty against.
 */
//---------------------------------------------------------------------------

public Condition (String p_Property, Operator p_Operator, Object p_Value)
	{
	this.setProperty (p_Property);
	this.setValue	 (p_Value);
	this.setOperator (p_Operator);
	m_CaseSensitive = true;
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body		                                                        *
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * The getProperty method returns the property to be compared specified for this
 * condition.
 * @return the property being compared with this condition
 */
//---------------------------------------------------------------------------

public String getProperty ()
	{
	return m_Property;
	}

//---------------------------------------------------------------------------
/**
 * The setProperty method sets the property to be compared by this condition
 * @param p_Property specifies the new property to be compared.
 */
//---------------------------------------------------------------------------

public void setProperty (String p_Property)
	{
	m_Property = p_Property;
	}

//---------------------------------------------------------------------------
/**
 * The getValue method returns the value to be compared against the property
 * set for this condition.
 * @return the value to compare property against. If value was set to
 * <code>NULL</code>, the method will return the constant value <code>c_Null</code>
 */
//---------------------------------------------------------------------------

public Object getValue ()
	{
	return (m_Value != null)?m_Value:c_Null;
	}

//---------------------------------------------------------------------------

public void setValue (Object p_Value)
	{
	m_Value = p_Value;
	}

//---------------------------------------------------------------------------

public Operator getOperator ()
	{
	return m_Operator;
	}

//---------------------------------------------------------------------------

public void setOperator (Operator p_Operator)
	{
	m_Operator = p_Operator;
	}

//---------------------------------------------------------------------------

public Boolean getCaseSensitive ()
	{
	return m_CaseSensitive;
	}

//---------------------------------------------------------------------------

public void setCaseSensitive (Boolean p_CaseSensitive)
	{
	m_CaseSensitive = p_CaseSensitive;
	}

//---------------------------------------------------------------------------

public String toString ()
	{
	return getProperty() +
		   getOperator().toString() +
		   getValue().toString();
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* End Of Class                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

}
