/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.utils;

import java.text.ParseException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.text.MaskFormatter;


/**
 * The UCMCodeField swing component eases the input of UCM (Union des Caisses de Maladie) 
 * codes. UCM codes come in the format XXXXXX-YY. The 6 Xs represent the actual UCM code
 * whereas the 2 Ys are merely a checksum calculated from the 6 digits. Checksum
 * calculation is a simple module 97 operation. The UCMCodeField coerces the code
 * format and features an autocomplete function, triggered when hitting the '-' key.
 * @author nmac
 */

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

public class UCMCodeFormatter extends MaskFormatter                                                              
    {
    private static final long serialVersionUID = 1L;
     
    private static Pattern 
    
    c_UCMCodePattern = Pattern.compile ("^(\\d{6})-?(\\d{2})?$",Pattern.CASE_INSENSITIVE);
    
    public static final String c_EmptyField = "00000000";
    
    public static final String c_EmptyString = "000000-00";
    
    
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

public UCMCodeFormatter() 
    {
	try {
		this.setMask("######-##");
		} 
    catch (ParseException p_Exception) 
        {
    		//Can be safely ignored here.
        }
    
    this.setAllowsInvalid(false);
    this.setValidCharacters("0123456789");
    this.setPlaceholderCharacter('0');
    this.setOverwriteMode(true);
    this.setValueContainsLiteralCharacters(true);
    this.setCommitsOnValidEdit(false);
    }
    
//***************************************************************************
//* Class Primitives                                                        *
//***************************************************************************
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------

private String pad (String p_String)
	{
	int 			l_ScanPosition = 0;
	int 			l_ReadPosition = 0;
	char			l_Character;
	boolean 		l_Exhausted    = false;
	StringBuffer	l_StringBuffer = new StringBuffer ();
	
	for (l_ScanPosition = 0; l_ScanPosition < c_EmptyString.length(); l_ScanPosition++)
		{
		l_Character = c_EmptyString.charAt(l_ScanPosition);
		if (l_Character == this.getPlaceholderCharacter())
			{
			if (!l_Exhausted)
				{
				l_Character = p_String.charAt (l_ReadPosition++);
				l_Exhausted = (l_ReadPosition == p_String.length()); 
				}
			l_StringBuffer.append(l_Character);
			}
		else l_StringBuffer.append(l_Character);
		}
	
	return l_StringBuffer.toString();
	}

//---------------------------------------------------------------------------
/**
 * computes the checksum of specified UCM code
 * @param p_UCMCode The six positions of the UCM code holding the representative code digits.
 * @return the checksum corresponding to the specified code
 */
//---------------------------------------------------------------------------

private static String buildChecksum (String p_UCMCode)
    {
    int     l_UCMCode;
    int     l_Modulo;
    String  l_Checksum;
    
    try {
        l_UCMCode = Integer.parseInt(p_UCMCode);
        }
    catch (NumberFormatException p_Exception)
        {
        l_UCMCode = 0;
        }
    
    l_Modulo = l_UCMCode % 97;
    
    l_Checksum = (l_Modulo < 10)?"0":"";
    l_Checksum += l_Modulo;
    
    return l_Checksum;
    }

//---------------------------------------------------------------------------

private boolean isCorrectUCMCode (String p_UCMCode)
	{
	String	l_CodeChecksum;
	String	l_RealChecksum;
	Matcher	l_Matcher;
	boolean	l_IsCorrect	= false;
	
	l_Matcher = c_UCMCodePattern.matcher (p_UCMCode);
	if ((l_Matcher.matches()) && (l_Matcher.groupCount() == 2))
		{
		l_RealChecksum = buildChecksum (l_Matcher.group(1));
		l_CodeChecksum = l_Matcher.group(2);
		l_IsCorrect = l_CodeChecksum.equals(l_RealChecksum);
		}
	return l_IsCorrect;	
	}

//---------------------------------------------------------------------------

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************
//---------------------------------------------------------------------------

public static String extractNumeric (String p_RawString)
	{
	if (p_RawString == null) return p_RawString;

	return p_RawString.replaceAll("[^0-9]", "");
	}

//---------------------------------------------------------------------------

public static String formatNumeric (String p_NumericString)
	{
	Matcher l_Matcher;
	String	l_Checksum;
	String	l_String;
	
	l_String  = p_NumericString;
	l_Matcher = c_UCMCodePattern.matcher (p_NumericString);
	if (l_Matcher.matches())
		{
		if (l_Matcher.groupCount() == 1)	
			{
			l_Checksum = buildChecksum (l_Matcher.group(1));
			l_String = l_Matcher.group(1) + "-" + l_Checksum;				
			}
		else if (l_Matcher.groupCount() == 2)
			{
			l_String = l_Matcher.group(1) + "-" + l_Matcher.group(2);
			}
		}
	return l_String;
	}

//---------------------------------------------------------------------------

public Object stringToValue (String p_UCMCode)
	{
	Matcher 	l_Matcher;
	String	l_Checksum;
	Object	l_Value;
	
	this.setEditValid(false);
	l_Value = p_UCMCode;
	
	if (p_UCMCode != null) 
		{
		l_Matcher = c_UCMCodePattern.matcher (p_UCMCode);
		if (l_Matcher.matches())
			{
			if (l_Matcher.groupCount() == 1)	
				{
				l_Checksum = buildChecksum (l_Matcher.group(1));
				p_UCMCode = l_Matcher.group(1) + "-" + l_Checksum;				
				this.setEditValid(true);
				l_Value = p_UCMCode;
				}
			else if (l_Matcher.groupCount() == 2)
				{
				this.setEditValid(isCorrectUCMCode(p_UCMCode));
				}
			}
		}
	else l_Value = c_EmptyString;
	
	return l_Value;
	}

//---------------------------------------------------------------------------

public String valueToString (Object p_Value)
	{
	Matcher 	l_Matcher;
	String		l_Checksum;
	String		l_String;
	
	this.setEditValid(false);
	l_String = (String) p_Value;
	
	if (p_Value != null) 
		{
		l_Matcher = c_UCMCodePattern.matcher (l_String);
		if (l_Matcher.matches())
			{
			if (l_Matcher.groupCount() == 1)	
				{
				l_Checksum = buildChecksum (l_Matcher.group(1));
				l_String = l_Matcher.group(1) + "-" + l_Checksum;				
				this.setEditValid(true);
				}
			else if (l_Matcher.groupCount() == 2)
				{
				l_String = l_Matcher.group(1) + "-" + l_Matcher.group(2);
				this.setEditValid(isCorrectUCMCode(l_String));
				}
			}
		else
			{
			l_String = this.pad (l_String);
			}
		}
	else l_String = c_EmptyString;
	
	return l_String;
	}

//***************************************************************************
//* End of Class                                                            *
//***************************************************************************

}

