/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.gui.widgets.templates;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.Template;
import lu.tudor.santec.gecamed.core.ejb.session.beans.TemplateManagerBean;
import lu.tudor.santec.gecamed.core.ejb.session.interfaces.TemplateManagerInterface;
import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.utils.ComponentTitledPanel;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

public class TemplateGroupPanel extends JPanel implements ActionListener,
														  Relocalizable
{
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	protected TemplateManagerInterface	m_TemplateInterface;
	private Physician					m_Physician;
	
	private ComponentTitledPanel		m_BorderPanel;
	private JButton						m_AddTemplateButton;
	private JPanel						m_GroupPanel;
	private String						m_TemplateType;
	
	private Vector <TemplatePanel>		m_Templates;
	
	private static Logger m_Logger = Logger.getLogger (TemplateGroupPanel.class.getName());
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************
//---------------------------------------------------------------------------
	
	public static final String c_TemplateAdded   = "TemplateGroupPanel.TemplateAdded";
	public static final String c_TemplateRemoved = "TemplateGroupPanel.TemplateRemoved";
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************
//---------------------------------------------------------------------------

public 	TemplateGroupPanel (String p_TemplateType, Physician p_Physician)
	{
 	
	this.setOpaque(false);
      
    m_Physician = p_Physician;
    
    m_GroupPanel = new JPanel ();
    m_GroupPanel.setOpaque(false);
    m_GroupPanel.setLayout (new BoxLayout (m_GroupPanel,BoxLayout.Y_AXIS));
 	
    m_AddTemplateButton = new JButton ();
    m_AddTemplateButton.setIcon (GECAMedModule.getScaledIcon (GECAMedIconNames.TEMPLATE_ADD,16));
    m_AddTemplateButton.addActionListener (this);

    m_TemplateType = p_TemplateType;

	this.relocalize();
	
    m_BorderPanel = new ComponentTitledPanel (m_AddTemplateButton,m_GroupPanel);
    m_BorderPanel.setOpaque(false);
    
    this.add(m_BorderPanel);
   
	m_Templates = new Vector <TemplatePanel> ();
	
	this.populateWithTemplates();
	}
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Class Primitives		                                                *
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * The private getStatisticsInterface returns an instance of the TemplateManagerBean
 * session bean. On the first call, the TemplateManagerBean will actualy be looked up
 * via JNDI. Once it has been found, the reference to the bean will be stored
 * in a private data member. Doing so avoids JNDI lookups on later calls.
 * @return an instance of the TemplateManagerBean session bean.
 */
//---------------------------------------------------------------------------

private TemplateManagerInterface getTemplateInterface ()
	{
	if (m_TemplateInterface != null) return m_TemplateInterface;

	try {
		m_TemplateInterface = (TemplateManagerInterface) ManagerFactory.getRemote(TemplateManagerBean.class);
		} 
	catch (Exception p_Exception) 
		{
		m_Logger.log (Level.ERROR, "Failed to get TemplateManagerInterface!", p_Exception);
		}

	return m_TemplateInterface;
	}

//---------------------------------------------------------------------------

private Collection <Template> getTemplatesByType (String p_Type)
	{
	Collection <Template>		l_Templates = null;
	TemplateManagerInterface	l_TemplateInterface;
	Integer						l_PhysicianID;
	
	l_TemplateInterface = this.getTemplateInterface();
	if (l_TemplateInterface != null)
		{
		try	{
			l_PhysicianID = (m_Physician != null)?m_Physician.getId():null;
			
			l_Templates = (Collection <Template>) l_TemplateInterface.getTemplatesByTypeAndPhysicianID (p_Type, l_PhysicianID);
			}
		catch (Exception p_Exception)
			{
			l_Templates = null;
			m_Logger.log (Level.ERROR, "Failed to get Templates by type!", p_Exception);
			}
		}
	
	return l_Templates;
	}

//---------------------------------------------------------------------------

private void deleteTemplate (Template p_Template)
	{
	TemplateManagerInterface	l_TemplateInterface;
	
	if ((p_Template == null) || (p_Template.isPersistent() == false)) return;
	
	l_TemplateInterface = this.getTemplateInterface();
	if (l_TemplateInterface != null)
		{
		try	{
			l_TemplateInterface.deleteTemplate(p_Template);
			}
		catch (Exception p_Exception)
			{
			m_Logger.log (Level.ERROR, "Failed to delete Template!", p_Exception);
			}
		}
	}

//---------------------------------------------------------------------------

private void populateWithTemplates ()
	{
	TemplatePanel			l_TemplatePanel;
	Collection <Template>	l_Templates;
	Iterator <Template>		l_TemplateIterator;
	Template				l_Template;
	
	l_Templates = this.getTemplatesByType (m_TemplateType);
	if ((l_Templates != null) && (!l_Templates.isEmpty()))
		{
		l_TemplateIterator = l_Templates.iterator();
		while (l_TemplateIterator.hasNext())
			{
			l_Template = l_TemplateIterator.next();
			l_TemplatePanel = new TemplatePanel (this,l_Template);
			this.addTemplatePanel(l_TemplatePanel);
			}
		}
	else
		{
		l_TemplatePanel = new TemplatePanel (this,null);
		this.addTemplatePanel(l_TemplatePanel);
		}
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

public void addTemplatePanel (TemplatePanel p_Panel)
	{
 	m_Templates.add(p_Panel);
	m_GroupPanel.add(p_Panel);
	
	this.validate();
	
	this.firePropertyChange (c_TemplateAdded, null, p_Panel);
	}

//---------------------------------------------------------------------------

public void removeTemplatePanel (TemplatePanel p_Panel)
	{
// 	TemplatePanel l_Panel;

 	this.deleteTemplate(p_Panel.getTemplate());
	
	if (m_Templates.size() > 1)
		{
		m_Templates.remove(p_Panel);
		m_GroupPanel.remove(p_Panel);
	
//		l_Panel = m_Templates.elementAt(0);
		
		this.validate();
		
		this.firePropertyChange (c_TemplateRemoved, null, p_Panel);
		}
	}

//---------------------------------------------------------------------------

public String getTemplateType ()
	{
	return m_TemplateType;
	}

//---------------------------------------------------------------------------

public Physician getPhysician ()
	{
	return m_Physician;
	}

//---------------------------------------------------------------------------

public Integer getPhysicianId ()
	{
	return (m_Physician != null)?m_Physician.getId():null;
	}

//---------------------------------------------------------------------------

public void actionPerformed (ActionEvent p_Event) 
	{
	TemplatePanel		l_TemplatePanel;
	
	if (p_Event.getSource().equals(m_AddTemplateButton))
		{
		l_TemplatePanel = new TemplatePanel (this,null);
		this.addTemplatePanel(l_TemplatePanel);
		}
	}		

//---------------------------------------------------------------------------

public void relocalize() 
	{
	if (m_AddTemplateButton != null)
		{
		m_AddTemplateButton.setText (Translatrix.getTranslationString ("TemplateType." + m_TemplateType));
		m_AddTemplateButton.setToolTipText (Translatrix.getTranslationString ("TemplateGroupPanel.AddTemplateButtonTip"));
		}
	}

//---------------------------------------------------------------------------	
//***************************************************************************
//* End of Class															*
//***************************************************************************
//---------------------------------------------------------------------------	

public static void main(String[] args)
	{	
	JFrame				l_Window;
	TemplateGroupPanel	l_GroupPanel;
	
	l_Window = new JFrame ();
	
	l_GroupPanel = new TemplateGroupPanel ("invoice",null);
	
	l_Window.add (l_GroupPanel);
	l_Window.pack();
	l_Window.setVisible(true);
	}
}
