/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/*
 * InternalDBExtension.java
 *
 * Created on August 1, 2005, 3:57 PM
 *
 * To change this template, choose Tools | Options and locate the template under
 * the Source Creation and Management node. Right-click the template and choose
 * Open. You can then make changes to the template in the Source Editor.
 */

package lu.tudor.santec.gecamed.core.gui.widgets.hotkeys;

import java.awt.BorderLayout;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Iterator;
import java.util.Vector;

import lu.tudor.santec.gecamed.core.gui.GECAMedAction;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.settings.PluginExtension;

import org.apache.log4j.Logger;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @Version
 * <br>$Log: HotkeySettingsExtension.java,v $
 * <br>Revision 1.13  2013-12-27 18:09:27  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.12  2013-07-15 06:18:39  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.11  2011-03-30 05:44:18  gbosch
 * <br>(ferring:) NullPointerException caught
 * <br>
 * <br>Revision 1.10  2011-03-24 08:50:50  ferring
 * <br>Hotkeys are checked against null. If a hotkey is null it is changed to empty string
 * <br>
 * <br>Revision 1.9  2011-03-04 09:53:11  ferring
 * <br>Hotkeys for physicians added.
 * <br>It is now possible to make a hotkey consisting of key + modifiers.
 * <br>
 * <br>Revision 1.8  2008-12-22 09:40:00  hermen
 * <br>fixed scrolling
 * <br>
 * <br>Revision 1.7  2008-12-19 08:42:14  hermen
 * <br>fixed scrolling issues
 * <br>
 * <br>Revision 1.6  2008-09-25 09:43:07  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.5  2008-01-15 09:29:38  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * <br>Revision 1.4  2007-12-04 14:59:04  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class HotkeySettingsExtension extends PluginExtension implements ItemListener
    {

	private static final long serialVersionUID = 1L;
	
	/** the logger Object for this class */
	private static Logger logger = Logger.getLogger(HotkeySettingsExtension.class.getName());
	
	private HotkeyPanel hotKeyPanel;
	private Vector<GECAMedAction> actions;
	private String moduleName;

    
//***************************************************************************
//* Constructor                                                             *
//***************************************************************************

//---------------------------------------------------------------------------
/** Creates a new instance of I18nPlugin                                   */
//---------------------------------------------------------------------------

public HotkeySettingsExtension(GECAMedModule module) 
    {
    super (module.getTranslatedName());
    this.setName(module.getTranslatedName());
    this.setLayout( new BorderLayout());
    this.hotKeyPanel = new HotkeyPanel();
    this.add(this.hotKeyPanel, BorderLayout.CENTER);
    this.moduleName  = module.getName();
    this.actions = module.getModuleActions();
    }

public HotkeySettingsExtension(String name, Vector<GECAMedAction> actions) 
{
	super (name);
	this.setName(name);
	this.setLayout( new BorderLayout());
	this.hotKeyPanel = new HotkeyPanel();
	this.add(this.hotKeyPanel, BorderLayout.CENTER);
	this.actions = actions;
}
    
//---------------------------------------------------------------------------

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************

//---------------------------------------------------------------------------
/**
 * extensionAdded method will be called automatically AFTER the extension has been
 * added to the plugin. Method could be used for instance to add stationary
 * values to those of the plugin.
 */
//---------------------------------------------------------------------------

protected void extensionAdded ()
    {
	String hotKey;
	for (Iterator<GECAMedAction> iter = actions.iterator(); iter.hasNext();) {
		GECAMedAction element = iter.next();
		hotKey = element.getHotKey();
		setStationary(createHotkeySettingName(element), hotKey == null ? "" : hotKey);
		setStationary(createShortcutSettingName(element), element.getShortcut());
	}
    }
    
//---------------------------------------------------------------------------

public void revertToDefaults ()
    {
	for (Iterator<GECAMedAction> iter = actions.iterator(); iter.hasNext();) {
		GECAMedAction element = iter.next();
		if (element.getValue(GECAMedAction.NOT_EDITABLE) != null) {
			continue;
		}
		setValue(createHotkeySettingName(element), getDefault(createHotkeySettingName(element)));
		setValue(createShortcutSettingName(element), getDefault(createShortcutSettingName(element)));
	}
    reflectSettings ();
    }

//---------------------------------------------------------------------------

/* (non-Javadoc)
 * @see lu.tudor.santec.settings.PluginExtension#updateSettings()
 */
public void updateSettings ()
    {
		for (Iterator<GECAMedAction> iter = actions.iterator(); iter.hasNext();) {
			GECAMedAction element = iter.next();
			if (element.getValue(GECAMedAction.NOT_EDITABLE) != null) {
				continue;
			}
			setValue(createHotkeySettingName(element), element.getHotKey() != null ? element.getHotKey() : "");
			setValue(createShortcutSettingName(element), element.getShortcut());
		}
    }
    
//---------------------------------------------------------------------------

/* (non-Javadoc)
 * @see lu.tudor.santec.settings.PluginExtension#reflectSettings()
 */
public void reflectSettings ()
    {
		for (Iterator<GECAMedAction> iter = actions.iterator(); iter.hasNext();) {
			GECAMedAction element = (GECAMedAction) iter.next();
			if (element.getValue(GECAMedAction.NOT_EDITABLE) != null) {
				continue;
			}
			
			try {
				if (getValue(createHotkeySettingName(element)) instanceof Integer){
					element.setHotKey((Integer)getValue(createHotkeySettingName(element)));
				} else {					
					element.setHotKey((String)getValue(createHotkeySettingName(element)));
				}
				element.setShortcut((Integer)getValue(createShortcutSettingName(element)));				
			} catch (Exception e) {
				 logger.warn(e.getLocalizedMessage());
			}
		}
		this.hotKeyPanel.showActions(actions);
		this.validate();
    }

public void itemStateChanged(ItemEvent e) {

}

private String createHotkeySettingName(GECAMedAction a) {
	return this.moduleName + "#" + a.getName() + "#HOTKEY" ;
}

private String createShortcutSettingName(GECAMedAction a) {
	return this.moduleName + "#" + a.getName() + "#SHORTCUT";
}

//---------------------------------------------------------------------------

//***************************************************************************
//* End Of Class                                                            *
//***************************************************************************

}
