/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/*
 * Author: Johannes Hermen Tudor/Santec
 * Mail: johannes.hermen@tudor.lu
 * Created: Jan 17, 2005
 *
 */
package lu.tudor.santec.gecamed.core.gui.widgets;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.util.Locale;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;

import lu.tudor.santec.gecamed.core.gui.GECAMedFonts;
import lu.tudor.santec.i18n.Translatrix;


/**
 * represents a JPanel with a JComponent and an JLabel located over it to
 * describe it. can be used to add JTextFields, etc. with their description to
 * a form
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @Version
 * <br>$Log: NamedComponent.java,v $
 * <br>Revision 1.13  2011-07-06 08:29:50  ferring
 * <br>changes for the examen clinique (mainly sub form component added)
 * <br>
 * <br>Revision 1.12  2010-11-26 13:16:39  ferring
 * <br>the NamedComponent will add a space before the ':', if the local is French
 * <br>
 * <br>Revision 1.11  2010-07-08 14:11:55  ferring
 * <br>a constructor added to change the alignment of the label
 * <br>
 * <br>Revision 1.10  2009-03-20 10:20:05  hermen
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.9  2008-12-05 17:34:23  heinemann
 * <br>moved some basic classes to lu.tudor.santec.widgets
 * <br>
 * <br>Revision 1.8  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.7  2008-04-16 14:24:28  heinemann
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.6  2008-01-15 09:29:38  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * <br>Revision 1.5  2007-12-06 14:46:44  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class NamedComponent extends JPanel
{

    //~ Static parameter options ===============================================	

	public static final int DIRECTION_VERTICAL = JLabel.VERTICAL;
	public static final int DIRECTION_HORIZONTAL = JLabel.HORIZONTAL;
	public static final int ALIGNMENT_TOP = JLabel.TOP;
	public static final int ALIGNMENT_CENTER = JLabel.CENTER;
	public static final int ALIGNMENT_BOTTOM = JLabel.BOTTOM;
	
	
    //~ Static fields/initializers =============================================
	
    private static final long serialVersionUID = 1L;
    private static final int vTextAlignment = JLabel.LEFT;
    private static final int hTextAlignment = JLabel.BOTTOM;
    private static final int preferedHeight = 22;
    
    
    //~ Instance fields ========================================================

    private JComponent component;
    private JLabel nameLabel;

    //~ Constructors ===========================================================
    /**
     * generates a JPanel with a JComponent and an JLabel located over it to
     * describe it.
     *
     * @param name The Text of the shown Label
     * @param component The JComponent shown under the Label
     */
    public NamedComponent(String name, JComponent component) {
    	/* ================================================== */
    	this(name, component, true, JLabel.VERTICAL);
		/* ================================================== */
    }

    /**
     * generates a JPanel with a JComponent and an JLabel located over it to
     * describe it.
     * @param name
     * @param component
     * @param bold
     */
    public NamedComponent(String name, JComponent component, boolean bold) {
	this(name, component, true, JLabel.VERTICAL);
    }
    

    /**
     * generates a JPanel with a JComponent and an JLabel located over it to
     * describe it.
     * @param name
     * @param component
     * @param bold
     */
    public NamedComponent(String name, JComponent component, boolean bold, int direction) {
    	this(name, component, bold, direction, hTextAlignment);
    }
    
    /**
     * generates a JPanel with a JComponent and an JLabel located over it to
     * describe it.
     * @param name
     * @param component
     * @param bold
     * @param labelAlignment the vertical alignment of the label (top, center, bottom)
     */
    public NamedComponent(String name, JComponent component, boolean bold, int direction, int labelAlignment)
    {
    	this.setOpaque(false);
//        this.setLayout(new GridLayout(0, 1));
    	this.setLayout(new BorderLayout());
        if (!"".equals(name))
        	
        	nameLabel = new JLabel(name + 
        			// add a space, if local is France
        			(Translatrix.getLocale().equals(Locale.FRANCE) ? " :" : ":"));
        else
        	nameLabel = new JLabel("");
        nameLabel.setHorizontalAlignment(vTextAlignment);
        nameLabel.setVerticalAlignment(labelAlignment);
        nameLabel.setOpaque(false);
        /* ------------------------------------------------------- */
        if (!bold)
        	nameLabel.setFont(GECAMedFonts.PLAIN_LABEL_FONT);
        /* ------------------------------------------------------- */
        this.setComponent(component);
        if (direction == JLabel.VERTICAL)
            this.add(this.nameLabel, BorderLayout.NORTH);
        else 
            this.add(this.nameLabel, BorderLayout.WEST);
    }

    /**
     * generates a JPanel with a JComponent and an JLabel located over it to
     * describe it.
     *
     * @param name The Text of the shown Label
     * @param component The JComponent shown under the Label
     * @param textColor the color of the labeltext
     */
    public NamedComponent(String name, JComponent component, Color textColor)
    {
        this.setLayout(new GridLayout(0, 1));
        nameLabel = new JLabel(name + 
    			// add a space, if local is France
    			(Translatrix.getLocale().equals(Locale.FRANCE) ? " :" : ":"));
        nameLabel.setHorizontalAlignment(vTextAlignment);
        nameLabel.setVerticalAlignment(hTextAlignment);
        nameLabel.setForeground(textColor);
        this.component = component;
        this.component.setPreferredSize(new Dimension(this.component.getPreferredSize().width, preferedHeight));
        this.add(this.nameLabel);
        this.add(this.component);
    }


	/* (non-Javadoc)
	 * @see javax.swing.JComponent#setEnabled(boolean)
	 */
	@Override
	public void setEnabled(boolean enabled) {
		/* ====================================================== */
		super.setEnabled(enabled);
		nameLabel.setEnabled(enabled);
		component.setEnabled(enabled);
		/* ====================================================== */
	}

	public String getLabel ()
	{
		return nameLabel.getText();
	}
	
	public void setLabel (String text)
	{
		nameLabel.setText(text);
	}
	
	public JComponent getComponent ()
	{
		return component;
	}
	
	public void setComponent (JComponent component)
	{
		if (this.component != null)
			this.remove(this.component);
        this.component = component;
        this.component.setPreferredSize(new Dimension(this.component.getPreferredSize().width, preferedHeight));
        this.add(this.component, BorderLayout.CENTER);
	}
}
