/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as
 * published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * (L-GPL)
 * along with GECAMed. If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor
 * (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.gui.widgets;

import java.awt.Color;

import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;

/**
 * A JTextField with a limit number of characters that can be inserted.
 * Use this instead of the normal TextField, to avoid longer Strings 
 * than the Database can handle.
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @Version
 * <br>$Log: LimitTextField.java,v $
 * <br>Revision 1.5  2009-05-29 13:51:21  hermen
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.4  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.3  2008-01-15 09:29:38  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * <br>Revision 1.2  2007-12-06 14:46:44  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class LimitTextField extends JTextField
{
	//~ Static fields/initializers =============================================
	
	private static final long	serialVersionUID	= 1L;
	
	//~ Instance fields ========================================================
	private int					limit				= 30;
	
	private boolean				mandantory;
	
	private Color				bgColor;
	private Color				emptyColor			= new Color(255, 230, 230);
	
	
	//~ Constructors ===========================================================
	/**
	 * creates a new LimitTextField
	 */
	public LimitTextField ()
	{
		super();
		this.init();
	}
	
	
	/**
	 * creates a new LimitTextField
	 */
	public LimitTextField (int limit)
	{
		this(limit, false);
	}
	
	
	public LimitTextField (int limit, boolean mandantory)
	{
		super();
		this.limit = limit;
		this.mandantory = mandantory;
		this.init();
	}
	
	
	//~ Methods ================================================================
	/**
	 * do initial settings
	 */
	private void init ()
	{
		this.bgColor = this.getBackground();
		final LimitValidation doc = new LimitValidation();
		this.setDocument(doc);
		doc.addDocumentListener(new DocumentListener()
		{
			public void changedUpdate (DocumentEvent e)
			{
				docChanged();
			}
			
			
			public void insertUpdate (DocumentEvent e)
			{
				docChanged();
			}
			
			
			public void removeUpdate (DocumentEvent e)
			{
				docChanged();
			}
			
			
			private void docChanged ()
			{
				if (mandantory)
				{
					if (doc.getLength() == 0)
					{
						setBackground(emptyColor);
					}
					else
					{
						setBackground(bgColor);
					}
				}
			}
		});
	}
	
	
	/* (non-Javadoc)
	 * @see javax.swing.text.JTextComponent#setText(java.lang.String)
	 */
	@Override
	public void setText (String t)
	{
		if (mandantory)
		{
			if (t == null || t.trim().length() == 0)
			{
				setBackground(emptyColor);
			}
			else
			{
				setBackground(bgColor);
			}
		}
		super.setText(t);
	}
	
	//~ Inner Classes ==========================================================
	
	public class LimitValidation extends PlainDocument
	{
		//~ Static fields/initializers =========================================
		
		private static final long	serialVersionUID	= 1L;
		
		
		//~ Constructors =======================================================
		/**
		 * Constructor for the Validationdocument
		 */
		public LimitValidation ()
		{
			super();
		}
		
		
		//~ Methods ============================================================
		
		public void insertString (int offset, String str, AttributeSet attr)
				throws BadLocationException
		{
			int length = getLength() + str.length();
			
			if ((length) <= limit)
				super.insertString(offset, str, attr);
		}
	}
	
}
