/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.gui.widgets;

import javax.swing.JTextField;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.PlainDocument;


/**
 * This Class implements a JTextfield that allows only integers to be inserted.
 * It provides setValue(int) and getValue(int) additional to setText getText
 * for easier int handling
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @Version
 * <br>$Log: IntegerField.java,v $
 * <br>Revision 1.7  2008-09-25 09:43:06  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.6  2008-08-19 10:25:08  heinemann
 * <br>cleanup
 * <br>
 * <br>Revision 1.5  2008-01-15 09:29:38  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * <br>Revision 1.4  2007-12-06 14:46:44  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
public class IntegerField extends JTextField
{
    //~ Static fields/initializers =============================================

    private static final long serialVersionUID = 1L;

    //~ Constructors ===========================================================
    /**
     * creates a new IntegerField
     */
    public IntegerField()
    {
        super();
        this.init();
    }

    /**
     * creates a new IntegerField
     *
     * @param text the Text to be shown
     */
    public IntegerField(String text)
    {
        super(text);
        this.init();
    }

    /**
     * creates a new IntegerField
     *
     * @param columns the length of the Field
     */
    public IntegerField(int columns)
    {
        super(columns);
        this.init();
    }

    /**
     *  creates a new IntegerField
     *
     * @param text the text to be shown
     * @param columns the length of the Field
     */
    public IntegerField(String text, int columns)
    {
        super(text, columns);
        this.init();
    }

    //~ Methods ================================================================
    /**
     * sets the Value of the Field
     *
     * @param value the int value to be shown
     */
    public void setValue(int value)
    {
        this.setText(value + "");
    }

    /**
     * gets the shown int value
     *
     * @return the shown int value
     */
    public int getValue()
    {
        try {
            return Integer.parseInt(this.getText());
        } catch (Exception e) {
            return 0;
        }
    }
    
    /**
     * gets the shown Integer value
     *
     * @return the shown Integer value
     */
    public Integer getValueInteger()
    {
        try {
            return Integer.parseInt(this.getText());
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * do initial settings
     */
    private void init()
    {
        this.setDocument(new IntegerValidation());
        this.setToolTipText("insert Integer Numbers only!");
    }

    //~ Inner Classes ==========================================================
    /**
     * a Document for the IntegerField that only allows integer Numbers to be
     * inserted.
     */
    public class IntegerValidation extends PlainDocument
    {
        //~ Static fields/initializers =========================================

        private static final long serialVersionUID = 1L;

        //~ Constructors =======================================================
        /**
         * Constructor for the Validationdocument
         */
        public IntegerValidation()
        {
            super();
        }

        //~ Methods ============================================================
        /**
         * overwrites insertString of Class PlainDocument to only allow
         * integers
         *
         * @param offset offset: Position
         * @param str str: the String
         * @param attr attr: Attributset
         *
         * @throws BadLocationException
         */
        public void insertString(int offset, String str, AttributeSet attr)
            throws BadLocationException
        {
            if ((str != null) && !str.equals("")) {
                try {
                    Integer.parseInt(str);
                    super.insertString(offset, str, attr);

                    //IntegerField.this.setBackground(VALID_BACKGROUND);
                } catch (NumberFormatException e) {
//                    java.awt.Toolkit.getDefaultToolkit().beep();

                    //IntegerField.this.setBackground(INVALID_BACKGROUND);
                }
            } else {
                //IntegerField.this.setBackground(VALID_BACKGROUND);
            }
        }
    }
}
