/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.gui.widgets;

/*
 * GenericListBox.java
 *
 * Created on April 3, 2006
 */

import java.awt.Component;
import java.awt.event.KeyListener;
import java.awt.event.MouseListener;

import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.utils.TableSorter;
import lu.tudor.santec.i18n.Relocalizable;

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

public class GenericListBox extends JScrollPane implements Relocalizable
    {
    /**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private JTable           				m_Table;
    private AbstractTableModel				m_TableModel;
    private TableSorter						m_Sorter;
    private int[]							m_ColumnWidths;   
    
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************

    private static int c_CellMargin = 7;
 
//***************************************************************************
//* Internal Class Definitions	                                            *
//***************************************************************************

//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

public GenericListBox (AbstractTableModel p_ListModel)
    {	
	super ();
      
	m_ColumnWidths = null;
	m_TableModel   = p_ListModel;
    
	m_Sorter = new TableSorter ();
	m_Sorter.setTableModel(m_TableModel);
    
	m_Table = new JTable (m_Sorter);
	m_Table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
	m_Table.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
        
	m_Table.setColumnSelectionAllowed (false);
	m_Table.setShowGrid (false);
 
	this.getViewport().setOpaque(false);
  	this.setBackground(GECAMedColors.c_ScrollPaneBackground);

	m_Sorter.setTableHeader(m_Table.getTableHeader());
    
    this.setViewportView (m_Table);   
    }

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Primitives		                                                *
//***************************************************************************
//---------------------------------------------------------------------------

public void addListSelectionListener (ListSelectionListener p_Listener)
    {
    ListSelectionModel	l_Selection;	
    
    l_Selection = m_Table.getSelectionModel();
    l_Selection.addListSelectionListener (p_Listener);
    }

//---------------------------------------------------------------------------

public void addMouseListener (MouseListener p_Listener)
    {
	m_Table.addMouseListener (p_Listener);
    }    

//---------------------------------------------------------------------------

public void addKeyListener (KeyListener p_Listener)
    {
	m_Table.addKeyListener (p_Listener);
    }    
 
//---------------------------------------------------------------------------

public int modelIndex (int p_RowInView)
	{
	int	l_ModelRow;
	
	l_ModelRow = p_RowInView;
	
	if ((m_Sorter != null) && m_Sorter.isSorting())
		{
		l_ModelRow = m_Sorter.modelIndex(l_ModelRow);
		}
	
	return l_ModelRow;
	}

//---------------------------------------------------------------------------

public int viewIndex (int p_RowInModel)
	{
	int	l_ViewRow;
	
	l_ViewRow = p_RowInModel;
	
	if ((m_Sorter != null) && m_Sorter.isSorting())
		{
		l_ViewRow = m_Sorter.viewIndex(p_RowInModel);
		}
	
	return l_ViewRow;
	}

//---------------------------------------------------------------------------
//Sets the preferred width of the visible column specified by vColIndex. The column
// will be just wide enough to show the column head and the widest cell in the column.
// margin pixels are added to the left and right
// (resulting in an additional width of 2*margin pixels).
//---------------------------------------------------------------------------

private void packColumn (int p_Column, int p_Margin) 
	{
	DefaultTableColumnModel	l_ColumnModel;
	TableColumn 				l_Column;
	TableCellRenderer		l_Renderer;
	Component				l_Component;
	
	int		   				l_Width;
    int						l_Row;
	
	l_ColumnModel = (DefaultTableColumnModel)m_Table.getColumnModel();
	l_Column = l_ColumnModel.getColumn(p_Column);
	l_Width = 0;

    // First step consists in getting width of column header

    l_Renderer = l_Column.getHeaderRenderer();
    if (l_Renderer == null) 
    		{
    		l_Renderer = m_Table.getTableHeader().getDefaultRenderer();
    		}
    
    l_Component = l_Renderer.getTableCellRendererComponent(m_Table,
    													    l_Column.getHeaderValue(), 
    													    false, false, 0, 0);
    l_Width = l_Component.getPreferredSize().width;

    // Next we're going to iterate through all the rows of this columns, in order to
    // to find the widest one

    for (l_Row=0; l_Row < m_Table.getRowCount(); l_Row++) 
    		{
    		l_Renderer = m_Table.getCellRenderer(l_Row, p_Column);
         
       		try {
    			l_Component = l_Renderer.getTableCellRendererComponent(m_Table, 
    																   m_Table.getValueAt(l_Row, p_Column), 
    				                                               	   false, false, l_Row, p_Column);
    			}
    		catch (Exception p_Exception)
    			{
    			// TODO Nasty Workaround
    			// Invoice Renderer sometimes throws unexplicable NullPointerExceptions.
    			// Not catching them results in Table not being properly packed.
    			}
 		
    		l_Width = Math.max (l_Width, l_Component.getPreferredSize().width);
    		}

    // Add margin
    l_Width += 2*p_Margin;

    // Set the width
    l_Column.setPreferredWidth(l_Width);
	}

//---------------------------------------------------------------------------
//Returns the preferred height of a row.
// The result is equal to the tallest cell in the row.
//---------------------------------------------------------------------------

private void packRow (int p_Row, int p_Margin) 
	{
    int					l_RowHeight;
    int					l_ComponentHeight;
	int					l_Column;
    TableCellRenderer	l_Renderer;
    Component			l_Component;
    
    // Get the current default height for all rows
    l_RowHeight = m_Table.getRowHeight();

    // Determine highest cell in the row
    for (l_Column = 0; l_Column < m_Table.getColumnCount(); l_Column++) 
    		{
    		l_Renderer = m_Table.getCellRenderer(p_Row, l_Column);
    		l_Component = m_Table.prepareRenderer(l_Renderer, p_Row, l_Column);
    		l_ComponentHeight = l_Component.getPreferredSize().height + 2*p_Margin;
    		l_RowHeight = Math.max (l_RowHeight, l_ComponentHeight);
    		}
    
    if (m_Table.getRowHeight() != l_RowHeight)
    	m_Table.setRowHeight (p_Row,l_RowHeight);
	}

//---------------------------------------------------------------------------

public int getSelectedRow ()
	{
	int[] l_SelectedRows;
		
	l_SelectedRows = this.getSelectedRows();	
	if ((l_SelectedRows != null) && (l_SelectedRows.length == 1))
		return  l_SelectedRows [0];
	else return -1;
	}

//---------------------------------------------------------------------------
/**
 * Method is part of the Relocalizable interface. The method does everything
 * required to reflect changes of active Locale
 */
//---------------------------------------------------------------------------

public void relocalize ()
    {
	TableColumn 		l_Column;
    int					l_NumberOfColumns;
	int					l_ColumnIndex;
 
     l_NumberOfColumns = m_TableModel.getColumnCount();
    
    for (l_ColumnIndex = 0; l_ColumnIndex < l_NumberOfColumns; l_ColumnIndex++) 
    	{
    	l_Column = m_Table.getColumnModel().getColumn(l_ColumnIndex);
    	l_Column.setHeaderValue(m_TableModel.getColumnName(l_ColumnIndex));
    	}
    }

//---------------------------------------------------------------------------

protected boolean setSelection (int p_Row)
	{
	ListSelectionModel	l_Selection;	
	JScrollBar			l_ScrollBar;
	float				l_ScrollValue;
	boolean 			l_Selected = false;
	    
	if ((p_Row >= 0) && (p_Row < m_Table.getRowCount()))
        {
		l_Selection = m_Table.getSelectionModel();
		l_Selection.setSelectionInterval (p_Row,p_Row);
			
		l_ScrollBar = getVerticalScrollBar ();
			
		l_ScrollValue  = (float) p_Row / (float) m_TableModel.getRowCount();
		l_ScrollValue *= (l_ScrollBar.getMaximum() - l_ScrollBar.getMinimum()); 
			
		l_ScrollBar.setValue ((int)l_ScrollValue);
			
		l_Selected = true;
		m_Table.revalidate();
        }
	   
	return (l_Selected);
	}


//***************************************************************************
//* Class Body		                                                    	*
//***************************************************************************
//---------------------------------------------------------------------------

public void setColumnWidths (int[] p_ColumnWidths)
	{
	if ((p_ColumnWidths != null) && (p_ColumnWidths.length >= m_TableModel.getColumnCount()))
		{
		m_Table.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
		m_ColumnWidths = p_ColumnWidths;			
		}
	else
		{
		m_Table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		m_ColumnWidths = null;	
		}
	}

//---------------------------------------------------------------------------
/**
 * doLayout is the main entry point for JTable resizing and row/column space
 * distribution. The ActListBox class overrides the method to impose its own
 * distribution scheme. Width of columns is expressed as percents and we want
 * the table to reflect this distribution. 
 */
//---------------------------------------------------------------------------

public void doLayout ()
	{
    TableColumnModel	l_ColumnModel;
	TableColumn  		l_Column;
	double				l_Width;
	int					l_ColumnWidth;
	int					l_Index;
	
	super.doLayout ();
	
	if (m_ColumnWidths == null) return;
	
	l_Width = (double) m_Table.getWidth () / 100;
	
	l_ColumnModel = m_Table.getColumnModel ();
	
	for (l_Index = 0; l_Index < m_TableModel.getColumnCount(); l_Index ++)
		{
		l_Column = l_ColumnModel.getColumn (l_Index);
		if (l_Column != null)
			{
			l_ColumnWidth = (int)(l_Width * m_ColumnWidths[l_Index]);
			l_Column.setPreferredWidth(l_ColumnWidth);
			}
		}
	}

//---------------------------------------------------------------------------

public void setRenderer (DefaultTableCellRenderer p_Renderer)
	{
    int l_Count;
	
	for (l_Count=0; l_Count < m_TableModel.getColumnCount(); l_Count++)
		{
		if (m_TableModel.getColumnClass(l_Count) != null)	
			m_Table.setDefaultRenderer (m_TableModel.getColumnClass(l_Count), p_Renderer);
		}
	}

//---------------------------------------------------------------------------

public void setSorting (int p_Column, int p_Direction)
	{
	int	l_Direction = TableSorter.NOT_SORTED;
	
	if ((p_Column >= 0) && (p_Column < m_TableModel.getColumnCount()))
		{
		l_Direction = (p_Direction > 0)?TableSorter.ASCENDING:TableSorter.NOT_SORTED;
		if (l_Direction == TableSorter.NOT_SORTED)
			l_Direction = (p_Direction < 0)?TableSorter.DESCENDING:TableSorter.NOT_SORTED;
	
		m_Sorter.setSortingStatus(p_Column, l_Direction);	
		}
	}

//---------------------------------------------------------------------------

public JTable getTable ()
	{
	return m_Table;
	}

//---------------------------------------------------------------------------

public AbstractTableModel getTableModel ()
	{
	return m_TableModel;
	}

//---------------------------------------------------------------------------

public int[] getSelectedRows ()
    {
    int []      l_SelectedRows;   
    int         l_Selection;
    int			l_ModelRow;
    
    int []      l_ModelRows = null;   
   
    l_Selection = m_Table.getSelectedRowCount();
    
    if (l_Selection > 0)
        {
        l_SelectedRows  = m_Table.getSelectedRows ();
        l_ModelRows     = new int [l_Selection];
        
        for (l_Selection = 0; l_Selection < l_SelectedRows.length; l_Selection++)
            {
        	l_ModelRow = this.modelIndex(l_SelectedRows [l_Selection]);
        	l_ModelRows [l_Selection] = l_ModelRow;
            }
        }

    return l_ModelRows;
    }

//---------------------------------------------------------------------------

public void setSelectedRow (int p_Row)
	{
	int	l_ViewRow;
		
	if (p_Row >= 0)
		{
		l_ViewRow = this.viewIndex (p_Row);
		this.setSelection(l_ViewRow);
		}
	}

//---------------------------------------------------------------------------

public void packColumns ()
	{
	int l_Column;
	
	for (l_Column=0; l_Column < m_Table.getColumnCount(); l_Column++)
		{
		packColumn (l_Column,c_CellMargin);
		}
	}

//---------------------------------------------------------------------------

public void packRows ()
	{
	int l_Row;
	
	for (l_Row=0; l_Row < m_Table.getRowCount(); l_Row++)
		{
		packRow (l_Row,c_CellMargin);
		}
	}

//***************************************************************************
//* End of Class															*
//***************************************************************************
}
