/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.core.gui.utils;

import java.io.File;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.filechooser.FileFilter;

/**
 * The RegexFileFilter class implements a FileFilter for FileChooser dialogs
 * which accepts only files matching a specific pattern. The file name pattern
 * to be accepted is defined by a regular expression.
 * @author  nico.mack@tudor.lu
 */

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

public class RegexFileFilter extends FileFilter
    {
    Pattern m_FilePattern;
    String	m_Description; 
    
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************
     
//---------------------------------------------------------------------------
/**
  * Creates a new instance of RegexFileFilter matching all files
  */
//---------------------------------------------------------------------------

public RegexFileFilter() 
    {
    super ();
    
	m_FilePattern = Pattern.compile (".*",Pattern.CASE_INSENSITIVE); 
    m_Description = ".*";
    }
 
//---------------------------------------------------------------------------
/**
 * Create a new instance of RegexFileFilter accepting only files matching the
 * specified file pattern. File pattern may be case sensitive or case insensitive
 * @param p_Pattern specifies the regular expression used to match filenames. Matching filenames
 * will be accepted by the accept method
 * @param p_CaseSensisitive specifies whether match pattern is case sensitive or not
 */
//---------------------------------------------------------------------------

public RegexFileFilter (String p_Pattern, boolean p_CaseSensitive)
    {
    setFilePattern (p_Pattern,p_CaseSensitive);
    m_Description = p_Pattern;
    }

//---------------------------------------------------------------------------
/**
 * Create a new instance of RegexFileFilter accepting only files matching the
 * specified file pattern. File pattern may be case sensitive or case insensitive
 * @param p_Pattern specifies the regular expression used to match filenames. Matching filenames
 * will be accepted by the accept method
 * @param p_CaseSensisitive specifies whether match pattern is case sensitive or not
 */
//---------------------------------------------------------------------------

public RegexFileFilter (String p_Pattern, String p_Description, boolean p_CaseSensitive)
    {
    setFilePattern (p_Pattern,p_CaseSensitive);
    m_Description = p_Description;
    }

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************

//---------------------------------------------------------------------------
/**
 * Sets the file match pattern for this instance. File pattern may be case sensitive or case insensitive
 * @param p_Pattern specifies the regular expression used to match filenames. Matching filenames
 * will be accepted by the accept method
 * @param p_CaseSensisitive specifies whether match pattern is case sensitive or not
 */
//---------------------------------------------------------------------------

public void setFilePattern (String p_Pattern, boolean p_CaseSensitive)
    {
    if (p_Pattern == null) p_Pattern = ".*";
        
    if (p_CaseSensitive)
         m_FilePattern = Pattern.compile (p_Pattern);
    else m_FilePattern = Pattern.compile (p_Pattern,Pattern.CASE_INSENSITIVE);
    }

//---------------------------------------------------------------------------
/**
 * method is part of the FileFilter interface. Method returns true if specified
 * file matches the previously set file pattern.
 * @return <CODE>true</CODE> if filename of specified file matches previously set pattern, <CODE>false</CODE>
 * otherwise.
 */
//---------------------------------------------------------------------------

public boolean accept (File p_File) 
    {
    if (p_File.isDirectory()) return true;
    	
    if (p_File.isFile())
        {
        Matcher l_Matcher = m_FilePattern.matcher (p_File.getName());
        return l_Matcher.matches();
        }
    else return true;    
    }     

//---------------------------------------------------------------------------
/**
 * sets the description for this FileFilter
 * @param p_Description specifies the description for this FileFilter
 */
//---------------------------------------------------------------------------

public void setDescription (String p_Description)
	{
	m_Description = p_Description;
	}

//---------------------------------------------------------------------------
/**
 * returns the description of this FileFilter
 * @return description of this FileFilter
 */
//---------------------------------------------------------------------------

public String getDescription() 
	{
	return m_Description;
	}

//***************************************************************************
//* End of Class                                                            *
//***************************************************************************
}
