/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/**
 * 
 */
package lu.tudor.santec.gecamed.core.gui;

import java.io.ByteArrayOutputStream;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.util.Enumeration;
import java.util.Stack;
import java.util.Timer;
import java.util.TimerTask;

import javax.naming.NamingException;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.log.Log;
import lu.tudor.santec.gecamed.core.ejb.entity.beans.log.LogType;
import lu.tudor.santec.gecamed.core.ejb.session.beans.LogManagerBean;
import lu.tudor.santec.gecamed.core.ejb.session.interfaces.LogManager;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;

import org.apache.log4j.Logger;

/**
 * Class to create Log messages.
 * <br>If the connection to the application server can not be granted
 * the messages are stored in a stack and an timer is started to
 * deliver them later.
 * 
 * @author martin.heinemann@tudor.lu
 *
 *
 * @version
 * <br>$Log: GECAMedLog.java,v $
 * <br>Revision 1.14  2013-12-27 18:09:26  donak
 * <br>Cleanup of imports
 * <br>
 * <br>Revision 1.13  2013-07-15 06:18:35  ferring
 * <br>logging changed
 * <br>
 * <br>Revision 1.12  2010-05-03 14:18:56  hermen
 * <br>enhanced logging
 * <br>
 * <br>Revision 1.11  2010-05-03 13:43:29  hermen
 * <br>enhanced logging
 * <br>
 * <br>Revision 1.10  2010-04-08 10:56:40  hermen
 * <br>enhanced logging
 * <br>
 * <br>Revision 1.9  2010-04-06 15:36:32  hermen
 * <br>added site to logentry
 * <br>
 * <br>Revision 1.8  2008-09-25 09:43:07  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.7  2007-11-20 08:58:53  hermen
 * <br>moved Managerfactory to core.utils and refactured code to use ManagerFactory instead of context.lookup
 * <br>
 * <br>Revision 1.6  2007/06/19 11:27:02  hermen
 * <br>added logging
 * <br>
 * <br>Revision 1.5  2007/03/02 08:28:41  hermen
 * <br>initial checkin after the merge of PatientModuleRebuild with the main HEAD
 * <br>
 * <br>Revision 1.4.2.1  2007/02/05 13:44:35  heinemann
 * <br>changed retry interval for unsend log messages from 2000 to 20000
 * <br>
 * <br>Revision 1.4  2006/11/15 08:48:23  hermen
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.3  2006/10/25 11:17:15  heinemann
 * <br>improved logging
 * <br>
 * <br>Revision 1.2  2006/10/24 12:47:13  heinemann
 * <br>The logger spools the msges in case of an interupted connection to the jboss
 * <br>
 * <br>Revision 1.1  2006/10/24 09:20:26  heinemann
 * <br>initial checkin
 * <br>
 */
public class GECAMedLog {

	
	private static final int RETRY_INTERVAL = 20000;
	
	/**
	 * static logger for this class
	 */
	private static Logger logger = Logger.getLogger(GECAMedLog.class.getName());
	
	private static LogManager manager;
	
	private static Stack<Log> logStack = new Stack<Log>();
	
	private static Timer logTimer = new Timer();
	
	private static LogWriterTimerTask timerTask;
	
	private static String host = "";
	
	
	
	
	static {
	        
		try {
			Enumeration<NetworkInterface> allInterfaces = NetworkInterface.getNetworkInterfaces();
			
			while(allInterfaces.hasMoreElements()) {
				/* ------------------------------------------------------ */
				NetworkInterface ni = allInterfaces.nextElement();
				Enumeration<InetAddress> iad = ni.getInetAddresses();
				while (iad.hasMoreElements()) {
					InetAddress addr = iad.nextElement();
					if (addr.getAddress().length == 4 && !addr.getCanonicalHostName().equals("localhost")) 
						host += addr.getHostAddress() + "/"+addr.getCanonicalHostName()+" ";
				}
				/* ------------------------------------------------------ */
			}
		} catch (Exception e) {
			logger.warn("Can not obtain local IP address and host name");
		}
	}
	
	
	
	/**
	 * Creates an error log message
	 * 
	 * @param module
	 * @param operation
	 * @param message
	 */
	public static void error(String module, String operation, String message, Throwable e , Long duration) {
		/* ====================================================== */
		ByteArrayOutputStream bout = new ByteArrayOutputStream();
		PrintWriter pw = new PrintWriter(bout);
		e.printStackTrace(pw);
		pw.flush();
	    	e.printStackTrace(pw);
	    	message += "\n" + bout.toString();
		createLogMessage(LogType.ERROR, module, operation, message, duration);
		/* ====================================================== */
	}
	
	/**
	 * Creates an error log message
	 * 
	 * @param module
	 * @param operation
	 * @param message
	 */
	public static void error(String module, String operation, String message, Throwable e ) {
		/* ====================================================== */
		error(module, operation, message, e, null);
		/* ====================================================== */
	}
	
	/**
	 * Creates an admin log message
	 * 
	 * @param module
	 * @param operation
	 * @param message
	 */
	public static void admin(String module, String operation, String message, Long duration) {
		/* ====================================================== */
		createLogMessage(LogType.ADMIN, module, operation, message, duration);
		/* ====================================================== */
	}	
	
	/**
	 * Creates an admin log message
	 * 
	 * @param module
	 * @param operation
	 * @param message
	 */
	public static void admin(String module, String operation, String message) {
		/* ====================================================== */
		createLogMessage(LogType.ADMIN, module, operation, message, null);
		/* ====================================================== */
	}
	
	/**
	 * Creates a system log message
	 * 
	 * @param module
	 * @param operation
	 * @param message
	 * @param duration
	 */
	public static void system(String module, String operation, String message, Long duration) {
		/* ====================================================== */
		createLogMessage(LogType.SYSTEM, module, operation, message, duration);
		/* ====================================================== */
	}
	
	/**
	 * Creates a system log message
	 * 
	 * @param module
	 * @param operation
	 * @param message
	 */
	public static void system(String module, String operation, String message) {
		/* ====================================================== */
		createLogMessage(LogType.SYSTEM, module, operation, message, null);
		/* ====================================================== */
	}
	
	/**
	 * Creates an user log message
	 * 
	 * @param module
	 * @param operation
	 * @param message
	 */
	public static void user(String module, String operation, String message, Long duration) {
		/* ====================================================== */
		createLogMessage(LogType.USER, module, operation, message, duration);
		/* ====================================================== */
	}
	
	/**
	 * Creates an user log message
	 * 
	 * @param module
	 * @param operation
	 * @param message
	 */
	public static void user(String module, String operation, String message) {
		/* ====================================================== */
		createLogMessage(LogType.USER, module, operation, message, null);
		/* ====================================================== */
	}
	
	
	/**
	 * Creates a log message 
	 * 
	 * @param logType
	 * @param module
	 * @param operation
	 * @param message
	 */
	public static void createLogMessage(Integer logType, String module, 
			String operation, String message, Long duration) {
		/* ====================================================== */

		// write a new log message
		Log log = new Log();
		log.setTypeId(logType);
		log.setModule(module);
		log.setOperation(operation);
		log.setText(message);
		log.setHost(host);
		log.setDuration(duration);
		
	    	try {
	    	    log.setSite(LoginScreen.getInstance().site.getName());	    
	    	} catch (Exception e) {}
		
		// push to the stack
		logStack.push(log);
		// start the timer
		startTimer(0);
		/* ====================================================== */
	}
	
	
	private static void startTimer(long time) {
		/* ====================================================== */
		/* ------------------------------------------------------ */
		try {
		logTimer.cancel();
		} catch (Exception e) {
		}
		logTimer = new Timer();
		// create a new task
		timerTask = new LogWriterTimerTask();
		logTimer.schedule(timerTask, time);
			/* ------------------------------------------------------ */
		/* ====================================================== */
	}
	
	
	/**
	 * Factory to obtain the log manager bean interface
	 * 
	 * @return
	 * @throws NamingException 
	 */
	private static LogManager getManager() throws NamingException {
		/* ====================================================== */
		if (manager == null) {
			manager = (LogManager) ManagerFactory.getRemote(LogManagerBean.class);
		}
		return manager;
		/* ====================================================== */
	}
	
	
	
	static class LogWriterTimerTask extends TimerTask {

		@Override
		public void run() {
			/* ============================================= */
			// try to write the logs to the server
			Log l = null;
			try {
				/* ------------------------------------------------------ */
//				System.out.println("Stack vor :  " + logStack.size());
				while(logStack.size() > 0) {
					/* ------------------------------------------------------ */
					// try to save the first log entry
//					System.out.println("Stack in :  " + logStack.size());
					l = logStack.pop();
//					System.out.println("Stack nach :  " + logStack.size());
					getManager().saveLog(l);
					// if we are here, everything is fine and
					/* ------------------------------------------------------ */
				}
				/* ------------------------------------------------------ */
			}catch (Exception e) {
				/* ------------------------------------------------------ */
				// if we are here, we must write the log message back to the stack
				// because the storing process returned with an exception
				logger.warn("Unable to write log messages to server! Connection Problem? " + e.getLocalizedMessage());
				logger.info("Queuing log messages");
				l.setText("Q:"  + l.getText());
				logStack.push(l);
				/* ------------------------------------------------------ */
			}
			// if the stack is not empty, we must repeat the task
			if (logStack.size() > 0)
				startTimer(RETRY_INTERVAL);
			/* ============================================= */
		}
		
	}
	
	public static boolean areAllLogsWritten() {
		return (logStack.size() == 0);
	}
	
	
	
}
