/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.cara.gui;

import java.awt.Color;
import java.awt.Font;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Locale;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.border.TitledBorder;

import lu.tudor.santec.gecamed.cara.ejb.entity.beans.Study;
import lu.tudor.santec.gecamed.cara.ejb.session.beans.CaraManagerBean;
import lu.tudor.santec.gecamed.cara.ejb.session.interfaces.CaraManagerInterface;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

public class DetailsPanel extends JPanel implements Relocalizable
	{
	private static final long serialVersionUID = 1L;
	
	private static Logger m_Logger = Logger.getLogger ("lu.tudor.santec.gecamed.cara.gui.DetailsPanel");

	private NumberFormat m_NumberFormatter  = NumberFormat.getInstance();
	
	private JLabel			m_ThumbnailCaption;
	
	private JLabel			m_OriginatingHospitalLabel;
	
	private JLabel			m_PrescribingPhysicianLabel;
	private JLabel			m_PrescriptionDateLabel;

	
	private JLabel			m_CaraLabel;
	private JLabel			m_HospitalLabel;
	private JLabel			m_CaraStudyIDLabel;
	private JLabel  		m_CaraStudyCodeLabel;
	private JLabel  		m_CaraStudyDescriptionLabel;
	
	private JLabel  		m_HospitalStudyIDLabel;
	private JLabel  		m_HospitalStudyCodeLabel;
	private JLabel  		m_HospitalStudyDescriptionLabel;
	
	private JLabel			m_ModalityLabel;
	private JLabel			m_IncidenceLabel;
	private JLabel			m_LateralityLabel;
	private JLabel			m_DateOfExamLabel;
	private JLabel			m_EventsLabel;
	private JLabel			m_ContrastLabel;
	private JLabel			m_VolumeLabel;
	private JLabel			m_IsotopeLabel;
	private JLabel			m_ActivityLabel;
	
	private JLabel			m_SummaryLabel;
	
	private TitledBorder	m_ThumbnailBorder;

	private	JLabel			m_Thumbnail;
	
	private TitledBorder	m_GeneralBorder;

	private JTextField		m_Hospital;
	
	private TitledBorder	m_PrescriptionBorder;
	
	private JTextField		m_PrescribingPhysician;
	private JTextField		m_PrescriptionDate;
	
	private TitledBorder	m_ExaminationBorder;
	
	private JTextField		m_CaraStudyID;
	private JTextField  	m_CaraStudyCode;
	private JTextField  	m_CaraStudyDescription;
	
	private JTextField  	m_HospitalStudyID;
	private JTextField  	m_HospitalStudyCode;
	private JTextField  	m_HospitalStudyDescription;
	
	private TitledBorder	m_ExaminationDetailsBorder;
	
	private JTextField		m_Modality;
	private JTextField		m_Incidence;
	private JTextField		m_Laterality;
	private JTextField		m_DateOfExam;
	private JTextField		m_StudyCode;
	private JTextField		m_Contrast;
	private JTextField		m_Volume;
	private JTextField		m_Isotope;
	private JTextField		m_Activity;
	
	private TitledBorder	m_ReportBorder;
	
	private JTextArea		m_Summary;

	private SimpleDateFormat	m_DateFormat;
	private SimpleDateFormat	m_DateTimeFormat;

	private CaraManagerInterface m_Manager;

	private PatientCaraPanel patientCaraPanel;

	private Study study;
	
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************
		
	private final static String c_Columns= 	"3dlu,fill:pref," +
											"3dlu,fill:pref:grow,3dlu";
		
	private final static String c_Rows=    	"3dlu,fill:pref," +
											"3dlu,fill:pref," +
											"3dlu,fill:pref," +
											"3dlu,fill:pref, 3dlu";

	private final static String c_ThumbnailColumns=		"3dlu,fill:300px,3dlu";
	
	private final static String c_ThumbnailRows=		"3dlu,fill:300px,3dlu,pref,3dlu";
	
	private final static String c_GeneralColumns=		"3dlu,fill:32px," +
														"3dlu,fill:max(50dlu;pref)," +
														"fill:pref:grow,3dlu";
	
	private final static String c_GeneralRows=    		"3dlu,fill:pref," +
														"3dlu,fill:pref,3dlu";
	
	private final static String c_PrescriptionColumns= 	"3dlu,fill:32px," +
														"3dlu,fill:max(100dlu;pref)," +
 														"3dlu,fill:max(100dlu;pref),3dlu";

	private final static String c_PrescriptionRows=    	"3dlu,fill:pref," +
														"3dlu,fill:pref,3dlu";

	private final static String c_ExaminationColumns= 	"3dlu,fill:25dlu," +
														"3dlu,fill:max(20dlu;pref)," +
														"3dlu,fill:160dlu:grow,3dlu";

	private final static String c_ExaminationRows=    	"3dlu,fill:pref," +
														"3dlu,fill:pref," +
														"3dlu,fill:pref," +
														"3dlu,fill:pref,3dlu";
	
	private final static String c_ExaminationDetailsColumns= 	"3dlu,fill:32px," +
																"3dlu,fill:max(75dlu;pref)," +
																"3dlu,fill:max(50dlu;pref)," +
																"3dlu,fill:max(50dlu;pref)," +
																"3dlu,fill:max(50dlu;pref),3dlu";

	private final static String c_ExaminationDetailsRows=    	"3dlu,fill:pref," +
																"3dlu,fill:pref," +
																"3dlu,fill:pref," +
																"3dlu,fill:pref," +
																"3dlu,fill:pref," +
																"3dlu,fill:pref,3dlu";

//---------------------------------------------------------------------------
//***************************************************************************
//* Constructor		                                                        *
//***************************************************************************
//---------------------------------------------------------------------------

public DetailsPanel(final PatientCaraPanel patientCaraPanel) 
	{
	CellConstraints		l_Constraints;
    FormLayout			l_Layout;
    Color				l_DefaultColor; 
     
    JLabel				l_GeneralIcon;
    JLabel				l_PrescriptionIcon;
    JLabel				l_ExaminationIcon;
    JLabel				l_ExaminationDetailsIcon;
     
    JPanel				l_ThumbnailPanel;
    JPanel				l_GeneralPanel;
    JPanel				l_PrescriptionPanel;
    JPanel				l_ExaminationPanel;
    JPanel				l_ExaminationDetailsPanel;
    JPanel				l_ReportPanel;
    
    this.patientCaraPanel = patientCaraPanel;
          
    this.setOpaque(false);

	m_ThumbnailCaption 				= new JLabel ();
	
	m_OriginatingHospitalLabel 		= new JLabel ();
	
	m_PrescribingPhysicianLabel		= new JLabel ();
	m_PrescriptionDateLabel			= new JLabel ();

	m_CaraLabel						= new JLabel ();	
	m_CaraStudyIDLabel				= new JLabel ();
	m_CaraStudyCodeLabel			= new JLabel ();
	m_CaraStudyDescriptionLabel		= new JLabel ();
	
	m_HospitalLabel					= new JLabel ();	
	m_HospitalStudyIDLabel			= new JLabel ();
	m_HospitalStudyCodeLabel		= new JLabel ();
	m_HospitalStudyDescriptionLabel	= new JLabel ();
	
	m_ModalityLabel					= new JLabel ();
	m_IncidenceLabel				= new JLabel ();
	m_LateralityLabel				= new JLabel ();
	m_DateOfExamLabel				= new JLabel ();
	m_EventsLabel					= new JLabel ();
	m_ContrastLabel					= new JLabel ();
	m_VolumeLabel					= new JLabel ();
	m_IsotopeLabel					= new JLabel ();
	m_ActivityLabel					= new JLabel ();
	
	m_SummaryLabel					= new JLabel ();;
	
	m_ThumbnailBorder				= new TitledBorder ("");
	m_GeneralBorder					= new TitledBorder ("");
	m_PrescriptionBorder			= new TitledBorder ("");
	m_ExaminationBorder				= new TitledBorder ("");
	m_ExaminationDetailsBorder		= new TitledBorder ("");
	m_ReportBorder					= new TitledBorder ("");

	this.relocalize();
	
	l_DefaultColor = (Color) UIManager.getDefaults().get("TextField.inactiveBackground");   
	UIManager.getDefaults().put("TextField.inactiveBackground", Color.WHITE );
	
	//------------------------------------------------------------------------
	// Build Thumbnail Panel
	//------------------------------------------------------------------------
	
	l_Constraints  = new CellConstraints();
    
	l_ThumbnailPanel       = new JPanel ();
	l_ThumbnailPanel.setBorder (m_ThumbnailBorder);
	    
	l_Layout = new FormLayout(c_ThumbnailColumns, c_ThumbnailRows);
	l_ThumbnailPanel.setLayout (l_Layout);
	l_ThumbnailPanel.setOpaque(false);

	m_Thumbnail = new JLabel ();
	m_Thumbnail.addMouseListener(new MouseAdapter() {
		@Override
		public void mouseClicked(MouseEvent e) {
			if (e.getClickCount() >= 2) {
				if (study != null && study.getDicomIDs() != null && ! study.getDicomIDs().startsWith("#"))
					patientCaraPanel.openImage(study.getDicomIDs());
			}
		}
	});
	
	
	
	m_ThumbnailCaption.setFont(new Font("Arial", Font.PLAIN, 10));
	
	l_ThumbnailPanel.add (m_Thumbnail, l_Constraints.xywh(2, 2, 1, 1));
	l_ThumbnailPanel.add (m_ThumbnailCaption, l_Constraints.xywh(2, 4, 1, 1));

	//------------------------------------------------------------------------
	// Build General Panel
	//------------------------------------------------------------------------
	
	l_Constraints  = new CellConstraints();
    
	l_GeneralPanel       = new JPanel ();
	l_GeneralPanel.setBorder (m_GeneralBorder);
	
	l_Layout = new FormLayout(c_GeneralColumns, c_GeneralRows);
	l_GeneralPanel.setLayout (l_Layout);
	l_GeneralPanel.setOpaque(false);
	
	l_GeneralIcon = new JLabel ();
	l_GeneralIcon.setIcon(PatientCaraPanel.getIcon("general.png"));
	
	m_Hospital = new JTextField ();
	m_Hospital.setEditable(false);

	l_GeneralPanel.add (l_GeneralIcon,   			l_Constraints.xywh(2, 2, 1, 1));
	l_GeneralPanel.add (m_OriginatingHospitalLabel, l_Constraints.xywh(4, 2, 1, 1));
	l_GeneralPanel.add (m_Hospital, 	 			l_Constraints.xywh(4, 4, 1, 1));

	//------------------------------------------------------------------------
	// Build Prescription Panel
	//------------------------------------------------------------------------
	
	l_Constraints  = new CellConstraints();
    
	l_PrescriptionPanel       = new JPanel ();
	l_PrescriptionPanel.setBorder (m_PrescriptionBorder);
	
	l_Layout = new FormLayout(c_PrescriptionColumns, c_PrescriptionRows);
	l_PrescriptionPanel.setLayout (l_Layout);
	l_PrescriptionPanel.setOpaque(false);
	
	l_PrescriptionIcon = new JLabel ();
	l_PrescriptionIcon.setIcon(PatientCaraPanel.getIcon("prescription.png"));

	m_PrescribingPhysician = new JTextField ();
	m_PrescribingPhysician.setEditable(false);
	
	m_PrescriptionDate 	   = new JTextField ();
	m_PrescriptionDate.setEditable(false);
	
	l_PrescriptionPanel.add (l_PrescriptionIcon, 			l_Constraints.xywh(2, 2, 1, 1));
	l_PrescriptionPanel.add (m_PrescribingPhysicianLabel, 	l_Constraints.xywh(4, 2, 1, 1));
	l_PrescriptionPanel.add (m_PrescribingPhysician, 	 	l_Constraints.xywh(4, 4, 1, 1));
	l_PrescriptionPanel.add (m_PrescriptionDateLabel, 		l_Constraints.xywh(6, 2, 1, 1));
	l_PrescriptionPanel.add (m_PrescriptionDate, 	 		l_Constraints.xywh(6, 4, 1, 1));
	
	//------------------------------------------------------------------------
	// Build Examination Panel
	//------------------------------------------------------------------------
	
	l_Constraints  = new CellConstraints();
    
	l_ExaminationPanel       = new JPanel ();
	l_ExaminationPanel.setBorder (m_ExaminationBorder);
	
	l_Layout = new FormLayout(c_ExaminationColumns, c_ExaminationRows);
	l_ExaminationPanel.setLayout (l_Layout);
	l_ExaminationPanel.setOpaque(false);
	
	l_ExaminationIcon = new JLabel ();
	l_ExaminationIcon.setIcon(PatientCaraPanel.getIcon("examination.png"));

	m_CaraStudyID = new JTextField ();
	m_CaraStudyID.setEditable(false);
	
	m_CaraStudyCode = new JTextField ();
	m_CaraStudyCode.setEditable(false);
	
	m_CaraStudyDescription = new JTextField ();
	m_CaraStudyDescription.setEditable(false);
		
	m_HospitalStudyID = new JTextField ();
	m_HospitalStudyID.setEditable(false);
	
	m_HospitalStudyCode = new JTextField ();
	m_HospitalStudyCode.setEditable(false);
	
	m_HospitalStudyDescription = new JTextField ();
	m_HospitalStudyDescription.setEditable(false);
	
	l_ExaminationPanel.add (l_ExaminationIcon, 				l_Constraints.xywh(2, 2, 1, 1));
	
	l_ExaminationPanel.add (m_CaraLabel, 					l_Constraints.xywh(2, 4, 1, 1));
//	l_ExaminationPanel.add (m_CaraStudyIDLabel, 			l_Constraints.xywh(4, 2, 1, 1));
//	l_ExaminationPanel.add (m_CaraStudyID, 	 				l_Constraints.xywh(4, 4, 1, 1));
	l_ExaminationPanel.add (m_CaraStudyCodeLabel, 			l_Constraints.xywh(4, 2, 1, 1));
	l_ExaminationPanel.add (m_CaraStudyCode, 	 			l_Constraints.xywh(4, 4, 1, 1));
	l_ExaminationPanel.add (m_CaraStudyDescriptionLabel, 	l_Constraints.xywh(6, 2, 1, 1));
	l_ExaminationPanel.add (m_CaraStudyDescription, 	 	l_Constraints.xywh(6, 4, 1, 1));
	
	l_ExaminationPanel.add (m_HospitalLabel, 					l_Constraints.xywh(2, 8, 1, 1));
//	l_ExaminationPanel.add (m_HospitalStudyIDLabel, 			l_Constraints.xywh(4, 6, 1, 1));
//	l_ExaminationPanel.add (m_HospitalStudyID, 	 				l_Constraints.xywh(4, 8, 1, 1));
	l_ExaminationPanel.add (m_HospitalStudyCodeLabel, 			l_Constraints.xywh(4, 6, 1, 1));
	l_ExaminationPanel.add (m_HospitalStudyCode, 	 			l_Constraints.xywh(4, 8, 1, 1));
	l_ExaminationPanel.add (m_HospitalStudyDescriptionLabel, 	l_Constraints.xywh(6, 6, 1, 1));
	l_ExaminationPanel.add (m_HospitalStudyDescription, 	 	l_Constraints.xywh(6, 8, 1, 1));

	//------------------------------------------------------------------------
	// Build Examination Details Panel
	//------------------------------------------------------------------------

	l_Constraints  = new CellConstraints();
    
	l_ExaminationDetailsPanel = new JPanel ();
	l_ExaminationDetailsPanel.setBorder (m_ExaminationDetailsBorder);
	
	l_Layout = new FormLayout(c_ExaminationDetailsColumns, c_ExaminationDetailsRows);
	l_ExaminationDetailsPanel.setLayout (l_Layout);
	l_ExaminationDetailsPanel.setOpaque(false);

	l_ExaminationDetailsIcon = new JLabel ();
	l_ExaminationDetailsIcon.setIcon(PatientCaraPanel.getIcon("examination_details.png"));

	m_DateOfExam = new JTextField ();
	m_DateOfExam.setEditable(false);
	
	m_Modality = new JTextField ();
	m_Modality.setEditable(false);

	m_Incidence = new JTextField ();
	m_Incidence.setEditable(false);

	m_Laterality = new JTextField ();
	m_Laterality.setEditable(false);

	m_Contrast = new JTextField ();
	m_Contrast.setEditable(false);

	m_Volume = new JTextField ();
	m_Volume.setEditable(false);
	
	m_Isotope = new JTextField ();
	m_Isotope.setEditable(false);

	m_Activity = new JTextField ();
	m_Activity.setEditable(false);
	
	m_StudyCode = new JTextField ();
	m_StudyCode.setEditable(false);
	
	l_ExaminationDetailsPanel.add (l_ExaminationDetailsIcon, l_Constraints.xywh(2, 2, 1, 1));
	
	l_ExaminationDetailsPanel.add (m_DateOfExamLabel, 		l_Constraints.xywh(4, 2, 1, 1));
	l_ExaminationDetailsPanel.add (m_DateOfExam, 	 		l_Constraints.xywh(4, 4, 1, 1));	
	l_ExaminationDetailsPanel.add (m_ModalityLabel, 		l_Constraints.xywh(6, 2, 1, 1));
	l_ExaminationDetailsPanel.add (m_Modality, 	 			l_Constraints.xywh(6, 4, 1, 1));
	l_ExaminationDetailsPanel.add (m_IncidenceLabel, 		l_Constraints.xywh(8, 2, 1, 1));
	l_ExaminationDetailsPanel.add (m_Incidence, 	 		l_Constraints.xywh(8, 4, 1, 1));
	l_ExaminationDetailsPanel.add (m_LateralityLabel, 		l_Constraints.xywh(10, 2, 1, 1));
	l_ExaminationDetailsPanel.add (m_Laterality, 	 		l_Constraints.xywh(10, 4, 1, 1));
	
	l_ExaminationDetailsPanel.add (m_ContrastLabel, 		l_Constraints.xywh(4, 6, 1, 1));
	l_ExaminationDetailsPanel.add (m_Contrast, 	 			l_Constraints.xywh(4, 8, 1, 1));
	l_ExaminationDetailsPanel.add (m_VolumeLabel, 			l_Constraints.xywh(6, 6, 1, 1));
	l_ExaminationDetailsPanel.add (m_Volume, 	 			l_Constraints.xywh(6, 8, 1, 1));
	l_ExaminationDetailsPanel.add (m_IsotopeLabel, 			l_Constraints.xywh(8, 6, 1, 1));
	l_ExaminationDetailsPanel.add (m_Isotope, 	 			l_Constraints.xywh(8, 8, 1, 1));
	l_ExaminationDetailsPanel.add (m_ActivityLabel, 		l_Constraints.xywh(10, 6, 1, 1));
	l_ExaminationDetailsPanel.add (m_Activity, 	 			l_Constraints.xywh(10, 8, 1, 1));
	
	l_ExaminationDetailsPanel.add (m_EventsLabel, 			l_Constraints.xywh(4, 10, 7, 1));
	l_ExaminationDetailsPanel.add (m_StudyCode, 				l_Constraints.xywh(4, 12, 7, 1));

	//------------------------------------------------------------------------
	// Build Main Panel
	//------------------------------------------------------------------------

	UIManager.getDefaults().put("TextField.inactiveBackground", l_DefaultColor );
	
	l_Layout = new FormLayout(c_Columns, c_Rows);
	this.setLayout (l_Layout);
	this.setOpaque(false);

	this.add (l_ThumbnailPanel, 			l_Constraints.xywh(2, 2, 1, 7));
	this.add (l_GeneralPanel, 	 			l_Constraints.xywh(4, 2, 1, 1));
	this.add (l_PrescriptionPanel, 			l_Constraints.xywh(4, 4, 1, 1));
	this.add (l_ExaminationPanel, 			l_Constraints.xywh(4, 6, 1, 1));
	this.add (l_ExaminationDetailsPanel, 	l_Constraints.xywh(4, 8, 1, 1));
	
	this.reset();
	
	m_Manager = (CaraManagerInterface) ManagerFactory.getRemote(CaraManagerBean.class);
	
	m_NumberFormatter.setMaximumFractionDigits(3);
	
	}
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body		                                                        *
//***************************************************************************
//---------------------------------------------------------------------------

public void setStudy(Study p_Study) 
	{
	String l_NotAvailable;
	
	this.study = p_Study;
	
	if (p_Study == null) 
		{
		this.reset ();
		return;
		}
	
	l_NotAvailable = Translatrix.getTranslationString("DetailsPanel.NotAvailable");
	
	if (p_Study.getHospital() != null)
		 m_Hospital.setText(p_Study.getHospital());
	else m_Hospital.setText(l_NotAvailable);
	
	if (p_Study.getPrescribingPhysician() != null)
		 m_PrescribingPhysician.setText(p_Study.getPrescribingPhysician());
	else m_PrescribingPhysician.setText(l_NotAvailable);
		
	if (p_Study.getPrescriptionDate() != null)
		 m_PrescriptionDate.setText(m_DateFormat.format(p_Study.getPrescriptionDate()));
	else m_PrescriptionDate.setText(l_NotAvailable);
	
	if (p_Study.getCaraID() != null)
		m_CaraStudyID.setText(p_Study.getCaraID());
	else m_CaraStudyID.setText(l_NotAvailable);
	
	if (p_Study.getCaraStudyCode() != null)
		m_CaraStudyCode.setText (p_Study.getCaraStudyCode());
	else m_CaraStudyCode.setText(l_NotAvailable);
			
	if (p_Study.getCaraDescription() != null)
		 m_CaraStudyDescription.setText (p_Study.getCaraDescription());
	else m_CaraStudyDescription.setText(l_NotAvailable);
	
	if (p_Study.getHospID() != null)
		m_HospitalStudyID.setText(p_Study.getHospID());
	else m_HospitalStudyID.setText(l_NotAvailable);
	
	if (p_Study.getHospStudyCode() != null)
		m_HospitalStudyCode.setText (p_Study.getHospStudyCode());
	else m_HospitalStudyCode.setText(l_NotAvailable);
			
	if (p_Study.getHospDescription() != null)
		 m_HospitalStudyDescription.setText (p_Study.getHospDescription().replaceAll("\\.", ","));
	else m_HospitalStudyDescription.setText(l_NotAvailable);
		
	if (p_Study.getDate() != null)
		 m_DateOfExam.setText (m_DateTimeFormat.format(p_Study.getDate()));
	else m_DateOfExam.setText (l_NotAvailable);

	if (p_Study.getModality() != null)
		m_Modality.setText (p_Study.getModality());
	else m_Modality.setText(l_NotAvailable);
	
	if (p_Study.getIncidence() != null)
		m_Incidence.setText (p_Study.getIncidence());
	else m_Incidence.setText(l_NotAvailable);
		
	if (p_Study.getLaterality() != null)
		m_Laterality.setText (p_Study.getLaterality());
	else m_Laterality.setText(l_NotAvailable);

	if (p_Study.getContrastProduct() != null)
		 m_Contrast.setText (p_Study.getContrastProduct());
	else m_Contrast.setText(l_NotAvailable);
	
	if (p_Study.getContrastProductVolume() != null)
		 m_Volume.setText (p_Study.getContrastProductVolume().toString());
	else m_Volume.setText(l_NotAvailable);

	if (p_Study.getIsotope() != null)
		m_Isotope.setText (p_Study.getIsotope());
	else m_Isotope.setText(l_NotAvailable);

	if (p_Study.getActivity() != null)
		 m_Activity.setText (p_Study.getActivity().toString());
	else m_Activity.setText(l_NotAvailable);

	if (p_Study.getDicomIDs() != null)
		m_StudyCode.setText (p_Study.getDicomIDs().replaceAll("#", ""));
	else m_StudyCode.setText(l_NotAvailable);
	
	
	try {
		byte[] l_Thumb = m_Manager.getThumbnail(p_Study.getDicomIDs());
		if (l_Thumb != null) {
			m_Thumbnail.setIcon(new ImageIcon(l_Thumb));
		} else {
			m_Thumbnail.setIcon(null);
		}
		Object[] l_Caption = m_Manager.getImageInfos(p_Study.getDicomIDs());
		if (l_Caption != null) {
			StringBuffer capString = new StringBuffer("<html><table>");
			String memorySize = m_NumberFormatter.format(Double.parseDouble((String) l_Caption[0])/1024/1024);
			capString.append("<tr><td><b>"+Translatrix.getTranslationString("DetailsPanel.FileSize")+":</b></td><td>" + memorySize + "mb</td></tr>");
			capString.append("<tr><td><b>"+Translatrix.getTranslationString("DetailsPanel.ImageRows")+":</b></td><td>" + l_Caption[1] + "</td></tr>");
			capString.append("<tr><td><b>"+Translatrix.getTranslationString("DetailsPanel.ImageColumns")+":</b></td><td>" + l_Caption[2] + "</td></tr>");
			m_ThumbnailCaption.setText(capString.toString());
		} else {
			m_ThumbnailCaption.setText("");
		}
		
	} catch (Exception e) {
		e.printStackTrace();
	}
	
	
	}	

//---------------------------------------------------------------------------

public void reset ()
	{
	m_Hospital.setText(null);
	m_PrescribingPhysician.setText(null);
	m_PrescriptionDate.setText(null);
	m_CaraStudyID.setText(null);
	m_CaraStudyCode.setText(null);
	m_CaraStudyDescription.setText(null);
	m_HospitalStudyID.setText(null);
	m_HospitalStudyCode.setText(null);
	m_HospitalStudyDescription.setText(null);
	m_DateOfExam.setText (null);
	m_Modality.setText(null);
	m_Incidence.setText(null);
	m_Laterality.setText(null);
	m_Contrast.setText(null);
	m_Volume.setText(null);
	m_Isotope.setText(null);
	m_Activity.setText(null);
	m_StudyCode.setText(null);
	m_ThumbnailCaption.setText(null);
	m_Thumbnail.setIcon(null);
	}

//---------------------------------------------------------------------------

public void relocalize() 
	{
  	Locale  l_Locale;
  	
	l_Locale = Translatrix.getLocale ();
	  	
	if (l_Locale != null) 
		{
		m_DateFormat = new SimpleDateFormat ("d MMMM yyyy",l_Locale);
		m_DateTimeFormat = new SimpleDateFormat ("d MMMM yyyy HH:MM",l_Locale);
		}
	else	
		{
		m_DateFormat = new SimpleDateFormat ("d MMMM yyyy");
		m_DateTimeFormat = new SimpleDateFormat ("d MMMM yyyy HH:MM");
		}
	
//	if (m_ThumbnailLabel != null)
//		m_ThumbnailLabel.setText(Translatrix.getTranslationString("DetailsPanel.ThumbnailLabel"));
	
	if (m_OriginatingHospitalLabel != null)
		m_OriginatingHospitalLabel.setText(Translatrix.getTranslationString("DetailsPanel.OriginatingHospitalLabel"));

	if (m_PrescribingPhysicianLabel != null)
		m_PrescribingPhysicianLabel.setText(Translatrix.getTranslationString("DetailsPanel.PrescribingPhysicianLabel"));

	if (m_PrescriptionDateLabel != null)
		m_PrescriptionDateLabel.setText(Translatrix.getTranslationString("DetailsPanel.PrescriptionDateLabel"));

	if (m_CaraLabel != null)
		m_CaraLabel.setText(Translatrix.getTranslationString("DetailsPanel.CARALabel"));

	if (m_CaraStudyIDLabel != null)
		m_CaraStudyIDLabel.setText(Translatrix.getTranslationString("DetailsPanel.CaraStudyIDLabel"));

	if (m_CaraStudyCodeLabel != null)
		m_CaraStudyCodeLabel.setText(Translatrix.getTranslationString("DetailsPanel.CaraStudyCodeLabel"));

	if (m_CaraStudyDescriptionLabel != null)
		m_CaraStudyDescriptionLabel.setText(Translatrix.getTranslationString("DetailsPanel.CaraStudyDescriptionLabel"));

	if (m_HospitalLabel != null)
		m_HospitalLabel.setText(Translatrix.getTranslationString("DetailsPanel.HospitalLabel"));

	if (m_HospitalStudyIDLabel != null)
		m_HospitalStudyIDLabel.setText(Translatrix.getTranslationString("DetailsPanel.HospitalStudyIDLabel"));

	if (m_HospitalStudyCodeLabel != null)
		m_HospitalStudyCodeLabel.setText(Translatrix.getTranslationString("DetailsPanel.HospitalStudyCodeLabel"));

	if (m_HospitalStudyDescriptionLabel != null)
		m_HospitalStudyDescriptionLabel.setText(Translatrix.getTranslationString("DetailsPanel.HospitalStudyDescriptionLabel"));

	if (m_ModalityLabel != null)
		m_ModalityLabel.setText(Translatrix.getTranslationString("DetailsPanel.ModalityLabel"));

	if (m_IncidenceLabel != null)
		m_IncidenceLabel.setText(Translatrix.getTranslationString("DetailsPanel.IncidenceLabel"));

	if (m_LateralityLabel != null)
		m_LateralityLabel.setText(Translatrix.getTranslationString("DetailsPanel.LateralityLabel"));

	if (m_DateOfExamLabel != null)
		m_DateOfExamLabel.setText(Translatrix.getTranslationString("DetailsPanel.DateOfExamLabel"));

	if (m_EventsLabel != null)
		m_EventsLabel.setText(Translatrix.getTranslationString("DetailsPanel.StudyID"));

	if (m_ContrastLabel != null)
		m_ContrastLabel.setText(Translatrix.getTranslationString("DetailsPanel.ContrastLabel"));

	if (m_VolumeLabel != null)
		m_VolumeLabel.setText(Translatrix.getTranslationString("DetailsPanel.VolumeLabel"));

	if (m_IsotopeLabel != null)
		m_IsotopeLabel.setText(Translatrix.getTranslationString("DetailsPanel.IsotopeLabel"));

	if (m_ActivityLabel != null)
		m_ActivityLabel.setText(Translatrix.getTranslationString("DetailsPanel.ActivityLabel"));

	if (m_SummaryLabel != null)
		m_SummaryLabel.setText(Translatrix.getTranslationString("DetailsPanel.SummaryLabel"));

	if (m_ThumbnailBorder != null)
		m_ThumbnailBorder.setTitle(Translatrix.getTranslationString("DetailsPanel.ThumbnailBorder"));
	
	if (m_GeneralBorder != null)
		m_GeneralBorder.setTitle(Translatrix.getTranslationString("DetailsPanel.GeneralBorder"));
	
	if (m_PrescriptionBorder != null)
		m_PrescriptionBorder.setTitle(Translatrix.getTranslationString("DetailsPanel.PrescriptionBorder"));
	
	if (m_ExaminationBorder != null)
		m_ExaminationBorder.setTitle(Translatrix.getTranslationString("DetailsPanel.ExaminationBorder"));
	
	if (m_ExaminationDetailsBorder != null)
		m_ExaminationDetailsBorder.setTitle(Translatrix.getTranslationString("DetailsPanel.ExaminationDetailsBorder"));
	
	if (m_ReportBorder != null)
		m_ReportBorder.setTitle(Translatrix.getTranslationString("DetailsPanel.ReportBorder"));
	}	
}
