/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.statistics;

import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.ScrollPaneConstants;
import javax.swing.UIManager;
import javax.swing.plaf.ColorUIResource;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Rate;
import lu.tudor.santec.gecamed.billing.gui.BillingModule;
import lu.tudor.santec.gecamed.billing.gui.rate.RateSearchDialog;
import lu.tudor.santec.gecamed.billing.gui.statement.PhysicianListModel;
import lu.tudor.santec.gecamed.billing.utils.BillingAdminSettings;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedLists;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.ComponentTitledPanel;
import lu.tudor.santec.gecamed.core.gui.utils.GECAMedGuiUtils;
import lu.tudor.santec.gecamed.core.gui.widgets.TimeEditorField;
import lu.tudor.santec.gecamed.core.gui.widgets.TimeFormatter;
import lu.tudor.santec.gecamed.core.utils.querybuilder.Group;
import lu.tudor.santec.gecamed.core.utils.querybuilder.HibernateCondition;
import lu.tudor.santec.gecamed.core.utils.querybuilder.HibernateOperator;
import lu.tudor.santec.gecamed.core.utils.querybuilder.WhereClause;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;
import com.toedter.calendar.JDateChooser;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

public class ActivityListDialog extends JDialog implements ItemListener,
														   ActionListener,
														   PropertyChangeListener,
														   Relocalizable
	{
	
	/**
	 * 
	 */
	
	private static final long serialVersionUID = 1L;
	
	private ComponentTitledPanel	m_PhysicianBorder;
	private JPanel					m_PhysicianPanel;
	private JCheckBox				m_SearchByPhysician;
	private PhysicianListModel		m_Physicians;
	private JList					m_PhysicianList;
	private JScrollPane				m_PhysicianScroller;

	private ComponentTitledPanel	m_RateBorder;
	private JPanel					m_RatePanel;
	private JCheckBox				m_SearchByRate;
	private JLabel					m_CodeLabel;
	private JTextField				m_Code;
	private JButton					m_BrowseButton;
	
	private ComponentTitledPanel	m_PerformedDateBorder;
	private JPanel					m_PerformedDatePanel;
	private JCheckBox				m_SearchByPerformedDate;
	private JLabel					m_PerformedFromDateLabel;
	private JDateChooser			m_PerformedFromDate;
	private JCheckBox				m_WithFromTime;
	private TimeEditorField			m_PerformedFromTime;
	
	private JLabel					m_PerformedUntilDateLabel;
	private JDateChooser			m_PerformedUntilDate;
	private JCheckBox				m_WithUntilTime;
	private TimeEditorField			m_PerformedUntilTime;

	private JButton					m_OkayButton;	
	private JButton					m_CancelButton;
	private RateSearchDialog		m_RateDialog;
	
	private WhereClause				m_WhereClause;
	private boolean					m_Validated;

//	private static Logger m_Logger = Logger.getLogger (ActivityListDialog.class.getName());
	
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************

	private final static String c_Columns				= 	"3dlu,fill:max(70dlu;pref)," + 
															"3dlu,fill:max(70dlu;pref),3dlu";
	
	private final static String c_Rows					=   "3dlu,fill:pref," +
															"3dlu,fill:pref," +
															"3dlu,fill:pref," +
															"3dlu,fill:pref,3dlu";

	private final static String c_PhysicianPanelColumns		=	"3dlu,fill:max(100dlu;pref),3dlu";
	
	private final static String c_PhysicianPanelRows		=	"3dlu,fill:max(70dlu;pref),3dlu";

	private final static String c_RatePanelColumns			=	"3dlu,fill:max(60dlu;pref)," + 
																"3dlu,fill:max(60dlu;pref),3dlu";
	
	private final static String c_RatePanelRows				=	"3dlu,fill:pref," +
																"3dlu,fill:pref,3dlu";

	private final static String c_PerformedDatePanelColumns	=	"3dlu,fill:max(80dlu;pref):grow," +
																"3dlu,fill:max(20dlu;pref),3dlu";

	private final static String c_PerformedDatePanelRows	=	"3dlu,fill:pref," +
																"3dlu,fill:pref," +
																"3dlu,fill:pref," +
																"3dlu,fill:pref,3dlu";

//---------------------------------------------------------------------------
//***************************************************************************
//* Constructor                                                             *
//***************************************************************************
//---------------------------------------------------------------------------
	
public ActivityListDialog ()	
	{	
	super (MainFrame.getInstance(),Translatrix.getTranslationString("ActivityListDialog.Title"),true);

	CellConstraints	l_Constraints;
	FormLayout		l_Layout;

	m_Validated = false;
	
	this.getContentPane().setBackground(GECAMedColors.c_GECAMedBackground);

//	this.setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);

	//------------------------------------------------------------------------
	// Build Physician Panel
	//------------------------------------------------------------------------

	this.buildPhysiciansPanel();
	this.buildRatePanel();
	this.buildPerformedDatePanel();
	
	l_Constraints  = new CellConstraints();
	l_Layout = new FormLayout(c_Columns, c_Rows);
	this.setLayout(l_Layout);
	
	this.add(m_PhysicianBorder,     	l_Constraints.xywh(2, 2, 3, 1));
	this.add(m_RateBorder,     			l_Constraints.xywh(2, 4, 3, 1));
	this.add(m_PerformedDateBorder,		l_Constraints.xywh(2, 6, 3, 1));

	m_OkayButton = new JButton (Translatrix.getTranslationString("core.ok"));
	m_OkayButton.setIcon(GECAMedModule.getSmallIcon(GECAMedIconNames.OK));
	m_OkayButton.addActionListener(this);

	m_CancelButton = new JButton (Translatrix.getTranslationString("core.cancel"));
	m_CancelButton.setIcon(GECAMedModule.getSmallIcon(GECAMedIconNames.CANCEL));
	m_CancelButton.addActionListener(this);

	this.add (m_OkayButton,  	l_Constraints.xywh(2, 8, 1, 1));
	this.add (m_CancelButton,   l_Constraints.xywh(4, 8, 1, 1));

	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Primitives                                                        *
//***************************************************************************
//---------------------------------------------------------------------------

private Date getFromDate ()
	{
	Date 				l_FromDate;
	Date 				l_UntilDate;
	GregorianCalendar	l_OriginalFromDate;
	GregorianCalendar	l_FromTime;
	
	l_FromDate  = m_PerformedFromDate.getDate();
	l_UntilDate = m_PerformedUntilDate.getDate();
	
	if ((l_FromDate != null) && (l_UntilDate != null))
		l_FromDate = BillingModule.getEarliestDate (l_FromDate, l_UntilDate);
	
	if (m_WithFromTime.isSelected() && (m_PerformedFromTime.getValue() instanceof Date))
		{
		l_OriginalFromDate = new GregorianCalendar();
		l_OriginalFromDate.setTime(l_FromDate);
		
		l_FromTime = new GregorianCalendar();
		l_FromTime.setTime((Date)m_PerformedFromTime.getValue());
		
		l_OriginalFromDate.set(Calendar.HOUR_OF_DAY,l_FromTime.get(Calendar.HOUR_OF_DAY));
		l_OriginalFromDate.set(Calendar.MINUTE,l_FromTime.get(Calendar.MINUTE));
		l_OriginalFromDate.set(Calendar.SECOND,0);
		l_OriginalFromDate.set(Calendar.MILLISECOND,0);
		
		l_FromDate = l_OriginalFromDate.getTime();
		}
	else
		{
		m_WithFromTime.setSelected(false);
		l_FromDate = BillingModule.earliestOfDay (l_FromDate);
		}
	
	return l_FromDate;
	}

//---------------------------------------------------------------------------

private Date getUntilDate ()
	{
	Date 				l_FromDate;
	Date 				l_UntilDate;
	GregorianCalendar	l_OriginalUntilDate;
	GregorianCalendar	l_UntilTime;
	
	l_FromDate  = m_PerformedFromDate.getDate();
	l_UntilDate = m_PerformedUntilDate.getDate();
	
	if ((l_FromDate != null) && (l_UntilDate != null))
		l_UntilDate = BillingModule.getLatestDate (l_FromDate, l_UntilDate);

	if (m_WithUntilTime.isSelected() && (m_PerformedUntilTime.getValue() instanceof Date))
		{
		l_OriginalUntilDate = new GregorianCalendar();
		l_OriginalUntilDate.setTime(l_UntilDate);
		
		l_UntilTime = new GregorianCalendar();
		l_UntilTime.setTime((Date)m_PerformedUntilTime.getValue());
		
		l_OriginalUntilDate.set(Calendar.HOUR_OF_DAY,l_UntilTime.get(Calendar.HOUR_OF_DAY));
		l_OriginalUntilDate.set(Calendar.MINUTE,l_UntilTime.get(Calendar.MINUTE));
		l_OriginalUntilDate.set(Calendar.SECOND,59);
		l_OriginalUntilDate.set(Calendar.MILLISECOND,999);
		
		l_UntilDate = l_OriginalUntilDate.getTime();
		}
	else
		{
		m_WithUntilTime.setSelected(false);
		l_UntilDate = BillingModule.latestOfDay (l_UntilDate);
		}
	
	return l_UntilDate;
	}

//---------------------------------------------------------------------------

private void buildPhysiciansPanel ()
	{
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;
	
	l_Constraints  = new CellConstraints();
	
	m_PhysicianPanel = new JPanel ();

	l_Layout = new FormLayout(c_PhysicianPanelColumns, c_PhysicianPanelRows);
	m_PhysicianPanel.setLayout (l_Layout);
	m_PhysicianPanel.setOpaque(true);
	
	m_SearchByPhysician = new JCheckBox (Translatrix.getTranslationString("ActivityListDialog.PhysiciansSwitch"));
    m_SearchByPhysician.setBackground(GECAMedColors.c_GECAMedBackground);
    m_SearchByPhysician.setSelected(false);
    m_SearchByPhysician.addItemListener (this);
	
	m_Physicians = new PhysicianListModel ();
	
	m_PhysicianList = new JList (m_Physicians);
	m_PhysicianList.setSelectionMode (ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
	m_PhysicianList.setLayoutOrientation(JList.VERTICAL);
	m_PhysicianList.setVisibleRowCount(-1);
	
	m_PhysicianScroller = new JScrollPane (m_PhysicianList);
	m_PhysicianScroller.setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
	
	m_PhysicianPanel.add (m_PhysicianScroller, l_Constraints.xywh(2, 2, 1, 1));

	m_PhysicianBorder = new ComponentTitledPanel (m_SearchByPhysician,m_PhysicianPanel);
    m_PhysicianBorder.setOpaque(false);	

	m_Physicians.setPhysicians(GECAMedLists.getListCopy(Physician.class));
	
	this.enablePhysicianPanel(false);
	}

//---------------------------------------------------------------------------

private void buildRatePanel ()
	{
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;
	ImageIcon		l_BrowseActIcon;
	
	l_Constraints  = new CellConstraints();
	
	m_RatePanel = new JPanel ();

	l_Layout = new FormLayout(c_RatePanelColumns, c_RatePanelRows);
	m_RatePanel.setLayout (l_Layout);
	m_RatePanel.setOpaque(true);
	
	m_SearchByRate = new JCheckBox (Translatrix.getTranslationString("ActivityListDialog.RateSwitch"));
    m_SearchByRate.setBackground(GECAMedColors.c_GECAMedBackground);
    m_SearchByRate.setSelected(false);
    m_SearchByRate.addItemListener (this);
	
	m_CodeLabel = new JLabel (Translatrix.getTranslationString("ActivityListDialog.CodeLabel"));
 	m_CodeLabel.setForeground(GECAMedColors.c_LabelColor);
 	m_Code = new JTextField ();
 	
 	m_BrowseButton = new JButton (Translatrix.getTranslationString("ActivityListDialog.BrowseButton"));
 	
 	l_BrowseActIcon = BillingModule.getScaledIcon("browse_act.png",18);
  	m_BrowseButton.setIcon(l_BrowseActIcon);
 	m_BrowseButton.addActionListener(this);
  	
 	m_RatePanel.add (m_CodeLabel, 		l_Constraints.xywh(2, 2, 1, 1));
	m_RatePanel.add (m_Code, 	   		l_Constraints.xywh(2, 4, 1, 1));
	m_RatePanel.add (m_BrowseButton, 	l_Constraints.xywh(4, 4, 1, 1));

	m_RateBorder = new ComponentTitledPanel (m_SearchByRate,m_RatePanel);
    m_RateBorder.setOpaque(false);	
	
    this.enableRatePanel(false);
	}

//---------------------------------------------------------------------------

private void buildPerformedDatePanel ()
	{
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;
	Color			l_DefaultColor;
	
	l_Constraints  = new CellConstraints();
	
	m_PerformedDatePanel = new JPanel ();
	
	l_Layout = new FormLayout(c_PerformedDatePanelColumns, c_PerformedDatePanelRows);
	m_PerformedDatePanel.setLayout (l_Layout);
	m_PerformedDatePanel.setOpaque(true);
	
	m_SearchByPerformedDate = new JCheckBox (Translatrix.getTranslationString("ActivityListDialog.PerformedDateSwitch"));
    m_SearchByPerformedDate.setBackground(GECAMedColors.c_GECAMedBackground);
    m_SearchByPerformedDate.setSelected(false);
    m_SearchByPerformedDate.addItemListener (this);
	
	m_PerformedFromDateLabel = new JLabel (Translatrix.getTranslationString("ActivityListDialog.PerformedFromDateLabel"));
 	m_PerformedFromDateLabel.setForeground(GECAMedColors.c_LabelColor);
 	m_PerformedFromDate = GECAMedGuiUtils.getDateChooser(false);
 	m_PerformedFromDate.setDate(new Date());
	m_PerformedFromDate.addPropertyChangeListener(this);
 	
	m_WithFromTime = new JCheckBox (Translatrix.getTranslationString ("ActivityListDialog.WithTime"));
    m_WithFromTime.setForeground(GECAMedColors.c_LabelColor);
	m_WithFromTime.setBackground(GECAMedColors.c_GECAMedBackground);
    m_WithFromTime.setSelected(false);
	m_WithFromTime.addItemListener (this);
    
    l_DefaultColor = (Color) UIManager.getDefaults().get("TextField.inactiveBackground");   
    UIManager.put("FormattedTextField.inactiveForeground", new ColorUIResource(Color.LIGHT_GRAY));
      
    m_PerformedFromTime = new TimeEditorField (new TimeFormatter());
	
	m_PerformedUntilDateLabel = new JLabel (Translatrix.getTranslationString("ActivityListDialog.PerformedUntilDateLabel"));
  	m_PerformedUntilDateLabel.setForeground(GECAMedColors.c_LabelColor);
	m_PerformedUntilDate = GECAMedGuiUtils.getDateChooser(false);
	m_PerformedUntilDate.setDate(new Date());
 	m_PerformedUntilDate.addPropertyChangeListener(this);
 
 	m_WithUntilTime = new JCheckBox (Translatrix.getTranslationString ("ActivityListDialog.WithTime"));
    m_WithUntilTime.setForeground(GECAMedColors.c_LabelColor);
    m_WithUntilTime.setBackground(GECAMedColors.c_GECAMedBackground);
    m_WithUntilTime.setSelected(false);
	m_WithUntilTime.addItemListener (this);
	
    m_PerformedUntilTime = new TimeEditorField (new TimeFormatter());

    UIManager.put("FormattedTextField.inactiveForeground", l_DefaultColor);

    m_PerformedDatePanel.add (m_PerformedFromDateLabel, 	l_Constraints.xywh(2, 2, 1, 1));
	m_PerformedDatePanel.add (m_PerformedFromDate, 			l_Constraints.xywh(2, 4, 1, 1));
	m_PerformedDatePanel.add (m_WithFromTime, 				l_Constraints.xywh(4, 2, 1, 1));
	m_PerformedDatePanel.add (m_PerformedFromTime, 			l_Constraints.xywh(4, 4, 1, 1));
		
	m_PerformedDatePanel.add (m_PerformedUntilDateLabel, 	l_Constraints.xywh(2, 6, 1, 1));
	m_PerformedDatePanel.add (m_PerformedUntilDate, 		l_Constraints.xywh(2, 8, 1, 1));
 	m_PerformedDatePanel.add (m_WithUntilTime, 				l_Constraints.xywh(4, 6, 1, 1));
	m_PerformedDatePanel.add (m_PerformedUntilTime, 		l_Constraints.xywh(4, 8, 1, 1));

    m_PerformedDateBorder = new ComponentTitledPanel (m_SearchByPerformedDate,m_PerformedDatePanel);
    m_PerformedDateBorder.setOpaque(false);	
	
	this.enablePerformedDatePanel (false);
	}

//---------------------------------------------------------------------------	

private void setBackground (JComponent p_Component, boolean p_EnableIt)
	{
	if (p_EnableIt)
		 p_Component.setBackground (GECAMedColors.c_GECAMedDarkerBackground);
	else p_Component.setBackground (GECAMedColors.c_GECAMedBackground);
	}

//---------------------------------------------------------------------------

private boolean showOnlyInvoicesOfCurrentPhysician ()
	{
	boolean l_ShowOnlyCurrentPhysician;
	
	l_ShowOnlyCurrentPhysician = 
	
	!BillingModule.userHasPermission(MainFrame.PERMISSIONS_CORE, MainFrame.PERMISSION_CHANGE_PHYSICIAN)
	
	&&
		
	(Boolean)BillingModule.getSetting(BillingAdminSettings.c_SplitBillingSetting);
	
	return l_ShowOnlyCurrentPhysician;
	}

//===========================================================================
//= Panel Enabling
//===========================================================================
//---------------------------------------------------------------------------	

private void enablePhysicianPanel (boolean p_EnableIt)
	{
	this.setBackground (m_PhysicianPanel,p_EnableIt);
	m_PhysicianList.setEnabled(p_EnableIt);
	m_PhysicianScroller.getVerticalScrollBar().setEnabled(p_EnableIt);	
	
	if (!p_EnableIt)
		{
		m_PhysicianList.clearSelection();
		}
	else
		{
		m_PhysicianList.setSelectedValue(GECAMedModule.getCurrentPhysician(), true);
		}
	}

//---------------------------------------------------------------------------	

private void enableRatePanel (boolean p_EnableIt)
	{
	this.setBackground (m_RatePanel,p_EnableIt);
	m_CodeLabel.setEnabled(p_EnableIt);
	m_Code.setEnabled(p_EnableIt);
	m_BrowseButton.setEnabled(p_EnableIt);
	}

//---------------------------------------------------------------------------	

private void enablePerformedDatePanel (boolean p_EnableIt)
	{
	Date l_Today;
	
	this.setBackground (m_PerformedDatePanel,p_EnableIt);
	this.setBackground (m_WithFromTime,p_EnableIt);
	this.setBackground (m_WithUntilTime,p_EnableIt);

	m_PerformedFromDateLabel.setEnabled(p_EnableIt);
	m_PerformedUntilDateLabel.setEnabled(p_EnableIt);	
	
	m_PerformedFromDate.setEnabled(p_EnableIt);
	m_WithFromTime.setEnabled (p_EnableIt);
	
	m_PerformedUntilDate.setEnabled(p_EnableIt);
	m_WithUntilTime.setEnabled (p_EnableIt);
	
	if (p_EnableIt == false)
		{
		m_PerformedFromDate.setDate(null);
		m_PerformedFromTime.setEnabled(p_EnableIt);
		m_PerformedFromTime.reset ();
		m_PerformedUntilDate.setDate(null);
		m_PerformedUntilTime.setEnabled (p_EnableIt);
		m_PerformedUntilTime.reset ();
		}
	else
		{
		l_Today = new Date ();
		m_PerformedFromDate.setDate(l_Today);
		m_PerformedUntilDate.setDate(null);
		}
	}

//---------------------------------------------------------------------------	

private void browseForRate ()
	{
	RateSearchDialog	l_RateDialog;
	Rate				l_ChosenRate;
	
	
	l_RateDialog = getRateSearchDialog();
	l_RateDialog.pack();
	l_RateDialog.setVisible(true);
	l_ChosenRate = l_RateDialog.getChosenRate();
	
	if (l_ChosenRate != null)
		{
		m_Code.setText(l_ChosenRate.getCode());
		}
	}


private RateSearchDialog getRateSearchDialog ()
	{
	if (m_RateDialog == null)
		m_RateDialog = new RateSearchDialog (ListSelectionModel.SINGLE_SELECTION);
	
	return m_RateDialog;
	}

//---------------------------------------------------------------------------	
//===========================================================================
//= Search Expressions
//===========================================================================
//---------------------------------------------------------------------------	

private Group getPhysicianSearchExpression ()
	{
	int[]				l_SelectedIndices;
	Physician			l_Physician;
	Group				l_SearchExpression 	= null;
	HibernateCondition	l_Condition 		= null;
	int					l_Index;
	
	if (m_SearchByPhysician.isSelected())
		{
		l_SearchExpression = new Group (HibernateOperator.c_OrOperator);
		
		l_SelectedIndices = m_PhysicianList.getSelectedIndices();
		if (l_SelectedIndices != null)
			{
			for (l_Index = 0; l_Index < l_SelectedIndices.length; l_Index++)
				{
				l_Physician = m_Physicians.getElementAt(l_SelectedIndices[l_Index]);
				
				l_Condition = new HibernateCondition ("physicianId",
													  HibernateOperator.c_EqualOperator,
													  l_Physician.getId());
					
				l_SearchExpression.addCondition (l_Condition);
				}			
			}
		}
	
	return l_SearchExpression;
	}

//---------------------------------------------------------------------------	

private Group getRateSearchExpression ()
	{
	Group				l_SearchExpression 	= null;
	HibernateCondition	l_Condition 		= null;
	
	if (m_SearchByRate.isSelected())
		{
		l_SearchExpression = new Group (HibernateOperator.c_OrOperator);
		l_Condition = new HibernateCondition ("code",
											   HibernateOperator.c_LikeOperator,
											   m_Code.getText() + "%");
		l_Condition.setCaseSensitive(false);
		l_SearchExpression.addCondition (l_Condition);
		}
	
	return l_SearchExpression;
	}

//---------------------------------------------------------------------------	

private Group getPerformedDateSearchExpression ()
	{
	Group				l_SearchExpression 	= null;
	HibernateCondition	l_Condition 		= null;
	
	if (m_SearchByPerformedDate.isSelected())
		{
		l_SearchExpression = new Group (HibernateOperator.c_AndOperator);
		
		if (m_PerformedFromDate.getDate() != null)
			{
			l_Condition = new HibernateCondition ("performedDate",
												  HibernateOperator.c_GreaterOrEqualOperator, 
												  this.getFromDate());
			
			l_Condition.setParameter("PerformedFromDate");
			}
		
		l_SearchExpression.addCondition (l_Condition);
		
		if (m_PerformedUntilDate.getDate() != null)
			{			
			l_Condition = new HibernateCondition ("performedDate",
												  HibernateOperator.c_LowerOrEqualOperator,
												  this.getUntilDate());
			
			l_Condition.setParameter("PerformedUntilDate");
			}
		
		l_SearchExpression.addCondition (l_Condition);
		}
	
	return l_SearchExpression;
	}

//---------------------------------------------------------------------------	

private Group getSplitBillingDiscriminator ()
	{
	Group				l_SearchDiscriminator   = null;
	HibernateCondition	l_Condition 			= null;
	Physician			l_CurrentPhysician  	= null;

	l_CurrentPhysician = BillingModule.getCurrentPhysician();
	if ((l_CurrentPhysician == null) || (!l_CurrentPhysician.isPersistent())) return null;
	
	l_SearchDiscriminator = new Group (HibernateOperator.c_AndOperator);
	
	l_Condition = new HibernateCondition ("physicianId",
										   HibernateOperator.c_EqualOperator,
										   l_CurrentPhysician.getId());
	
	l_SearchDiscriminator.addCondition(l_Condition);
	
	return l_SearchDiscriminator;
	}

//---------------------------------------------------------------------------	

private void assembleWhereClause ()
	{
	Group		l_SearchExpression;

	m_WhereClause = new WhereClause();
	m_WhereClause.setOperator(HibernateOperator.c_AndOperator);
		
	l_SearchExpression = this.getPhysicianSearchExpression();
	m_WhereClause.addGroup(l_SearchExpression);
	
	l_SearchExpression = this.getPerformedDateSearchExpression();
	m_WhereClause.addGroup(l_SearchExpression);
	
	l_SearchExpression = this.getRateSearchExpression();
	m_WhereClause.addGroup(l_SearchExpression);
	
	if (showOnlyInvoicesOfCurrentPhysician ())
		{
		l_SearchExpression = this.getSplitBillingDiscriminator();
		m_WhereClause.addGroup(l_SearchExpression);
		}
	}
//---------------------------------------------------------------------------	

public void itemStateChanged(ItemEvent p_ItemEvent) 
	{
	boolean l_EnableIt;
	
	l_EnableIt = (p_ItemEvent.getStateChange() == ItemEvent.SELECTED)?true:false; 
	
	if (p_ItemEvent.getSource().equals (m_SearchByPhysician))
		{
		this.enablePhysicianPanel(l_EnableIt);
		}
	else if (p_ItemEvent.getSource().equals (m_SearchByRate))
		{
		this.enableRatePanel(l_EnableIt);
		}
	else if (p_ItemEvent.getSource().equals (m_SearchByPerformedDate))
		{
		this.enablePerformedDatePanel(l_EnableIt);
		}
	else if (p_ItemEvent.getSource().equals (m_WithFromTime))
		{
		m_PerformedFromTime.setEnabled(l_EnableIt);
		}
	else if (p_ItemEvent.getSource().equals (m_WithUntilTime))
		{
		m_PerformedUntilTime.setEnabled(l_EnableIt);
		}
	}

//---------------------------------------------------------------------------	

public void setVisible (boolean p_ShowIt)
	{
	if ((p_ShowIt) && (showOnlyInvoicesOfCurrentPhysician()))
		{	
		this.enablePhysicianPanel(false);
		m_SearchByPhysician.setSelected(false);
		m_SearchByPhysician.setEnabled(false);
		m_PhysicianList.setSelectedValue(BillingModule.getCurrentPhysician(), true);
		}
	super.setVisible(p_ShowIt);
	}

//---------------------------------------------------------------------------	

public void propertyChange(PropertyChangeEvent p_Event) 
	{
	Date				l_StartDate;
	Date				l_EndDate;
	
	if (   (p_Event.getSource().equals(m_PerformedFromDate))
		&& ("date".equals(p_Event.getPropertyName()))) 
		{
		l_StartDate = this.getFromDate();
		l_EndDate   = this.getUntilDate();
		
		if (l_StartDate == null) return;
		
		if (l_EndDate == null) l_EndDate = BillingModule.lastOfMonth(l_StartDate);
		
		m_PerformedFromDate.removePropertyChangeListener(this);
		m_PerformedUntilDate.removePropertyChangeListener(this);
		
		m_PerformedFromDate.setDate (l_StartDate);
		m_PerformedUntilDate.setDate (l_EndDate);
		
		m_PerformedFromDate.addPropertyChangeListener(this);
		m_PerformedUntilDate.addPropertyChangeListener(this);

		this.repaint();
		}
	
	else if (   (p_Event.getSource().equals(m_PerformedUntilDate))
		&& ("date".equals(p_Event.getPropertyName()))) 
		{
		l_StartDate = this.getFromDate();
		l_EndDate   = this.getUntilDate();
		
		if (l_EndDate == null) return;
		
		if (l_StartDate == null) l_StartDate = BillingModule.firstOfMonth(l_EndDate);
		
		m_PerformedFromDate.removePropertyChangeListener(this);
		m_PerformedUntilDate.removePropertyChangeListener(this);
		
		m_PerformedFromDate.setDate (l_StartDate);
		m_PerformedUntilDate.setDate (l_EndDate);
		
		m_PerformedFromDate.addPropertyChangeListener(this);
		m_PerformedUntilDate.addPropertyChangeListener(this);

		this.repaint();
		}
	}

//---------------------------------------------------------------------------	

public void actionPerformed(ActionEvent p_Event) 
	{
	if (p_Event.getSource().equals(m_BrowseButton))
		{
		this.browseForRate ();
		}	
	else if (p_Event.getSource().equals(m_OkayButton))
		{
		m_Validated = true;
		this.assembleWhereClause();
		this.setVisible(false);
		}	
	else if (p_Event.getSource().equals(m_CancelButton))
		{
		m_Validated = false;
		this.setVisible(false);
		}	
	}

//---------------------------------------------------------------------------	

public boolean wasValidated ()
	{
	return m_Validated;
	}

//---------------------------------------------------------------------------	

public WhereClause getWhereClause ()
	{
	return m_WhereClause;
	}

//---------------------------------------------------------------------------	

public Date getPerformedFromDate ()
	{
	return this.getFromDate();
	}

//---------------------------------------------------------------------------	

public Date getPerformedUntilDate ()
	{
	return this.getUntilDate();
	}

//---------------------------------------------------------------------------	

public void relocalize() 
	{
	if (m_SearchByPhysician != null)	
		m_SearchByPhysician.setText (Translatrix.getTranslationString("ActivityListDialog.PhysiciansSwitch"));

	if (m_SearchByRate != null)	
		m_SearchByRate.setText (Translatrix.getTranslationString("ActivityListDialog.RateSwitch"));

	if (m_CodeLabel != null)		
		m_CodeLabel.setText (Translatrix.getTranslationString("ActivityListDialog.CodeLabel"));
 
	if (m_BrowseButton != null)		
		m_BrowseButton.setText (Translatrix.getTranslationString("ActivityListDialog.BrowseButton"));
 
	if (m_SearchByPerformedDate != null)	
		m_SearchByPerformedDate.setText (Translatrix.getTranslationString("ActivityListDialog.PerformedDateSwitch"));
	
	if (m_PerformedFromDateLabel != null)	
		m_PerformedFromDateLabel.setText (Translatrix.getTranslationString("ActivityListDialog.PerformedFromDateLabel"));
	
//	if (m_PerformedFromDate != null) 
//		{
//		m_PerformedFromDate.setLocale (Translatrix.getLocale());
//		m_PerformedFromDate.setDateFormatString("d MMMM yyyy");
//		m_PerformedFromDate.getDateEditor().setLocale (Translatrix.getLocale());
//		}
	
	if (m_PerformedUntilDateLabel != null)	
		m_PerformedUntilDateLabel.setText (Translatrix.getTranslationString("ActivityListDialog.PerformedUntilDateLabel"));
	
//	if (m_PerformedUntilDate != null) 
//		{
//		m_PerformedUntilDate.setLocale (Translatrix.getLocale());
//		m_PerformedUntilDate.setDateFormatString("d MMMM yyyy");   
//		m_PerformedUntilDate.getDateEditor().setLocale (Translatrix.getLocale());
//		}

	}

//---------------------------------------------------------------------------	
//***************************************************************************
//* End of Class				                            				*
//***************************************************************************
//---------------------------------------------------------------------------	
}

