/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
/*
 * InvoiceListModel.java
 *
 * Created on March 31, 2006
 */

package lu.tudor.santec.gecamed.billing.gui.statement;

/**
 *
 * @author  santec
 */

import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.table.AbstractTableModel;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Statement;
import lu.tudor.santec.gecamed.billing.utils.StatementWorkflow;
import lu.tudor.santec.i18n.Translatrix;

//***************************************************************************
//* Class Members                                                           *
//***************************************************************************

public class StatementListModel extends AbstractTableModel
    {
 	private static final long serialVersionUID = 1L;
 	
    private Vector      	<Statement> 		m_Statements;        
    private Hashtable   	<String, Integer> 	m_StatementLookUp;
    
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************

    public static final Date c_Epoch = new Date (0);

    public static final String c_StateHeader 			= "StatementListModel.StateHeader";
    public static final String c_ThirdPartyPayerHeader 	= "StatementListModel.ThirdPartyPayerHeader";
    public static final String c_PhysicianHeader 		= "StatementListModel.PhysicianHeader";
    public static final String c_ReferenceHeader		= "StatementListModel.ReferenceHeader";
    public static final String c_StartDateHeader		= "StatementListModel.StartDateHeader";
    public static final String c_EndDateHeader		 	= "StatementListModel.EndDateHeader";
           
    public static final int		c_StateColumn			= 0;
    public static final int		c_ThirdPartyPayerColumn	= 1;
    public static final int		c_PhysicianColumn		= 2;
    public static final int		c_ReferenceColumn		= 3;
    public static final int		c_StartDateColumn		= 4;
    public static final int		c_EndDateColumn			= 5;
    
    static final public String c_TableHeaders [] = { 	c_StateHeader,
    													c_ThirdPartyPayerHeader,
    													c_PhysicianHeader,
    													c_ReferenceHeader,
    													c_StartDateHeader,
    													c_EndDateHeader };
 
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

public StatementListModel ()
    {
	m_Statements	    = new Vector <Statement> ();
	m_StatementLookUp  	= new Hashtable <String, Integer> (); 
    }

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body		                                                        *
//***************************************************************************
//---------------------------------------------------------------------------

public String getColumnName (int p_Column)
    {
    if ((p_Column >=0) && (p_Column < c_TableHeaders.length))
    		{
    		return (Translatrix.getTranslationString(c_TableHeaders[p_Column]));
    		}
    else return ("");
    }

//---------------------------------------------------------------------------

public int getColumnCount ()
    {
    return (c_TableHeaders.length);
    }

//---------------------------------------------------------------------------

public int getRowCount ()
    {
    return (m_Statements.size());
    }

//---------------------------------------------------------------------------

public Object getValueAt (int p_Row, int p_Column)
    {
    Statement  l_Statement;
    
    l_Statement = new Statement ();
    
    if ((p_Row >= 0) && (p_Row < m_Statements.size()))
        {
    	l_Statement = m_Statements.elementAt (p_Row);
    	if (l_Statement == null) l_Statement = new Statement ();
        }
	 
    switch (p_Column)
        {
        case  c_StateColumn: 			return (l_Statement.getState() != null) ? l_Statement.getState() : Integer.valueOf (StatementWorkflow.c_NewState);
        case  c_ThirdPartyPayerColumn: 	return l_Statement.formatThirdPartyPayers();
        case  c_PhysicianColumn: 		return l_Statement.formatPhysicians();
        case  c_ReferenceColumn: 		return ((l_Statement.getReference() != null && l_Statement.getReference().length() > 0) ? l_Statement.getReference() : l_Statement.formatStatementNumber());    
        case  c_StartDateColumn: 		return (l_Statement.getStartDate() != null) ? l_Statement.getStartDate() : c_Epoch;
        case  c_EndDateColumn: 			return (l_Statement.getEndDate() != null)   ? l_Statement.getEndDate()   : c_Epoch;
               
        default: return "";
        }
    }
        
//---------------------------------------------------------------------------

public Class getColumnClass (int p_Column) 
    {
    if (getValueAt(0, p_Column) != null)
    		 return (Class) getValueAt(0, p_Column).getClass();
    else return null;
    }

//---------------------------------------------------------------------------
/**
 * queries the ability to be edited of the cell specified by p_Row and p_Column
 * @param p_Row specifies the row to be queried
 * @param p_Column specifies the column to be queried
 * @return <CODE>true</CODE> if the specified cell is editable, <CODE>false</CODE> otherwise
 */
//---------------------------------------------------------------------------

public boolean isCellEditable   (int p_Row, int p_Column)
    {
	return false;
    }
 
//---------------------------------------------------------------------------

public Statement	getStatementAt (int p_Row)
    {
    Statement  l_Statement;
     
    if ((p_Row >= 0) && (p_Row < m_Statements.size()))
        {
    	l_Statement = m_Statements.elementAt (p_Row);
        return l_Statement;
        }
    else return null;
    }
	
//---------------------------------------------------------------------------

public void	setStatementAt (Statement p_Statement, int p_Row)
    {
    if ((p_Row >= 0) && (p_Row < m_Statements.size()))
        {
    	m_Statements.setElementAt(p_Statement, p_Row);
        }
     }
	
//---------------------------------------------------------------------------

public void addStatement (Statement p_Statement)
    {
    int	l_Row;
	
	if (p_Statement != null)    
        {    
		m_Statements.addElement (p_Statement);
    		
		l_Row = m_Statements.size() - 1;
		
		m_StatementLookUp.put (p_Statement.formatStatementNumber(), Integer.valueOf(l_Row));        
    	fireTableRowsInserted (l_Row,l_Row);
        }
    }
	
//---------------------------------------------------------------------------

public void removeAllStatements ()
	{
	int	l_LastRow;
	
	l_LastRow = m_Statements.size() -1;
	
	m_Statements.removeAllElements ();
	m_StatementLookUp.clear();
	
	if (l_LastRow >= 0) fireTableRowsDeleted (0,l_LastRow);
	}

//---------------------------------------------------------------------------

public void removeStatementAt (int p_Row)
    {
    Statement  l_Statement;

    if ((p_Row >= 0) && (p_Row < m_Statements.size()))
        {
    	l_Statement = m_Statements.elementAt (p_Row);
    	m_Statements.removeElementAt(p_Row);    
    	m_StatementLookUp.remove (l_Statement.formatStatementNumber());       
    	fireTableRowsDeleted (p_Row,p_Row);
        }
    }

//---------------------------------------------------------------------------

public int findStatement (String p_StatementNumber)
    {
    Integer l_Row;
 	
    l_Row = m_StatementLookUp.get (p_StatementNumber);
    
    if (l_Row != null)
        {
        return l_Row.intValue();
        }
    else return (-1);
    }

//---------------------------------------------------------------------------

public void setStatements (Collection <Statement> p_Statements)
    {
    int			l_Row;
    Statement	l_Statement;
    
    if (p_Statements == null) return;
        
    removeAllStatements ();
    m_Statements.addAll (p_Statements);
      
    for (l_Row = 0; l_Row < m_Statements.size(); l_Row++)
        {
    	l_Statement = m_Statements.elementAt(l_Row);
    	m_StatementLookUp.put (l_Statement.formatStatementNumber(), Integer.valueOf(l_Row));   
        }
    
    fireTableRowsInserted (0,m_Statements.size());
    }

//---------------------------------------------------------------------------

public Collection <Statement> getStatements ()
    {
	HashSet <Statement> l_Statements;
	
	l_Statements = new HashSet <Statement> (m_Statements);
	
	return l_Statements;
	}    
        
//***************************************************************************
//* End of Class															*
//***************************************************************************
}
