/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.statement;

/*
 * StatementListBox.java
 *
 * Created on December 15, 2006
 */

 import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.Collection;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelListener;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import lu.tudor.santec.gecamed.addressbook.ejb.entity.beans.Contact;
import lu.tudor.santec.gecamed.addressbook.gui.AddressbookModule;
import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Statement;
import lu.tudor.santec.gecamed.billing.gui.medpres.MedPres;
import lu.tudor.santec.gecamed.core.gui.GECAMedMessage;
import lu.tudor.santec.gecamed.core.gui.IconFetcher;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.TableSorter;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

public class StatementListBox extends JScrollPane implements Relocalizable
    {
    /**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private JTable       				m_StatementTable;
    private StatementListModel			m_Statements;
    private StatementRenderer			m_StatementRenderer;
    private TableSorter					m_StatementSorter;
     
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************

    private static int c_CellMargin = 5;
    
    private static final int c_ThirdPartyColumnWidth 	= 25;
    private static final int c_ReferenceColumnWidth 	= 25;
    private static final int c_StartDateColumnWidth 	= 25;
    private static final int c_EndDateColumnWidth 		= 25;
        
//***************************************************************************
//* Internal Class Definitions	                                          *
//***************************************************************************

//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

public StatementListBox ()
    {	
	super ();
    
	int	l_Count;
	
    m_Statements = new StatementListModel ();
    m_StatementSorter = new TableSorter ();
    m_StatementSorter.setTableModel(m_Statements);
    
    m_StatementRenderer = new StatementRenderer ();
    
    m_StatementTable = new JTable (m_StatementSorter);
    m_StatementTable.setSelectionMode (ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
    
    for (l_Count=0; l_Count < m_Statements.getColumnCount(); l_Count++)
		{
    	if (m_Statements.getColumnClass (l_Count) != null)	
    		m_StatementTable.setDefaultRenderer (m_Statements.getColumnClass(l_Count), m_StatementRenderer);
		}

    m_StatementTable.getColumnModel().getColumn(0).setMaxWidth(22);
    m_StatementTable.getColumnModel().getColumn(0).setMinWidth(22);
    
    m_StatementTable.getColumnModel().getColumn(2).setMaxWidth(50);
    m_StatementTable.getColumnModel().getColumn(2).setMinWidth(50);
    
    m_StatementTable.setColumnSelectionAllowed (false);
    m_StatementTable.setShowGrid (false);
        
    m_StatementSorter.setTableHeader(m_StatementTable.getTableHeader());
//    m_StatementSorter.setSortingStatus(StatementListModel.c_ReferenceColumn, TableSorter.DESCENDING);
    
    this.setViewportView (m_StatementTable);
    }

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Primitives		                                                  *
//***************************************************************************
//---------------------------------------------------------------------------

public void addListSelectionListener (ListSelectionListener p_Listener)
    {
    ListSelectionModel	l_Selection;	
    
    l_Selection = m_StatementTable.getSelectionModel();
    l_Selection.addListSelectionListener (p_Listener);
    }

//---------------------------------------------------------------------------

public void addMouseListener (MouseListener p_Listener)
    {
	m_StatementTable.addMouseListener (p_Listener);
    }    

//---------------------------------------------------------------------------

public void addKeyListener (KeyListener p_Listener)
    {
	m_StatementTable.addKeyListener (p_Listener);
    }    

//---------------------------------------------------------------------------

public void addTableModelListener (TableModelListener p_Listener)
    {
	m_Statements.addTableModelListener(p_Listener);
    }

//---------------------------------------------------------------------------
//Sets the preferred width of the visible column specified by vColIndex. The column
// will be just wide enough to show the column head and the widest cell in the column.
// margin pixels are added to the left and right
// (resulting in an additional width of 2*margin pixels).
//---------------------------------------------------------------------------

public void packColumn (int p_Column, int p_Margin) 
	{
	DefaultTableColumnModel	l_ColumnModel;
	TableColumn 				l_Column;
	TableCellRenderer		l_Renderer;
	Component				l_Component;
	
	int		   				l_Width;
    int						l_Row;
	
	l_ColumnModel = (DefaultTableColumnModel)m_StatementTable.getColumnModel();
	l_Column = l_ColumnModel.getColumn(p_Column);
	l_Width = 0;

    // First step consists in getting width of column header

    l_Renderer = l_Column.getHeaderRenderer();
    if (l_Renderer == null) 
    		{
    		l_Renderer = m_StatementTable.getTableHeader().getDefaultRenderer();
    		}
    
    l_Component = l_Renderer.getTableCellRendererComponent(m_StatementTable,
    													    l_Column.getHeaderValue(), 
    													    false, false, 0, 0);
    l_Width = l_Component.getPreferredSize().width;

    // Next we're going to iterate through all the rows of this columns, in order to
    // to find the widest one

    for (l_Row=0; l_Row < m_StatementTable.getRowCount(); l_Row++) 
    		{
    		l_Renderer = m_StatementTable.getCellRenderer(l_Row, p_Column);
    		l_Component = l_Renderer.getTableCellRendererComponent(m_StatementTable, 
    															   m_StatementTable.getValueAt(l_Row, p_Column), 
    				                                               false, false, l_Row, p_Column);
        
    		l_Width = Math.max (l_Width, l_Component.getPreferredSize().width);
    		}

    // Add margin
    l_Width += 2*p_Margin;

    // Set the width
    l_Column.setPreferredWidth(l_Width);
	}

//---------------------------------------------------------------------------
//Returns the preferred height of a row.
// The result is equal to the tallest cell in the row.
//---------------------------------------------------------------------------

public void packRow (int p_Row, int p_Margin) 
	{
    int					l_RowHeight;
    int					l_ComponentHeight;
	int					l_Column;
    TableCellRenderer		l_Renderer;
    Component			l_Component;
    
    // Get the current default height for all rows
    l_RowHeight = m_StatementTable.getRowHeight();

    // Determine highest cell in the row
    for (l_Column = 0; l_Column < m_StatementTable.getColumnCount(); l_Column++) 
    		{
    		l_Renderer = m_StatementTable.getCellRenderer(p_Row, l_Column);
    		l_Component = m_StatementTable.prepareRenderer(l_Renderer, p_Row, l_Column);
    		l_ComponentHeight = l_Component.getPreferredSize().height + 2*p_Margin;
    		l_RowHeight = Math.max (l_RowHeight, l_ComponentHeight);
    		}
    
    if (m_StatementTable.getRowHeight() != l_RowHeight)
    	m_StatementTable.setRowHeight (p_Row,l_RowHeight);
	}

//---------------------------------------------------------------------------
/**
 * Method is part of the Relocalizable interface. The method does everything
 * required to reflect changes of active Locale
 */
//---------------------------------------------------------------------------

public void relocalize ()
    {
    // Make sure JTable re-creates columns using re-localized headers
        
//	if (m_ActRenderer != null) m_ActRenderer.relocalize ();
    	
    if (m_StatementTable != null) m_StatementTable.createDefaultColumnsFromModel ();
    }

//---------------------------------------------------------------------------
/**
 * doLayout is the main entry point for JTable resizing and row/column space
 * distribution. The ActListBox class overrides the method to impose its own
 * distribution scheme. Width of columns is expressed as percents and we want
 * the table to reflect this distribution. 
 */
//---------------------------------------------------------------------------

public void doLayout ()
	{
    TableColumn  l_Column;
	double		l_Width;
	int			l_ColumnWidth;
	
	super.doLayout ();
	
	l_Width = (double) m_StatementTable.getWidth () / 100;
	
	l_Column = m_StatementTable.getColumnModel().getColumn(0);
	l_ColumnWidth = (int)(l_Width * c_ThirdPartyColumnWidth);
	l_Column.setPreferredWidth(l_ColumnWidth);
	
	l_Column = m_StatementTable.getColumnModel().getColumn(1);
	l_ColumnWidth = (int)(l_Width * c_ReferenceColumnWidth);
	l_Column.setPreferredWidth(l_ColumnWidth);
    
    l_Column = m_StatementTable.getColumnModel().getColumn(2);
	l_ColumnWidth = (int)(l_Width * c_StartDateColumnWidth);
	l_Column.setPreferredWidth(l_ColumnWidth);

    l_Column = m_StatementTable.getColumnModel().getColumn(3);
 	l_ColumnWidth = (int)(l_Width * c_EndDateColumnWidth);
	l_Column.setPreferredWidth(l_ColumnWidth);
	}

//***************************************************************************
//* Class Body		                                                    	*
//***************************************************************************

//---------------------------------------------------------------------------

public void addStatement (Statement p_Statement)
    {
    if (p_Statement != null)
        {
        m_Statements.addStatement (p_Statement);
        m_StatementTable.revalidate();
        }
    }

//---------------------------------------------------------------------------

public void removeSelectedStatements ()
    {
    int []      l_SelectedRows;
    int         l_Selection;
    int			l_ModelRow;
    
    l_Selection = m_StatementTable.getSelectedRowCount();
    
    if (l_Selection > 0)
        {
        l_SelectedRows    = m_StatementTable.getSelectedRows ();
       
        for (l_Selection = l_SelectedRows.length - 1; l_Selection >= 0; l_Selection--)
            {
        	l_ModelRow = m_StatementSorter.modelIndex(l_SelectedRows [l_Selection]);
            m_Statements.removeStatementAt (l_ModelRow);
            }
        
        m_StatementTable.clearSelection();
        m_StatementTable.revalidate();
        }
    }

//---------------------------------------------------------------------------

public void removeAllStatements ()
    {
	m_Statements.removeAllStatements();
	m_StatementTable.clearSelection();
	m_StatementTable.revalidate();
    }

//---------------------------------------------------------------------------

public Statement[] getSelectedStatements ()
    {
	Statement[]  	l_SelectedStatements = null;    
    int[]      		l_SelectedRows;   
    int         	l_Selection;
    int				l_ModelRow;
    
    l_Selection = m_StatementTable.getSelectedRowCount();
    
    if (l_Selection > 0)
        {
        l_SelectedRows    = m_StatementTable.getSelectedRows ();
        l_SelectedStatements = new Statement [l_Selection];
        
        for (l_Selection = 0; l_Selection < l_SelectedRows.length; l_Selection++)
            {
        	l_ModelRow = m_StatementSorter.modelIndex(l_SelectedRows [l_Selection]);
        	l_SelectedStatements [l_Selection] = m_Statements.getStatementAt (l_ModelRow);
            }
        }

    return l_SelectedStatements;
    }

//---------------------------------------------------------------------------

public Statement getSelectedStatement ()
    {
	Statement  	l_SelectedStatement = null;    
    int      	l_SelectedRow;   
    int         l_Selection;
    int			l_ModelRow;
    
    l_Selection = m_StatementTable.getSelectedRowCount();
    
    if (l_Selection == 1)
        {
    	l_SelectedRow = m_StatementTable.getSelectedRow();
    	l_ModelRow = m_StatementSorter.modelIndex(l_SelectedRow);
    	l_SelectedStatement = m_Statements.getStatementAt (l_ModelRow);
         }

    return l_SelectedStatement;
    }

//---------------------------------------------------------------------------

public void setSelectedStatement (Statement	p_Statement)
    {
    int      	l_SelectedRow;   
    int         l_Selection;
    int			l_ModelRow;
    
    l_Selection = m_StatementTable.getSelectedRowCount();
    
    if ((l_Selection == 1) && (p_Statement != null))
        {
    	l_SelectedRow = m_StatementTable.getSelectedRow();
    	l_ModelRow = m_StatementSorter.modelIndex(l_SelectedRow);
    	m_Statements.setStatementAt(p_Statement, l_ModelRow);
         }
    }

//---------------------------------------------------------------------------

public boolean updateStatement (Statement	p_Statement)
	{
	int	l_Row;
	
	if (p_Statement == null) return false;
	
	l_Row = m_Statements.findStatement(p_Statement.formatStatementNumber());
	if (l_Row >= 0)
		{
		m_Statements.setStatementAt(p_Statement, l_Row);
		return true;
		}
	else return false;
	}

//---------------------------------------------------------------------------

public boolean selectStatement (String p_Reference)
    {
    ListSelectionModel	l_Selection;	
    JScrollBar			l_ScrollBar;
    int 				l_Row;
    float				l_ScrollValue;
    boolean 			l_Selected = false;
    
    l_Row = m_Statements.findStatement (p_Reference);
	
    if (l_Row >= 0)
        {
    	l_Row = m_StatementSorter.viewIndex(l_Row);
    	
    	l_Selection = m_StatementTable.getSelectionModel();
    	l_Selection.setSelectionInterval (l_Row,l_Row);
		
    	l_ScrollBar = getVerticalScrollBar ();
		
    	l_ScrollValue  = (float) l_Row / (float) m_Statements.getRowCount();
    	l_ScrollValue *= (l_ScrollBar.getMaximum() - l_ScrollBar.getMinimum()); 
		
    	l_ScrollBar.setValue ((int)l_ScrollValue);
		
    	l_Selected = true;
        }
   
    return (l_Selected);
    }
  
//---------------------------------------------------------------------------

public void deselectAll ()
	{
	ListSelectionModel	l_Selection;
	
	l_Selection = m_StatementTable.getSelectionModel();
	l_Selection.clearSelection();
	}


//---------------------------------------------------------------------------

public boolean containsStatement (String p_Reference)
    {
    if (m_Statements.findStatement (p_Reference) >= 0) return true;
                                 				  else return false;
    }    
  
//---------------------------------------------------------------------------

public void setStatements (Collection <Statement> p_Statements)
    {
	m_Statements.setStatements(p_Statements);
    }    
 
//---------------------------------------------------------------------------

public Collection <Statement> getStatements ()
    {
    return m_Statements.getStatements();
    }    
    
//---------------------------------------------------------------------------

public void packColumns ()
	{
	int l_Column;
	
	for (l_Column=0; l_Column < m_StatementTable.getColumnCount(); l_Column++)
		{
		packColumn (l_Column,c_CellMargin);
		}
	}

//---------------------------------------------------------------------------

public void packRows ()
	{
	int l_Row;
	
	for (l_Row=0; l_Row < m_StatementTable.getRowCount(); l_Row++)
		{
		packRow (l_Row,c_CellMargin);
		}
	}

//***************************************************************************
//* End of Class															*
//***************************************************************************
}
