/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.reminder;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseListener;
import java.beans.PropertyChangeEvent;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Vector;

import javax.swing.AbstractButton;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Account;
import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Invoice;
import lu.tudor.santec.gecamed.billing.ejb.entity.beans.InvoiceStub;
import lu.tudor.santec.gecamed.billing.ejb.session.beans.BillingPrinterBean;
import lu.tudor.santec.gecamed.billing.ejb.session.interfaces.BillingPrinterInterface;
import lu.tudor.santec.gecamed.billing.ejb.session.interfaces.InvoiceInterface;
import lu.tudor.santec.gecamed.billing.gui.BillingModule;
import lu.tudor.santec.gecamed.billing.gui.event.invoice.InvoiceChangeEvent;
import lu.tudor.santec.gecamed.billing.gui.invoice.stub.GenericInvoiceStubListBox;
import lu.tudor.santec.gecamed.billing.gui.invoice.stub.InvoiceStubComparator;
import lu.tudor.santec.gecamed.billing.gui.print.InvoicePrinter;
import lu.tudor.santec.gecamed.billing.utils.BillingAdminSettings;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.GECAMedLists;
import lu.tudor.santec.gecamed.core.gui.GECAMedLog;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.gui.utils.GECAMedGuiUtils;
import lu.tudor.santec.gecamed.core.gui.widgets.CheckBoxEditorField;
import lu.tudor.santec.gecamed.core.gui.widgets.print.PrintParameterFetcher;
import lu.tudor.santec.gecamed.core.gui.widgets.searchtableheader.SearchResetAction;
import lu.tudor.santec.gecamed.core.gui.widgets.searchtableheader.SearchTableHeader;
import lu.tudor.santec.gecamed.core.gui.widgets.searchtableheader.SearchableTable;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.HospitalisationClass;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.Patient;
import lu.tudor.santec.gecamed.patient.ejb.session.beans.PatientAdminBean;
import lu.tudor.santec.gecamed.patient.ejb.session.interfaces.PatientAdminInterface;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

public class ReminderInvoicesPanel extends JPanel implements ActionListener,
															 SearchableTable,
															 TableModelListener,
															 Relocalizable
{
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	/**
	 * static logger for this class
	 */
	private static Logger m_Logger = Logger.getLogger(ReminderInvoicesPanel.class.getName());
	
	private int								m_Tab;
	private Account							m_DefaultAccount;
	
	private GenericInvoiceStubListBox		m_InvoiceListBox;
	private InvoiceStubListModel			m_Invoices;
	private InvoiceStubRenderer				m_InvoiceRenderer;	
	private InvoiceStubSearchTableHeader	m_SearchHeader;
	
	private JButton							m_PrintListButton;
	private JButton							m_PrintCopyButton;
	private JButton							m_BatchPrintButton;
	private JButton							m_RevokeReminderButton;
	private JButton							m_SetThirdPartyButton;
	
	private Collection <Integer>			m_InvoiceIDs;

	private boolean m_CanEditStatements;

	private boolean m_CanRevokeReminders;
	
//	private static Logger m_Logger = Logger.getLogger (ReminderInvoicesPanel.class.getName());

	private static final boolean [][] m_ButtonStateTable = 
	{
					// Print List	Batch Print		Revoke		ThirdParty
	/* Open */ 		{ true,			true,			false,		false		},
	/* Unpaid */ 	{ true,			true,			false,		true		},
	/* First */ 	{ true,			true,			true,		true		},
	/* Second */ 	{ true,			true,			true,		true		},
	/* Order */ 	{ true,			false,			true,		true		}	
	};
	
//***************************************************************************
//* Class Constants				                                            *
//***************************************************************************

	private static final String c_PrintListButton    		= "ReminderInvoicesPanel.PrintListButton";
	private static final String c_PrintListTip    			= "ReminderInvoicesPanel.PrintListTip";
	private static final String c_PrintInvoicesButton 		= "ReminderInvoicesPanel.PrintInvoicesButton";
	private static final String c_PrintInvoicesTip 			= "ReminderInvoicesPanel.PrintInvoicesTip";
	private static final String c_PrintRemindersButton 		= "ReminderInvoicesPanel.PrintRemindersButton";
	private static final String c_PrintFirstRemindersButton = "ReminderInvoicesPanel.PrintFirstRemindersButton";
	private static final String c_PrintSecondRemindersButton= "ReminderInvoicesPanel.PrintSecondRemindersButton";
	private static final String c_PrintPaymentOrderButton 	= "ReminderInvoicesPanel.PrintPaymentOrdersButton";
	
	private static final String c_PrintInvoicesButtonCopy 		= "ReminderInvoicesPanel.PrintInvoicesButtonCopy";
	private static final String c_PrintFirstRemindersButtonCopy = "ReminderInvoicesPanel.PrintFirstRemindersButtonCopy";
	private static final String c_PrintSecondRemindersButtonCopy= "ReminderInvoicesPanel.PrintSecondRemindersButtonCopy";
	private static final String c_PrintPaymentOrderButtonCopy 	= "ReminderInvoicesPanel.PrintPaymentOrdersButtonCopy";
	
	private static final String c_PrintRemindersTip 		= "ReminderInvoicesPanel.PrintRemindersTip";
	private static final String c_RevokeReminderButton 		= "ReminderInvoicesPanel.RevokeReminderButton";
	private static final String c_RevokeReminderTip 		= "ReminderInvoicesPanel.RevokeReminderTip";
	private static final String c_SetThirdPartyButton 		= "ReminderInvoicesPanel.SetThirdPartyButton";
	private static final String c_SetThirdPartyTip 			= "ReminderInvoicesPanel.SetThirdPartyTip";

	private static final String c_PrintIncompleteTitle 		= "ReminderInvoicesPanel.PrintIncompleteTitle";
	private static final String c_PrintIncompleteMessage 	= "ReminderInvoicesPanel.PrintIncompleteMessage";
	
	private static final String c_RevokeIncompleteTitle 	= "ReminderInvoicesPanel.RevokeIncompleteTitle";
	private static final String c_RevokeIncompleteMessage 	= "ReminderInvoicesPanel.RevokeIncompleteMessage";
	private static final String c_RevokeCompleteMessage 	= "ReminderInvoicesPanel.RevokeCompleteMessage";
	
	private static final String c_SwitchIncompleteTitle 	= "ReminderInvoicesPanel.SwitchIncompleteTitle";
	private static final String c_SwitchIncompleteMessage 	= "ReminderInvoicesPanel.SwitchIncompleteMessage";
	private static final String c_SwitchCompleteMessage 	= "ReminderInvoicesPanel.SwitchCompleteMessage";
		
	public static final String c_SearchCriterionChanged		= "ReminderInvoicesPanel.SearchCriterionChanged";
	public static final String c_InvoicesChanged			= "ReminderInvoicesPanel.InvoicesChanged";

	private final static String c_Columns= 	"3dlu,fill:pref:grow,3dlu,fill:max(50dlu;pref)";
	
	private final static String c_Rows=    	"3dlu,fill:max(10dlu;pref)," +
											"3dlu,fill:max(10dlu;pref)," + 
											"3dlu,fill:max(10dlu;pref)," + 
											"3dlu,fill:max(10dlu;pref)," + 
											"3dlu,fill:max(10dlu;pref),fill:pref:grow";
	
	private static final Hashtable <Integer,Integer> m_SortingProperties = new Hashtable <Integer,Integer> ();
	
	static {
			m_SortingProperties.put(InvoiceStubListModel.c_InvoiceNumberColumn, InvoiceStubComparator.c_Id);
			m_SortingProperties.put(InvoiceStubListModel.c_InvoiceDateColumn, InvoiceStubComparator.c_InvoiceDate);
			m_SortingProperties.put(InvoiceStubListModel.c_PatientSSNColumn, InvoiceStubComparator.c_PatientSSN);
			m_SortingProperties.put(InvoiceStubListModel.c_PatientNameColumn, InvoiceStubComparator.c_PatientFullName);
			m_SortingProperties.put(InvoiceStubListModel.c_DueDateColumn, InvoiceStubComparator.c_DueDate);
			m_SortingProperties.put(InvoiceStubListModel.c_ReminderDateColumn, InvoiceStubComparator.c_ReminderDate);
			m_SortingProperties.put(InvoiceStubListModel.c_AmountColumn, InvoiceStubComparator.c_Amount);
			m_SortingProperties.put(InvoiceStubListModel.c_PaymentColumn, InvoiceStubComparator.c_Payment);
			m_SortingProperties.put(InvoiceStubListModel.c_DeductionColumn, InvoiceStubComparator.c_Deduction);
			m_SortingProperties.put(InvoiceStubListModel.c_BalanceColumn, InvoiceStubComparator.c_Balance);
	};

//---------------------------------------------------------------------------
//***************************************************************************
//* Constructor					                                            *
//***************************************************************************
//---------------------------------------------------------------------------
	
public ReminderInvoicesPanel (int	p_Tab)
	{
	CellConstraints		l_Constraints;
	FormLayout			l_Layout;
	TableColumn			l_Column;
	TableColumnModel	l_ColumnModel;
	ImageIcon			l_Icon;
	
	m_Tab = p_Tab;
	
	l_Constraints  	= new CellConstraints();
	l_Layout		= new FormLayout(c_Columns, c_Rows);

	this.setLayout(l_Layout);
	
	this.setOpaque(false);

	m_Invoices = new InvoiceStubListModel ();
	m_Invoices.addTableModelListener(this);
	
	m_InvoiceRenderer = new InvoiceStubRenderer ();
	m_InvoiceListBox = new GenericInvoiceStubListBox (m_Invoices);
	m_InvoiceListBox.setInvoiceStubRenderer(m_InvoiceRenderer);
	
	m_SearchHeader = new InvoiceStubSearchTableHeader (m_InvoiceListBox.getTable());
	m_SearchHeader.addPropertyChangeListener(this);  
	
	m_InvoiceRenderer.setSearchHeader(m_SearchHeader);
	
	l_ColumnModel = m_InvoiceListBox.getTable().getColumnModel(); 
	
	l_Column = l_ColumnModel.getColumn(InvoiceStubListModel.c_TickedColumn);
	l_Column.setCellEditor(new CheckBoxEditorField());
	
	m_InvoiceListBox.getViewport().setOpaque(false);
	m_InvoiceListBox.setBackground(GECAMedColors.c_ScrollPaneBackground);
 
	m_InvoiceListBox.setSorting(InvoiceStubListModel.c_PatientNameColumn, GenericInvoiceStubListBox.c_Ascending);
	
	l_Icon = BillingModule.getButtonIcon ("rmndr_print_list.png");
	m_PrintListButton = new JButton ();
	m_PrintListButton.setIcon(l_Icon);
	m_PrintListButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_PrintListButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_PrintListButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
	m_PrintListButton.setEnabled (BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_printReminderLists));
	m_PrintListButton.addActionListener(this);
	
	l_Icon = BillingModule.getButtonIcon ("rmndr_batch_print.png");
	m_BatchPrintButton = new JButton ();
	m_BatchPrintButton.setIcon(l_Icon);
	m_BatchPrintButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_BatchPrintButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_BatchPrintButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
	m_BatchPrintButton.setEnabled (false);
	m_BatchPrintButton.addActionListener(this);

	l_Icon = BillingModule.getButtonIcon ("rmndr_batch_print.png");
	m_PrintCopyButton = new JButton ();
	m_PrintCopyButton.setIcon(l_Icon);
	m_PrintCopyButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_PrintCopyButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_PrintCopyButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
	m_PrintCopyButton.setEnabled (false);
	m_PrintCopyButton.addActionListener(this);
	
	l_Icon = BillingModule.getButtonIcon ("rmndr_revoke.png");
	m_RevokeReminderButton = new JButton ();
	m_RevokeReminderButton.setIcon(l_Icon);
	m_RevokeReminderButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_RevokeReminderButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_RevokeReminderButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
	m_RevokeReminderButton.setEnabled (false);
	m_RevokeReminderButton.addActionListener(this);
	
	l_Icon = BillingModule.getButtonIcon ("rmndr_thirdparty.png");
	m_SetThirdPartyButton = new JButton ();
	m_SetThirdPartyButton.setIcon(l_Icon);
	m_SetThirdPartyButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_SetThirdPartyButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_SetThirdPartyButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
	m_SetThirdPartyButton.setEnabled (false);
	m_SetThirdPartyButton.addActionListener(this);

	this.refelectSelectedTab(p_Tab);
	this.relocalize();
	
	this.add (m_InvoiceListBox, 		l_Constraints.xywh(2, 2, 1, 10));
	
	int row = 2;
	this.add (m_PrintListButton, 		l_Constraints.xywh(4, row, 1, 1));
	
	if (m_Tab != ReminderPanel.c_DemandsForPaymentTab) {
		row +=2;
		this.add (m_BatchPrintButton, 		l_Constraints.xywh(4, row, 1, 1));		
	}
	
	row +=2;
	this.add (m_PrintCopyButton, 		l_Constraints.xywh(4, row, 1, 1));
	row +=2;
	this.add (m_RevokeReminderButton, 	l_Constraints.xywh(4, row, 1, 1));
	row +=2;
	this.add (m_SetThirdPartyButton, 	l_Constraints.xywh(4, row, 1, 1));

	KeyStroke l_Stroke = KeyStroke.getKeyStroke("ESCAPE");
	
	this.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT).put(l_Stroke,"cancelSearch");
	this.getActionMap().put("cancelSearch", new SearchResetAction (this,"cancelSearch"));	
	
	m_InvoiceIDs = new Vector <Integer> ();
	}
		
//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives					                                            *
//***************************************************************************
//---------------------------------------------------------------------------

private Collection <InvoiceStub> getTickedInvoiceStubs ()
{
Collection <InvoiceStub>	l_InvoiceStubs;

l_InvoiceStubs = m_Invoices.getTickedInvoiceStubs();

return l_InvoiceStubs;
}

private Collection <InvoiceStub> getInvoiceStubs ()
	{
	Collection <InvoiceStub>	l_InvoiceStubs;
	
	l_InvoiceStubs = m_Invoices.getTickedInvoiceStubs();
	if ((l_InvoiceStubs == null) || (l_InvoiceStubs.size() == 0)) {
		l_InvoiceStubs = m_Invoices.getInvoiceStubs();		
	}
	
	return l_InvoiceStubs;
	}

//---------------------------------------------------------------------------

private void removeFromList (Collection <InvoiceStub> p_Candidates)
	{
	Collection <InvoiceStub>	l_InvoiceStubs;
		
	l_InvoiceStubs = m_Invoices.getInvoiceStubs();
	if ((l_InvoiceStubs != null) && (p_Candidates != null))
		{
		l_InvoiceStubs.removeAll(p_Candidates);
		this.setInvoiceStubs(l_InvoiceStubs);
		}
	}

//---------------------------------------------------------------------------

private void refelectSelectedTab (int p_Tab)
	{
	boolean[]	l_States;
	boolean		l_Enabled = false;
	
	l_States = m_ButtonStateTable [p_Tab];
	
	l_Enabled = (l_States[0])?BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_printReminderLists):false;
	m_PrintListButton.setEnabled (l_Enabled);
	
	if (m_Tab == ReminderPanel.c_OpenInvoicesTab) {
		l_Enabled = (l_States[1])?BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_printInvoices):false;		
	} else 	{
		l_Enabled = (l_States[1])?BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_printReminders):false;
	}
	
	m_BatchPrintButton.setEnabled (l_Enabled);
	
	m_PrintCopyButton.setEnabled (true);
	
	m_CanRevokeReminders = (l_States[2])?BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_revokeReminders):false;
	m_RevokeReminderButton.setEnabled	(m_CanRevokeReminders);
	
	m_CanEditStatements = (l_States[3])?BillingModule.userHasPermission(BillingModule.c_BillingModule, BillingModule.c_editStatements):false;
	m_SetThirdPartyButton.setEnabled(m_CanEditStatements && m_Tab == ReminderPanel.c_DemandsForPaymentTab);
	}

//---------------------------------------------------------------------------

private void handlePrintListButton ()
	{
	HashMap<String, Object> 	l_Parameter;
	InvoicePrinter 				l_Printer;
	Collection<InvoiceStub>		l_InvoiceStubs;
	String						l_Title;
	int							l_SortingColumn = -1;
	int							l_SortDirection	= 0;
	
	l_SortingColumn = m_InvoiceListBox.getSortColumn();
	l_SortDirection = m_InvoiceListBox.getSortDirection(l_SortingColumn);
	
	if (m_SortingProperties.containsKey(l_SortingColumn))
		 l_SortingColumn = m_SortingProperties.get(l_SortingColumn);
	else l_SortingColumn = -1;
		
	switch (m_Tab)
		{
		case ReminderPanel.c_OpenInvoicesTab:
				
			l_Title = Translatrix.getTranslationString("ReminderInvoicesPanel.OpenInvoiceListTitle");
			break;
				
		case ReminderPanel.c_UnpaidInvoicesTab:
				
			l_Title = Translatrix.getTranslationString("ReminderInvoicesPanel.UnpaidInvoiceListTitle");
			break;
				
		case ReminderPanel.c_FirstNoticeTab:
				
			l_Title = Translatrix.getTranslationString("ReminderInvoicesPanel.FirstNoticeListTitle");
			break;
				
		case ReminderPanel.c_SecondNoticeTab:
				
			l_Title = Translatrix.getTranslationString("ReminderInvoicesPanel.SecondNoticeListTitle");
			break;

		case ReminderPanel.c_DemandsForPaymentTab:
			
			l_Title = Translatrix.getTranslationString("ReminderInvoicesPanel.DemandForPaymentListTitle");
			break;
		
		default:
			
			l_Title = "";
		}
	
	l_InvoiceStubs = m_Invoices.getInvoiceStubs();
	l_Parameter = PrintParameterFetcher.createDefaultPrintParameter(l_InvoiceStubs);
		
	// fetch and fill print parameters
	l_InvoiceStubs = fillPrintParams(l_InvoiceStubs);


	l_Printer = new InvoicePrinter ();
	l_Printer.printReminderList(l_InvoiceStubs, l_Title, l_SortingColumn,l_SortDirection, l_Parameter, askGroupOption(l_Title));
	l_Printer.close();
	}

//---------------------------------------------------------------------------

private Integer askGroupOption(String l_Title) {
	LinkedHashMap<String, Integer> groupOptions = new LinkedHashMap<String, Integer>();
	groupOptions.put(Translatrix.getTranslationString("BillingPrinterBean.ListGroupNONE"), 		InvoiceStubComparator.GROUP_NONE);
	groupOptions.put(Translatrix.getTranslationString("BillingPrinterBean.ListGroupPhysician"), InvoiceStubComparator.GROUP_PHYSICIAN);
	if (m_Tab == ReminderPanel.c_DemandsForPaymentTab) {
		groupOptions.put(Translatrix.getTranslationString("BillingPrinterBean.ListGroupBailif"), 	InvoiceStubComparator.GROUP_BAILIF);
		groupOptions.put(Translatrix.getTranslationString("BillingPrinterBean.ListGroupBailifMonth"),InvoiceStubComparator.GROUP_BAILIF_MONTH);		
	}
	
	Object[] possibilities = groupOptions.keySet().toArray();
	
	String choice = (String)JOptionPane.showInputDialog(
	                    MainFrame.getInstance(),
	                    Translatrix.getTranslationString("BillingPrinterBean.ListDescription"),
	                    l_Title,
	                    JOptionPane.PLAIN_MESSAGE,
	                    BillingModule.getButtonIcon ("rmndr_print_list.png"),
	                    possibilities,
	                    Translatrix.getTranslationString("BillingPrinterBean.ListGroupNONE"));
	
	if (groupOptions.get(choice) == null) {
		return InvoiceStubComparator.GROUP_NONE;
	}
	
	return groupOptions.get(choice);
}


/**
 * Fetch the Bailifs for the invoice stubs and set them to a transient field.
 * @param l_InvoiceStubs
 * @return
 */
private Collection<InvoiceStub> fillPrintParams(Collection<InvoiceStub> l_InvoiceStubs) {
	try {
		HashMap<Integer, Physician> l_PhysicianHash = GECAMedLists.getIdMapReference(Physician.class);
		PatientAdminInterface patAdmin = (PatientAdminInterface) ManagerFactory.getRemote(PatientAdminBean.class);
		BillingPrinterInterface printerFace = (BillingPrinterInterface) ManagerFactory.getRemote(BillingPrinterBean.class);
		for (InvoiceStub invoiceStub : l_InvoiceStubs) {
			try {
				invoiceStub.setPhysician(""+l_PhysicianHash.get(invoiceStub.getPhysicianId()));
				
				if (m_Tab == ReminderPanel.c_DemandsForPaymentTab) {
					// fetch Bailifs
					Integer patID = invoiceStub.getPatientId();
					Patient p = patAdmin.getPatient(patID);
					String bailif = printerFace.getBailiffForPatient(p, true);
					invoiceStub.setBailif(bailif);				
				} else {
					invoiceStub.setBailif("");
				}
			} catch (Exception e) {
				m_Logger.error("error Fetching PrintParams for invoicestub: " + invoiceStub, e);
			}
		}		
	} catch (Exception e) {
		m_Logger.error("error Fetching PrintParams", e);
	}
	return l_InvoiceStubs;
}

/**
 * check each invoice stub if it might be splittable.
 * 
 * @param l_InvoiceStubs
 * @return
 */
private Collection <Invoice> findSplittableInvoices(Collection<InvoiceStub>	l_InvoiceStubs) {
	Collection <Invoice>	l_SplittableInvoices = new ArrayList<Invoice>();

	try {
		InvoiceInterface l_InvoiceInterface = BillingModule.getInstance().getInvoiceInterface();
		HospitalisationClass l_FirstClass = l_InvoiceInterface.getHospitalisationClass(HospitalisationClass.c_FirstClass);
		
		for (InvoiceStub invoiceStub : l_InvoiceStubs) {
			
			// pre check for Class 1 Hospitalisation
			if (invoiceStub.getHospitalisationClassId() != null && l_FirstClass.getId() != null  && invoiceStub.getHospitalisationClassId().equals(l_FirstClass.getId())) {
				// fetch complete invoice
				Invoice l_Invoice = l_InvoiceInterface.getInvoiceByID(invoiceStub.getId());
				// and check it.
				if (l_InvoiceInterface.isInvoiceSplittable(l_Invoice)) {
					l_SplittableInvoices.add(l_Invoice);
				}
			}
		}
	} catch (Exception e) {
		m_Logger.warn("Error checking for splitabble invoices!", e);
	}
	return l_SplittableInvoices;
}

private void handleBatchPrintButton (boolean progressWorkflow)
	{
	Collection <InvoiceStub>	l_InvoiceStubs;
	Collection <InvoiceStub>	l_NotPrintedInvoiceStubs;
	Collection <Integer>		l_InvoiceIDs;
	Integer						l_NotPrinted;
	String []					l_Filler;
	int							l_SortingColumn = -1;
	int							l_SortDirection	= 0;

	MailingPrintDialog l_Dialog;
		
	l_InvoiceStubs = this.getInvoiceStubs();
	
	// Check for Splitting First Class Invoice into CNS and Private Invoice
	Collection <Invoice> l_SplittableInvoices = findSplittableInvoices(l_InvoiceStubs);
	if (progressWorkflow && l_SplittableInvoices.size() > 0) {
		HashMap<Integer, InvoiceStub> stubHash = new HashMap<Integer, InvoiceStub>();
		for (InvoiceStub stub : l_InvoiceStubs) {
			stubHash.put(stub.getId(), stub);
		}
		
		DateFormat df = GECAMedGuiUtils.getDateFormat(false);
		StringBuffer sb = new StringBuffer();
		for (Invoice invoice : l_SplittableInvoices) {
			sb.append(invoice.formatInvoiceNumber(true, false)).append(":    ");
			sb.append(df.format(invoice.getInvoiceDate())).append("  -  ");
			sb.append(stubHash.get(invoice.getId()).getPatientFullName()).append("\n");
		}
		
		String[] vars = new String[] { sb.toString() };
		int n = JOptionPane.showConfirmDialog(this, Translatrix.getTranslationString("InvoiceEditorPanel.SplitFirstClassInvoiceText", vars), Translatrix.getTranslationString("InvoiceEditorPanel.SplitFirstClassInvoiceTitle"), JOptionPane.YES_NO_CANCEL_OPTION);

		if (n == JOptionPane.CANCEL_OPTION || n == JOptionPane.CLOSED_OPTION ) {
			return;
		} else if (n == JOptionPane.YES_OPTION) {
			try {
				InvoiceInterface l_InvoiceInterface = BillingModule.getInstance().getInvoiceInterface();
				Collection<Integer> splittedIDs = new ArrayList<Integer>();
				Collection<Integer> createdIDs = new ArrayList<Integer>();
				
				for (Invoice invoice : l_SplittableInvoices) {
					List<Invoice> newInvoices = l_InvoiceInterface.splitFirstClassHospitalizedInvoice(invoice);
					splittedIDs.add(invoice.getId());
					createdIDs.add(newInvoices.get(0).getId());
					createdIDs.add(newInvoices.get(1).getId());
					GECAMedLog.user("Billing", "SPLITTING Invoice", "Splitted First Class Hospitalized Invoice with ID " + invoice.getId() + " into 2 new Invoices ");				
				}
				BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceDeleted, splittedIDs);
				BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceUpdated, createdIDs);
				return;
			} catch (Exception e) {
				m_Logger.error("Error splitting Invoices!", e);
			}
			
		}
	}
	
	l_Dialog = new MailingPrintDialog ();
	l_Dialog.setMarkAsCopy(true);
	
	l_SortingColumn = m_InvoiceListBox.getSortColumn();
	l_SortDirection = m_InvoiceListBox.getSortDirection(l_SortingColumn);
		
//	if (m_SortingProperties.containsKey(l_SortingColumn))
//		 l_SortingColumn = m_SortingProperties.get(l_SortingColumn);
//	else l_SortingColumn = -1;
//
//	
//	if (l_SortingColumn != InvoiceStubComparator.c_None)
//		{
//		l_Dialog.setSortingCriteria(l_SortingColumn, l_SortDirection);
//		}
//	
	
	// fetch and fill print parameters
	l_InvoiceStubs = fillPrintParams(l_InvoiceStubs);
	l_InvoiceStubs = InvoicePrinter.setStubGroupingString(l_InvoiceStubs,  askGroupOption(Translatrix.getTranslationString("BillingPrinterBean.ListDescription")));
	
	// sort Collection
	l_InvoiceStubs = InvoicePrinter.sortCollection(l_InvoiceStubs, l_SortingColumn, l_SortDirection);

	
	Collection<InvoiceStub> l_NonluxStubs = new ArrayList<InvoiceStub>();
	
	if (progressWorkflow) {
		switch (m_Tab)
		{
		case ReminderPanel.c_OpenInvoicesTab:
			
			l_Dialog.setNumberOfInvoiceCopies(1);
			l_Dialog.setJoinReminders(null); //(Boolean)BillingModule.getSetting(BillingAdminSettings.c_JoinRemindersOfPatient));
			l_Dialog.setReminderLevel (BillingPrinterBean.c_PrintInvoice);
			break;
			
		case ReminderPanel.c_UnpaidInvoicesTab:
			
			l_Dialog.setNumberOfInvoiceCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_ReminderInvoicesSetting));
			l_Dialog.setNumberOfReminderCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_ReminderCopiesSetting));
			l_Dialog.setJoinReminders((Boolean)BillingModule.getSetting(BillingAdminSettings.c_JoinRemindersOfPatient));
			l_Dialog.setReminderLevel (BillingPrinterBean.c_PrintFirstNotice);
			break;
			
		case ReminderPanel.c_FirstNoticeTab:
			
			l_Dialog.setNumberOfInvoiceCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_ReminderInvoicesSetting));
			l_Dialog.setNumberOfReminderCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_ReminderCopiesSetting));				
			l_Dialog.setJoinReminders((Boolean)BillingModule.getSetting(BillingAdminSettings.c_JoinRemindersOfPatient));
			l_Dialog.setReminderLevel (BillingPrinterBean.c_PrintLastNotice);
			break;
			
		case ReminderPanel.c_SecondNoticeTab:
			
			l_NonluxStubs = getNonLux(l_InvoiceStubs);
			
			l_Dialog.setAccount (m_DefaultAccount);
			l_Dialog.setNumberOfInvoiceCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_PaymentOrderInvoicesSetting));				
			l_Dialog.setNumberOfReminderCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_PaymentOrderCopiesSetting));
//			l_Dialog.setJoinReminders((Boolean)BillingModule.getSetting(BillingAdminSettings.c_JoinPaymentOrdersOfPatient));
			l_Dialog.setJoinReminders(null);
			l_Dialog.setReminderLevel (BillingPrinterBean.c_PrintPaymentOrder);
			break;
			
		case ReminderPanel.c_DemandsForPaymentTab:
		}
	} else {
		switch (m_Tab)
		{
		case ReminderPanel.c_OpenInvoicesTab:
		case ReminderPanel.c_UnpaidInvoicesTab:
			
			l_Dialog.setNumberOfInvoiceCopies(1);
			l_Dialog.setJoinReminders(null); //(Boolean)BillingModule.getSetting(BillingAdminSettings.c_JoinRemindersOfPatient));
			l_Dialog.setReminderLevel (BillingPrinterBean.c_PrintInvoice);
			break;
			
		case ReminderPanel.c_FirstNoticeTab:
			
			l_Dialog.setNumberOfInvoiceCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_ReminderInvoicesSetting));
			l_Dialog.setNumberOfReminderCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_ReminderCopiesSetting));
			l_Dialog.setJoinReminders((Boolean)BillingModule.getSetting(BillingAdminSettings.c_JoinRemindersOfPatient));
			l_Dialog.setReminderLevel (BillingPrinterBean.c_PrintFirstNotice);
			break;
			
		case ReminderPanel.c_SecondNoticeTab:
			
			l_Dialog.setNumberOfInvoiceCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_ReminderInvoicesSetting));
			l_Dialog.setNumberOfReminderCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_ReminderCopiesSetting));				
			l_Dialog.setJoinReminders((Boolean)BillingModule.getSetting(BillingAdminSettings.c_JoinRemindersOfPatient));
			l_Dialog.setReminderLevel (BillingPrinterBean.c_PrintLastNotice);
			break;
			
		case ReminderPanel.c_DemandsForPaymentTab:
			
			l_NonluxStubs = getNonLux(l_InvoiceStubs);
			
			l_Dialog.setAccount (m_DefaultAccount);
			l_Dialog.setNumberOfInvoiceCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_PaymentOrderInvoicesSetting));				
			l_Dialog.setNumberOfReminderCopies((Integer)BillingModule.getSetting(BillingAdminSettings.c_PaymentOrderCopiesSetting));
//			l_Dialog.setJoinReminders((Boolean)BillingModule.getSetting(BillingAdminSettings.c_JoinPaymentOrdersOfPatient));
			l_Dialog.setJoinReminders(null);
			l_Dialog.setReminderLevel (BillingPrinterBean.c_PrintPaymentOrder);
			break;
		}
		
	}
	
	if (l_NonluxStubs.size() > 0) {
		// ask to not print Payment Order for these invoices
		int n = JOptionPane.showConfirmDialog(
			    MainFrame.getInstance(),
			    Translatrix.getTranslationString("ReminderInvoicesPanel.paymentorderForNonLuxText", new String[] {l_NonluxStubs.size()+""}),
			    Translatrix.getTranslationString("ReminderInvoicesPanel.paymentorderForNonLuxTitle"),
			    JOptionPane.YES_NO_OPTION);
		
		if (n == JOptionPane.NO_OPTION) {
			l_InvoiceStubs.removeAll(l_NonluxStubs);			
		}
	}
	
	// fetch and fill print parameters
	l_InvoiceStubs = fillPrintParams(l_InvoiceStubs);
	
	l_Dialog.setInvoiceStubs (l_InvoiceStubs);	
	l_Dialog.setProgressWorkFlow(progressWorkflow);

	l_Dialog.pack();
	MainFrame.showDialogCentered (l_Dialog);
	l_NotPrintedInvoiceStubs = l_Dialog.getNotPrintedInvoiceStubs();
		
	if (l_Dialog.wasAborted() == false)
		{
		l_NotPrinted = l_NotPrintedInvoiceStubs.size();
		
		if (l_NotPrinted > 0)
			{
			l_Filler = new String [1];
			l_Filler [0] = l_NotPrinted.toString();
				
			BillingModule.notifyUser(c_PrintIncompleteTitle, c_PrintIncompleteMessage, l_Filler);
			
			l_InvoiceStubs.removeAll(l_NotPrintedInvoiceStubs);
			}
		}
		
	if (progressWorkflow) {
		this.removeFromList (l_InvoiceStubs);
		this.firePropertyChange (c_InvoicesChanged, null,null);
		
		l_InvoiceIDs = InvoiceStub.getInvoiceIDs (l_InvoiceStubs);
		BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceStatusChanged, l_InvoiceIDs);		
	}
	
	}

//---------------------------------------------------------------------------

private Collection<InvoiceStub> getNonLux(Collection<InvoiceStub> l_InvoiceStubs) {

	PatientAdminInterface patAdmin = (PatientAdminInterface) ManagerFactory.getRemote(PatientAdminBean.class);
	BillingPrinterInterface printerFace = (BillingPrinterInterface) ManagerFactory.getRemote(BillingPrinterBean.class);
	Collection<InvoiceStub> nonLuxStubs = new ArrayList<InvoiceStub>();
	
	for (InvoiceStub invoiceStub : l_InvoiceStubs) {
		try {
				// fetch Bailifs if not already done.
				if (invoiceStub.getBailif() == null || invoiceStub.getBailif().equals("")) {
					Integer patID = invoiceStub.getPatientId();
					Patient p = patAdmin.getPatient(patID);
					String bailif = printerFace.getBailiffForPatient(p, false);
					invoiceStub.setBailif(bailif);
				}
				// if we have a Bailif -> add to list, otherwise this is a foreigner, 
				// so we don't need to print a payment order. 
				if (invoiceStub.getBailif() == null || invoiceStub.getBailif().equals("")) {
					nonLuxStubs.add(invoiceStub);
					m_Logger.info("Not printing payment order for Invoice: " + invoiceStub + " (Foreigner)");
				}
		} catch (Exception e) {
			m_Logger.error("error Fetching PrintParams for invoicestub: " + invoiceStub, e);
		}
	}
	
	return nonLuxStubs;
}

private void handleRevokeReminderButton ()
	{
	Collection <InvoiceStub>	l_InvoiceStubs;
	Collection <InvoiceStub>	l_RevokedInvoiceStubs;
	Collection <Integer>		l_InvoiceIDs;
	RevokeReminderDialog		l_Dialog;
	String[]					l_Filler;
	
	l_InvoiceStubs = this.getTickedInvoiceStubs();
	
	l_Dialog = new RevokeReminderDialog ();
	l_Dialog.setInvoiceStubs(l_InvoiceStubs);
	l_Dialog.pack ();
	MainFrame.showDialogCentered (l_Dialog);
	
	l_RevokedInvoiceStubs = l_Dialog.getRevokedInvoiceStubs();
	if ((l_RevokedInvoiceStubs != null) && (l_RevokedInvoiceStubs.size() > 0))
		{
		this.removeFromList(l_RevokedInvoiceStubs);
		this.firePropertyChange(c_InvoicesChanged, null,null);
		
		l_InvoiceIDs = InvoiceStub.getInvoiceIDs (l_RevokedInvoiceStubs);
		BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceStatusChanged, l_InvoiceIDs);		
		}
	
	if ((l_RevokedInvoiceStubs == null) || (l_RevokedInvoiceStubs.size() != l_InvoiceStubs.size()))
		{
		
		if (!l_Dialog.canceled())
			BillingModule.notifyUser (c_RevokeIncompleteTitle, c_RevokeIncompleteMessage,null);
	
		if (l_RevokedInvoiceStubs != null) l_InvoiceStubs.removeAll(l_RevokedInvoiceStubs);
	
		m_Invoices.tickInvoiceStubs(l_InvoiceStubs);
		m_InvoiceListBox.validate();
		
		l_InvoiceIDs = InvoiceStub.getInvoiceIDs (l_InvoiceStubs);
		BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceStatusChanged, l_InvoiceIDs);
		}
	else
		{
		l_InvoiceIDs = InvoiceStub.getInvoiceIDs (l_InvoiceStubs);
		BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceStatusChanged, l_InvoiceIDs);
		
		l_Filler = new String [1];
		l_Filler [0] = Integer.valueOf(l_InvoiceStubs.size()).toString();
		
		MainFrame.getInstance().showMessage(Translatrix.getTranslationString(c_RevokeCompleteMessage,l_Filler));
		}
	}

//---------------------------------------------------------------------------

private void handleThirdPartyButton ()
	{
	Collection <InvoiceStub>	l_InvoiceStubs = null;
	Collection <InvoiceStub>	l_SwitchedInvoiceStubs;
	Collection <Integer>		l_InvoiceIDs;
	ThirdPartyDialog			l_Dialog;
	String[]					l_Filler;
	
	// get selected invoiced
	l_InvoiceStubs = this.getTickedInvoiceStubs();		

	// for payment order tab take all expired invoices if non is selected
	if ( (l_InvoiceStubs==null || l_InvoiceStubs.size()==0) && m_Tab == ReminderPanel.c_DemandsForPaymentTab) {
		l_InvoiceStubs = new ArrayList <InvoiceStub>();
		Collection<InvoiceStub> l_Stubs = this.getInvoiceStubs();
		for (InvoiceStub stub : l_Stubs) {
			if (stub.isExpired()) {
				l_InvoiceStubs.add(stub);
			}
		}
	}
	
	if (l_InvoiceStubs == null || l_InvoiceStubs.size() == 0 ) {
		return;
	}

	l_Dialog = new ThirdPartyDialog ();
	l_Dialog.setInsurances (BillingModule.getInstance().getThirdPartyPayers());
	l_Dialog.setInvoiceStubs (l_InvoiceStubs);
	l_Dialog.pack ();
	l_Dialog.setSize(l_Dialog.getWidth(), l_Dialog.getHeight() + 50);
	MainFrame.showDialogCentered (l_Dialog);
	
	l_SwitchedInvoiceStubs = l_Dialog.getSwitchedInvoiceStubs();
	if ((l_SwitchedInvoiceStubs != null) && (l_SwitchedInvoiceStubs.size() > 0))
		{
		this.removeFromList(l_SwitchedInvoiceStubs);
		this.firePropertyChange(c_InvoicesChanged, null,null);
		
		l_InvoiceIDs = InvoiceStub.getInvoiceIDs (l_SwitchedInvoiceStubs);
		BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceStatusChanged, l_InvoiceIDs);		
		}
	
	if ((l_SwitchedInvoiceStubs == null) || (l_SwitchedInvoiceStubs.size() != l_InvoiceStubs.size()))
		{
		
		if (!l_Dialog.canceled())
			BillingModule.notifyUser (c_SwitchIncompleteTitle, c_SwitchIncompleteMessage,null);
		
		if (l_SwitchedInvoiceStubs != null) l_InvoiceStubs.removeAll(l_SwitchedInvoiceStubs);
		
		m_Invoices.tickInvoiceStubs(l_InvoiceStubs);
		m_InvoiceListBox.validate();
		
//		l_InvoiceIDs = InvoiceStub.getInvoiceIDs (l_InvoiceStubs);
//		BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceStatusChanged, l_InvoiceIDs);
		}
	else
		{
//		l_InvoiceIDs = InvoiceStub.getInvoiceIDs (l_InvoiceStubs);
//		BillingModule.getInstance().fireMultipleInvoiceChange(InvoiceChangeEvent.c_InvoiceStatusChanged, l_InvoiceIDs);
		
		l_Filler = new String [2];
		l_Filler [0] = Integer.valueOf(l_SwitchedInvoiceStubs.size()).toString();
		l_Filler [1] = (l_Dialog.getThirdPartySponsor() != null)?l_Dialog.getThirdPartySponsor().getAcronym():"n/a";
			
		MainFrame.getInstance().showMessage(Translatrix.getTranslationString(c_SwitchCompleteMessage,l_Filler));
		}
	}

//---------------------------------------------------------------------------

//***************************************************************************
//* Class Body					                                            *
//***************************************************************************
//---------------------------------------------------------------------------

public void addListSelectionListener (ListSelectionListener p_Listener)
	{
	m_InvoiceListBox.addListSelectionListener(p_Listener);
	}

//---------------------------------------------------------------------------

public void addMouseListener (MouseListener p_Listener)
	{
	m_InvoiceListBox.addMouseListener(p_Listener);
	}

//---------------------------------------------------------------------------

public void setDefaultAccount (Account p_DefaultAccount)
	{
	m_DefaultAccount = p_DefaultAccount;	
	}

//---------------------------------------------------------------------------

public void setInvoiceStubs (Collection <InvoiceStub> p_InvoiceStubs)
	{
	m_InvoiceIDs.clear();
	m_InvoiceListBox.setInvoiceStubs(p_InvoiceStubs);
	m_InvoiceListBox.packColumns();
	}

//---------------------------------------------------------------------------

public void selectFirstInvoiceStub ()
	{
	m_InvoiceListBox.selectFirstInvoiceStub();	
	m_InvoiceListBox.getTable().requestFocusInWindow();
	}

//---------------------------------------------------------------------------

public InvoiceStub getSelectedInvoiceStub ()
	{
	return m_InvoiceListBox.getSelectedInvoiceStub();
	}

//---------------------------------------------------------------------------
/**
 * Returns a collection holding all invoice IDs of invoices currently
 * shown in this panel.
 * @return a collection holding all invoice IDs of currently shown
 * invoices.
 */
//---------------------------------------------------------------------------

public Collection <Integer> getInvoiceIDs ()
	{
//	if (m_InvoiceIDs.isEmpty())
//		m_InvoiceIDs.addAll(InvoiceStub.getInvoiceIDs(m_InvoiceListBox.getInvoiceStubs()));
//		
//	return m_InvoiceIDs; 
	
	return InvoiceStub.getInvoiceIDs (m_Invoices.getInvoiceStubs());
	}

//---------------------------------------------------------------------------

public void addInvoiceStubs (Collection <InvoiceStub> p_InvoiceStubs)
	{
	m_Invoices.addInvoiceStubs(p_InvoiceStubs);
	}

//---------------------------------------------------------------------------

public void updateInvoiceStubs (Collection <InvoiceStub> p_InvoiceStubs)
	{
	m_Invoices.updateInvoiceStubs(p_InvoiceStubs);
	}

//---------------------------------------------------------------------------

public void removeInvoiceStub (InvoiceStub p_InvoiceStub)
	{
	m_Invoices.removeInvoiceStub (p_InvoiceStub);
	}

//---------------------------------------------------------------------------

public void removeInvoiceStubs (Collection <Integer> p_InvoiceIDs)
	{
	m_Invoices.removeInvoiceStubs (p_InvoiceIDs);
	}

//---------------------------------------------------------------------------

public void propertyChange (PropertyChangeEvent p_Event) 
	{
	Integer l_Column;
	String	l_ColumnName;
	Object	l_Criterion;
	
	if (p_Event.getSource() == m_SearchHeader)
		{
		l_Column     = (Integer) p_Event.getOldValue();
		l_Criterion  = 			p_Event.getNewValue();
		l_ColumnName = InvoiceStubListModel.c_TableHeaders[l_Column];
		
		if (SearchTableHeader.c_ExclusiveCriterionChanged.equals(p_Event.getPropertyName()))
			{		
			this.firePropertyChange(c_SearchCriterionChanged, l_ColumnName,l_Criterion);
			}
		}
	}

//---------------------------------------------------------------------------

public void resetSearch ()
	{
	m_SearchHeader.resetTableHeaders();
	this.firePropertyChange(c_SearchCriterionChanged, null,null);
	}

//---------------------------------------------------------------------------

public void tableChanged(TableModelEvent p_Event) 
	{
		try {
			boolean enable = (m_CanEditStatements && (getTickedInvoiceStubs().size() > 0 || m_Tab == ReminderPanel.c_DemandsForPaymentTab));
			m_SetThirdPartyButton.setEnabled(enable);
		} catch (Exception e) {}
		
		try {
			boolean enable = (m_CanRevokeReminders && (getTickedInvoiceStubs().size() > 0 ));
			m_RevokeReminderButton.setEnabled(enable);
		} catch (Exception e) {}
	}

//---------------------------------------------------------------------------

public void actionPerformed (ActionEvent p_Event) 
	{
	if (p_Event.getSource().equals(m_PrintListButton))
		{
		this.handlePrintListButton();
		}
	else if (p_Event.getSource().equals(m_BatchPrintButton))
		{
		this.handleBatchPrintButton(true);
		}
	else if (p_Event.getSource().equals(m_PrintCopyButton))
		{
		this.handleBatchPrintButton(false);
		}
	else if (p_Event.getSource().equals(m_RevokeReminderButton))
		{
		this.handleRevokeReminderButton();
		}
	else if (p_Event.getSource().equals(m_SetThirdPartyButton))
		{
		this.handleThirdPartyButton();
		}
	}

//---------------------------------------------------------------------------

public void relocalize() 
	{
	if (m_PrintListButton != null)
		{
		m_PrintListButton.setText (Translatrix.getTranslationString(c_PrintListButton));
		m_PrintListButton.setToolTipText (Translatrix.getTranslationString(c_PrintListTip));
		}
		
	if (m_RevokeReminderButton != null)
		{
		m_RevokeReminderButton.setText (Translatrix.getTranslationString(c_RevokeReminderButton));
		m_RevokeReminderButton.setToolTipText (Translatrix.getTranslationString(c_RevokeReminderTip));
		}

	if (m_SetThirdPartyButton != null)
		{
		m_SetThirdPartyButton.setText (Translatrix.getTranslationString(c_SetThirdPartyButton));
		m_SetThirdPartyButton.setToolTipText (Translatrix.getTranslationString(c_SetThirdPartyTip));
		}

	if (m_BatchPrintButton != null)
		{
		switch (m_Tab)
			{
			case ReminderPanel.c_OpenInvoicesTab:
				
				m_BatchPrintButton.setText (Translatrix.getTranslationString		(c_PrintInvoicesButton));
				m_BatchPrintButton.setToolTipText (Translatrix.getTranslationString (c_PrintInvoicesTip));
				break;
			
			case ReminderPanel.c_UnpaidInvoicesTab:
				
				m_BatchPrintButton.setText (Translatrix.getTranslationString		(c_PrintFirstRemindersButton));
				m_BatchPrintButton.setToolTipText (Translatrix.getTranslationString (c_PrintRemindersTip));
				break;
			
			case ReminderPanel.c_FirstNoticeTab:
				
				m_BatchPrintButton.setText (Translatrix.getTranslationString		(c_PrintSecondRemindersButton));
				m_BatchPrintButton.setToolTipText (Translatrix.getTranslationString (c_PrintRemindersTip));
				break;
				
			case ReminderPanel.c_SecondNoticeTab:
				
				m_BatchPrintButton.setText (Translatrix.getTranslationString		(c_PrintPaymentOrderButton));
				m_BatchPrintButton.setToolTipText (Translatrix.getTranslationString (c_PrintRemindersTip));
				break;
			
			case ReminderPanel.c_DemandsForPaymentTab:
				m_BatchPrintButton.setText (Translatrix.getTranslationString		(c_PrintPaymentOrderButton));
				m_BatchPrintButton.setToolTipText (Translatrix.getTranslationString (c_PrintRemindersTip));
				break;
			}
		}
	
	if (m_PrintCopyButton != null)
	{
	switch (m_Tab)
		{
		case ReminderPanel.c_OpenInvoicesTab:
			
			m_PrintCopyButton.setText (Translatrix.getTranslationString		(c_PrintInvoicesButtonCopy));
//			m_PrintCopyButton.setToolTipText (Translatrix.getTranslationString (c_PrintInvoicesTip));
			break;
		
		case ReminderPanel.c_UnpaidInvoicesTab:
			
			m_PrintCopyButton.setText (Translatrix.getTranslationString		(c_PrintInvoicesButtonCopy));
//			m_PrintCopyButton.setToolTipText (Translatrix.getTranslationString (c_PrintRemindersTip));
			break;
		
		case ReminderPanel.c_FirstNoticeTab:
			
			m_PrintCopyButton.setText (Translatrix.getTranslationString		(c_PrintFirstRemindersButtonCopy));
//			m_PrintCopyButton.setToolTipText (Translatrix.getTranslationString (c_PrintRemindersTip));
			break;
			
		case ReminderPanel.c_SecondNoticeTab:
			
			m_PrintCopyButton.setText (Translatrix.getTranslationString		(c_PrintSecondRemindersButtonCopy));
//			m_PrintCopyButton.setToolTipText (Translatrix.getTranslationString (c_PrintRemindersTip));
			break;
		
		case ReminderPanel.c_DemandsForPaymentTab:
			
			m_PrintCopyButton.setText (Translatrix.getTranslationString		(c_PrintPaymentOrderButtonCopy));
//			m_PrintCopyButton.setToolTipText (Translatrix.getTranslationString (c_PrintRemindersTip));
			break;
		}
	}
	
	
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* End Of Class                                                            *
//***************************************************************************
//---------------------------------------------------------------------------
}

