/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.rate;

/*
 * InvoiceListBox.java
 *
 * Created on April 3, 2006
 */

import java.awt.Component;
import java.awt.event.KeyListener;
import java.awt.event.MouseListener;
import java.util.Collection;

import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableColumnModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Rate;
import lu.tudor.santec.gecamed.billing.gui.TextWrappingCellRenderer;
import lu.tudor.santec.gecamed.core.gui.utils.TableSorter;
import lu.tudor.santec.i18n.Relocalizable;

//***************************************************************************
//* Class Definition                                                        *
//***************************************************************************

public class RateListBox extends JScrollPane implements Relocalizable
    {
    /**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	
	private JTable           	m_RateTable;
    private RateListModel		m_Rates;
    private TableSorter			m_RateSorter;
    private RateRenderer		m_RateRenderer;
    
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************

//  ***************************************************************************
//  * Class Constants                                                         *
//  ***************************************************************************

      private static int c_CellMargin = 5;
      
      private static final int c_CodeColumnWidth 	= 10;
      private static final int c_LabelColumnWidth 	= 56;
      private static final int c_CATColumnWidth 	= 6;
      private static final int c_CACColumnWidth 	= 6;
      private static final int c_APCMColumnWidth 	= 6;
      private static final int c_ACMColumnWidth 	= 6;
      private static final int c_AmountColumnWidth = 10;
 
      static final public int c_ColumnWidths [] = { c_CodeColumnWidth,
    	  											c_LabelColumnWidth,
    	  											c_CATColumnWidth,
    	  											c_CACColumnWidth,
    	  											c_APCMColumnWidth,
    	  											c_ACMColumnWidth,
    	  											c_AmountColumnWidth   };

      
//***************************************************************************
//* Internal Class Definitions	                                          *
//***************************************************************************

//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

public RateListBox (int selectionMode)
    {	
	super ();
    
    int			l_Count;	
    TableColumn  l_Column;
    
    m_Rates 		= new RateListModel ();
    m_RateSorter = new TableSorter ();
    m_RateSorter.setTableModel(m_Rates);
    m_RateRenderer = new RateRenderer ();
    
    m_RateTable = new JTable (m_RateSorter);
    //m_RateTable.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
//    m_RateTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
    m_RateTable.setSelectionMode(selectionMode);
    
    for (l_Count=0; l_Count < m_Rates.getColumnCount(); l_Count++)
    		{
    	    if (m_Rates.getColumnClass(l_Count) != null)	
    	    	m_RateTable.setDefaultRenderer (m_Rates.getColumnClass(l_Count), m_RateRenderer);
     	}
    
    l_Column = m_RateTable.getColumnModel().getColumn(1);
    l_Column.setCellRenderer(new TextWrappingCellRenderer());
  
    m_RateTable.setColumnSelectionAllowed (false);
    m_RateTable.setShowGrid (false);
 	
    m_RateSorter.setTableHeader(m_RateTable.getTableHeader());
        
    this.setViewportView (m_RateTable);
    }

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Primitives		                                                  *
//***************************************************************************
//---------------------------------------------------------------------------

public void addListSelectionListener (ListSelectionListener p_Listener)
    {
    ListSelectionModel	l_Selection;	
    
    l_Selection = m_RateTable.getSelectionModel();
    l_Selection.addListSelectionListener (p_Listener);
    }

//---------------------------------------------------------------------------

public void addMouseListener (MouseListener p_Listener)
    {
	m_RateTable.addMouseListener (p_Listener);
    }    

//---------------------------------------------------------------------------

public void addKeyListener (KeyListener p_Listener)
    {
	m_RateTable.addKeyListener (p_Listener);
    }    

//---------------------------------------------------------------------------
//Sets the preferred width of the visible column specified by vColIndex. The column
// will be just wide enough to show the column head and the widest cell in the column.
// margin pixels are added to the left and right
// (resulting in an additional width of 2*margin pixels).
//---------------------------------------------------------------------------

public void packColumn (int p_Column, int p_Margin) 
	{
	DefaultTableColumnModel	l_ColumnModel;
	TableColumn 				l_Column;
	TableCellRenderer		l_Renderer;
	Component				l_Component;
	
	int		   				l_Width;
    int						l_Row;
	
	l_ColumnModel = (DefaultTableColumnModel)m_RateTable.getColumnModel();
	l_Column = l_ColumnModel.getColumn(p_Column);
	l_Width = 0;

    // First step consists in getting width of column header

    l_Renderer = l_Column.getHeaderRenderer();
    if (l_Renderer == null) 
    		{
    		l_Renderer = m_RateTable.getTableHeader().getDefaultRenderer();
    		}
    
    l_Component = l_Renderer.getTableCellRendererComponent(m_RateTable,
    													    l_Column.getHeaderValue(), 
    													    false, false, 0, 0);
    l_Width = l_Component.getPreferredSize().width;

    // Next we're going to iterate through all the rows of this columns, in order to
    // to find the widest one

    for (l_Row=0; l_Row < m_RateTable.getRowCount(); l_Row++) 
    		{
    		l_Renderer = m_RateTable.getCellRenderer(l_Row, p_Column);
    		l_Component = l_Renderer.getTableCellRendererComponent(m_RateTable, 
    															m_RateTable.getValueAt(l_Row, p_Column), 
    				                                               false, false, l_Row, p_Column);
        
    		l_Width = Math.max (l_Width, l_Component.getPreferredSize().width);
    		}

    // Add margin
    l_Width += 2*p_Margin;

    // Set the width
    l_Column.setPreferredWidth(l_Width);
	}

//---------------------------------------------------------------------------
//Returns the preferred height of a row.
// The result is equal to the tallest cell in the row.
//---------------------------------------------------------------------------

public void packRow (int p_Row, int p_Margin) 
	{
    int					l_RowHeight;
    int					l_ComponentHeight;
	int					l_Column;
    TableCellRenderer		l_Renderer;
    Component			l_Component;
    
    // Get the current default height for all rows
    l_RowHeight = m_RateTable.getRowHeight();

    // Determine highest cell in the row
    for (l_Column = 0; l_Column < m_RateTable.getColumnCount(); l_Column++) 
    		{
    		l_Renderer = m_RateTable.getCellRenderer(p_Row, l_Column);
    		l_Component = m_RateTable.prepareRenderer(l_Renderer, p_Row, l_Column);
    		l_ComponentHeight = l_Component.getPreferredSize().height + 2*p_Margin;
    		l_RowHeight = Math.max (l_RowHeight, l_ComponentHeight);
    		}
    
    if (m_RateTable.getRowHeight() != l_RowHeight)
    		m_RateTable.setRowHeight (p_Row,l_RowHeight);
	}

//---------------------------------------------------------------------------
/**
 * Method is part of the Relocalizable interface. The method does everything
 * required to reflect changes of active Locale
 */
//---------------------------------------------------------------------------

public void relocalize ()
    {
    // Make sure JTable re-creates columns using re-localized headers
        
    //m_InvoiceRenderer.relocalize ();
    m_RateTable.createDefaultColumnsFromModel ();
    }

//---------------------------------------------------------------------------
/**
 * doLayout is the main entry point for JTable resizing and row/column space
 * distribution. The ActListBox class overrides the method to impose its own
 * distribution scheme. Width of columns is expressed as percents and we want
 * the table to reflect this distribution. 
 */
//---------------------------------------------------------------------------

public void doLayout ()
	{
    TableColumn  l_Column;
	double		l_Width;
	int			l_Index;
	int			l_ColumnWidth;
	
	super.doLayout ();
	
	l_Width = (double) m_RateTable.getWidth () / 100;
	
	for (l_Index = 0; l_Index < c_ColumnWidths.length; l_Index++)
		{
		l_Column = m_RateTable.getColumnModel().getColumn(l_Index);
		l_ColumnWidth = (int)(l_Width * c_ColumnWidths[l_Index]);
		l_Column.setPreferredWidth(l_ColumnWidth);
		}
	}

//***************************************************************************
//* Class Body		                                                    *
//***************************************************************************

//---------------------------------------------------------------------------

public void addRate (Rate p_Rate)
    {
    if (p_Rate != null)
        {
        m_Rates.addRate (p_Rate);
        m_RateTable.repaint();
        }
    }

//---------------------------------------------------------------------------

public void removeSelectedRate ()
    {
    int []      l_SelectedRows;
    
    int         l_Selection;
    int			l_ModelRow;
    
    l_Selection = m_RateTable.getSelectedRowCount();
    
    if (l_Selection > 0)
        {
        l_SelectedRows    = m_RateTable.getSelectedRows ();
       
        for (l_Selection = l_SelectedRows.length - 1; l_Selection >= 0; l_Selection--)
            {
        	l_ModelRow = m_RateSorter.modelIndex(l_SelectedRows [l_Selection]);
        	m_Rates.removeRateAt (l_ModelRow);
            }
        
        m_RateTable.clearSelection();
        m_RateTable.repaint();
        }
    }

//---------------------------------------------------------------------------

public void removeAllRates ()
    {
	m_Rates.removeAllRates();
	m_RateTable.clearSelection();
	m_RateTable.repaint();
    }

//---------------------------------------------------------------------------

public Rate[] getSelectedRates ()
    {
	Rate []  l_SelectedRates = null;    
    int []   l_SelectedRows;   
    int      l_Selection;
    int		 l_ModelRow;
    
    l_Selection = m_RateTable.getSelectedRowCount();
    
    if (l_Selection > 0)
        {
        l_SelectedRows    = m_RateTable.getSelectedRows ();
        l_SelectedRates = new Rate [l_Selection];
        
        for (l_Selection = 0; l_Selection < l_SelectedRows.length; l_Selection++)
            {
        	l_ModelRow = m_RateSorter.modelIndex(l_SelectedRows [l_Selection]);
        	l_SelectedRates [l_Selection] = m_Rates.getRateAt (l_ModelRow);
            }
        }

    return (l_SelectedRates);
    }


public void clearSelection ()
	{
	m_RateTable.clearSelection();
	}

//---------------------------------------------------------------------------

public void selectFirstRate ()
	{
    ListSelectionModel	l_Selection;	
    JScrollBar			l_ScrollBar;
    	
	if (getRateCount() > 0)
		{
   		l_Selection = m_RateTable.getSelectionModel();
		l_Selection.setSelectionInterval (0,0);
	
		l_ScrollBar = getVerticalScrollBar ();
		l_ScrollBar.setValue (0);
		}	
	}

//---------------------------------------------------------------------------

public void requestFocus ()
	{
	m_RateTable.requestFocus();
	}

//---------------------------------------------------------------------------

public int getRateCount ()
	{
	return m_Rates.getRowCount();
	}

//---------------------------------------------------------------------------

public void setRates (Collection <Rate> p_Rates)
    {
    m_Rates.setRates(p_Rates);
    }    
 
//---------------------------------------------------------------------------

public Collection <Rate> getRates ()
    {
    return m_Rates.getRates();
    }    
    
//---------------------------------------------------------------------------

public void packColumns ()
	{
	int l_Column;
	
	for (l_Column=0; l_Column < m_RateTable.getColumnCount(); l_Column++)
		{
		packColumn (l_Column,c_CellMargin);
		}
	}

//---------------------------------------------------------------------------

public void packRows ()
	{
	int l_Row;
	
	for (l_Row=0; l_Row < m_RateTable.getRowCount(); l_Row++)
		{
		packRow (l_Row,c_CellMargin);
		}
	}

//***************************************************************************
//* End of Class															*
//***************************************************************************
}
