/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.hospitalisation;

import java.awt.Color;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.UIManager;
import javax.swing.border.TitledBorder;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Invoice;
import lu.tudor.santec.gecamed.billing.gui.act.HospitalisationClassListener;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.utils.DefaultComboBoxRenderer;
import lu.tudor.santec.gecamed.patient.ejb.entity.beans.HospitalisationClass;
import lu.tudor.santec.gecamed.patient.gui.HospitalisationClassComboBoxRenderer;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

public class HospitalisationPanel extends JPanel implements ItemListener,
															Relocalizable
	{
	private static final long serialVersionUID = 1L;
	
	private static HospitalisationPanel m_Instance;
	private static HospitalisationClass m_HospitalisationClass2;
	private static HospitalisationClass m_HospitalisationClassAmbulant;
	
	private TitledBorder			m_Border;
	
	private JLabel					m_HospitalisationClassLabel;

	@SuppressWarnings("rawtypes")
	private JComboBox  				m_HospitalisationClass;
	private JTextField				m_HospitalisationClassField;
	
	private JCheckBox				m_FirstClassRequired;
	
	private Invoice 				m_CurrentInvoice;
	
	private LinkedList<HospitalisationClassListener> m_HospitalisationClassListener = new LinkedList<HospitalisationClassListener>();

	private HospitalisationClass m_Class;

	private boolean editableClass;

	private boolean m_isInEditMode;

	
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************

	public final static String	c_FirstClassRequired 	= "HospitalisationPanel.FirstClassRequired";
	public final static String	c_HospitalisationClass  = "HospitalisationPanel.HospitalisationClass";
	
	private final static String c_Columns= 	"fill:max(60dlu;pref),3dlu," + 
											"fill:max(60dlu;pref)";
	
	private final static String c_Rows=    "fill:pref,3dlu,fill:pref";
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************

public HospitalisationPanel ()
    {
	m_Instance = this;
	
	CellConstraints		l_Constraints;
    FormLayout			l_Layout;
    Color				l_DefaultColor; 
    
    l_Constraints  = new CellConstraints();
    
    l_Layout = new FormLayout(c_Columns, c_Rows);
    this.setLayout(l_Layout);
    this.setOpaque(false);
    
	m_Border = new TitledBorder (Translatrix.getTranslationString("HospitalisationPanel.Title"));
    this.setBorder (m_Border);
     
    m_HospitalisationClassLabel	= new JLabel ();
    m_HospitalisationClassLabel.setForeground (GECAMedColors.c_LabelColor);
    m_HospitalisationClassLabel.setText (Translatrix.getTranslationString("HospitalisationPanel.HospitalisationClassLabel"));
    
    l_DefaultColor = (Color) UIManager.getDefaults().get("TextField.inactiveBackground");   
    UIManager.getDefaults().put("TextField.inactiveBackground", Color.WHITE );
    
    m_HospitalisationClass 		= new JComboBox ();
    m_HospitalisationClass.setRenderer(new HospitalisationClassComboBoxRenderer(true,true));
    m_HospitalisationClass.addItemListener(this);
    m_HospitalisationClassField	= new JTextField ();
    m_HospitalisationClassField.setEditable(false);
    
    m_FirstClassRequired = new JCheckBox (Translatrix.getTranslationString("HospitalisationPanel.FirstClassRequired"));
    m_FirstClassRequired.setOpaque(false);
    m_FirstClassRequired.setEnabled (false);
    m_FirstClassRequired.setSelected(false);
    m_FirstClassRequired.addItemListener(this);
    
    UIManager.getDefaults().put("TextField.inactiveBackground", l_DefaultColor );
    
    this.relocalize ();
    
    this.add (m_HospitalisationClassLabel, l_Constraints.xywh(1, 1, 1, 1));
    this.add (m_HospitalisationClass, 	   l_Constraints.xywh(1, 3, 1, 1)); 
    this.add (m_FirstClassRequired, 	   l_Constraints.xywh(3, 3, 1, 1)); 
    }

//***************************************************************************
//* Class Primitives                                                        *
//***************************************************************************

//---------------------------------------------------------------------------
/**
 * Method is part of the Relocalizable interface. The method does everything
 * required to reflect changes of active Locale
 */
//---------------------------------------------------------------------------

public void relocalize() 
	{	  	
	if (m_Border != null) 
		m_Border.setTitle (Translatrix.getTranslationString("HospitalisationPanel.Title"));

	if (m_HospitalisationClassLabel != null)
		m_HospitalisationClassLabel.setText (Translatrix.getTranslationString("HospitalisationPanel.HospitalisationClassLabel"));
		
	if (m_FirstClassRequired != null)
		m_FirstClassRequired.setText (Translatrix.getTranslationString("HospitalisationPanel.FirstClassRequired"));
	}

//---------------------------------------------------------------------------

public void populateHospitalisationClasses (Collection <HospitalisationClass> p_Classes)
	{
	Iterator <HospitalisationClass> l_ClassIterator;
	HospitalisationClass			l_Class;

	if (p_Classes == null) return;
	
	m_HospitalisationClass.removeAllItems();
	
	l_ClassIterator = p_Classes.iterator();
	while (l_ClassIterator.hasNext())
		{
		l_Class = l_ClassIterator.next();
		if (l_Class.getAcronym().length() == 1) {
			m_HospitalisationClass.addItem (l_Class);			
		}
		if (HospitalisationClass.c_SecondClass.equals(l_Class.getAcronym())) {
			m_HospitalisationClass2 = l_Class;
		} else if (HospitalisationClass.c_Ambulant.equals(l_Class.getAcronym())) {
			m_HospitalisationClassAmbulant = l_Class;
		}
	}
	}

//---------------------------------------------------------------------------

private void swapComponents (JComponent p_Out, JComponent p_In, CellConstraints p_Constraints)
	{
	if (p_Out.getParent() == this)
		{
		this.remove (p_Out);
		this.add( p_In,p_Constraints);
		//p_In.repaint();
		}
	}

//---------------------------------------------------------------------------

private void setCheckBoxEnabled ()
	{
	HospitalisationClass	l_Class;

	l_Class = (HospitalisationClass) m_HospitalisationClass.getSelectedItem();
	if (HospitalisationClass.c_FirstClass.equals(l_Class.getAcronym()))
		m_FirstClassRequired.setEnabled (true);
	else 
		{
		m_FirstClassRequired.setSelected(false);
		m_FirstClassRequired.setEnabled (false);
		}
	}

//---------------------------------------------------------------------------

public void itemStateChanged (ItemEvent p_ChangeEvent) 
	{
	if (p_ChangeEvent.getSource() == m_HospitalisationClass)
		{
		this.setCheckBoxEnabled();
		if (ItemEvent.SELECTED == p_ChangeEvent.getStateChange()) {
			this.fireHospitalisationClassChanged();			
		}
		this.firePropertyChange(c_HospitalisationClass, null, p_ChangeEvent.getItem());
		return;
		}
	
	if (p_ChangeEvent.getSource() == m_FirstClassRequired)
		{
		if (p_ChangeEvent.getStateChange() == ItemEvent.DESELECTED)
			{
			this.firePropertyChange(c_FirstClassRequired, true, false);
			}
		else if (p_ChangeEvent.getStateChange() == ItemEvent.SELECTED)
			{
			this.firePropertyChange(c_FirstClassRequired, false, true);
			}
		this.fireHospitalisationClassChanged();
		}
	}

//***************************************************************************
//* Class Body                                                              *
//***************************************************************************

//---------------------------------------------------------------------------

public void setEnabled (boolean p_EnableIt)
	{
	CellConstraints		l_Constraints;
	
	l_Constraints  = new CellConstraints();
	
	m_isInEditMode = p_EnableIt;
	
    if (p_EnableIt && editableClass)
		{
		this.setCheckBoxEnabled();
    	this.swapComponents(m_HospitalisationClassField, m_HospitalisationClass, l_Constraints.xywh(1, 3, 1, 1));
		}
	else
		{
		m_FirstClassRequired.setEnabled (false);
		this.swapComponents(m_HospitalisationClass, m_HospitalisationClassField, l_Constraints.xywh(1, 3, 1, 1));
		}
	this.repaint();
	}


//---------------------------------------------------------------------------

public void setInvoice (Invoice p_Invoice)
	{
	if (p_Invoice == null) return;
	
	m_CurrentInvoice = p_Invoice;
	
	this.setHospitalisationClass(p_Invoice.getHospitalisationClass());
	
	m_FirstClassRequired.setSelected (p_Invoice.getFirstClassRequired());
	
	}

//---------------------------------------------------------------------------

public boolean updateInvoice (Invoice p_Invoice)
	{
	boolean l_InvoiceModified = false;
	
	if (p_Invoice == null) return false;
	
	if (p_Invoice.getFirstClassRequired() != m_FirstClassRequired.isSelected())
		{
		p_Invoice.setFirstClassRequired(m_FirstClassRequired.isSelected());
		p_Invoice.updateFirstClassRequired();
		l_InvoiceModified = true;
		}

	if (!this.getHospitalisationClass().equals (p_Invoice.getHospitalisationClass()))
		{
		p_Invoice.setHospitalisationClass (this.getHospitalisationClass());
		l_InvoiceModified = true;
		}

	return l_InvoiceModified;
	}

//---------------------------------------------------------------------------

public void reset ()
	{
	this.setFirstClassRequired(false);
	this.resetHospitalisationClass();
	}

//---------------------------------------------------------------------------

public boolean getFirstClassRequired ()
	{
	return m_FirstClassRequired.isSelected();
	}

//---------------------------------------------------------------------------

public void setFirstClassRequired (boolean p_IsRequired)
	{
	m_FirstClassRequired.setSelected(p_IsRequired);
	}

//---------------------------------------------------------------------------
/**
 * Returns the currently selected hospitalisation class
 */
//---------------------------------------------------------------------------

public HospitalisationClass getHospitalisationClass ()
	{
	HospitalisationClass	l_Class;
	
	if (editableClass) {
		l_Class = (HospitalisationClass) m_HospitalisationClass.getSelectedItem();		
	} else {
		l_Class = m_Class;
	}
	return l_Class;
	}

//---------------------------------------------------------------------------

public void setHospitalisationClass (HospitalisationClass p_Class)
	{
	DefaultComboBoxRenderer	l_Renderer;
	
	m_Class = p_Class;		
	if (p_Class != null && !m_HospitalisationClass.equals(p_Class)) 
		{
		if (p_Class.getAcronym().length() != 1) {
			editableClass = false;
		} else {
			editableClass = true;
		}
		
		m_HospitalisationClass.setSelectedItem(p_Class);
		l_Renderer = (DefaultComboBoxRenderer) m_HospitalisationClass.getRenderer();
		m_HospitalisationClassField.setText(l_Renderer.renderObject(p_Class));
		fireHospitalisationClassChanged();
		}
	}

//---------------------------------------------------------------------------

public void resetHospitalisationClass ()
	{
	DefaultComboBoxRenderer	l_Renderer;
	HospitalisationClass	l_Class;
	
	if (m_HospitalisationClass.getItemCount() > 0) 
		{
		m_HospitalisationClass.setSelectedIndex(0);
		l_Renderer = (DefaultComboBoxRenderer) m_HospitalisationClass.getRenderer();
		l_Class    = (HospitalisationClass) m_HospitalisationClass.getSelectedItem();
		m_HospitalisationClassField.setText(l_Renderer.renderObject(l_Class));
		}
	}

//---------------------------------------------------------------------------

public void addHospitalisationClassListener (HospitalisationClassListener p_Listener)
	{
	m_HospitalisationClassListener.add(p_Listener);
	}

//---------------------------------------------------------------------------

public void removeHospitalisationClassListener (HospitalisationClassListener p_Listener)
	{
	m_HospitalisationClassListener.remove(p_Listener);
	}

//---------------------------------------------------------------------------

private void fireHospitalisationClassChanged ()
	{
	if (m_CurrentInvoice != null && editableClass && m_isInEditMode)
		{
		m_CurrentInvoice.setHospitalisationClass((HospitalisationClass)m_HospitalisationClass.getSelectedItem());
		m_CurrentInvoice.setFirstClassRequired(m_FirstClassRequired.isSelected() ? Boolean.TRUE : Boolean.FALSE);
		for (HospitalisationClassListener l_Listener : m_HospitalisationClassListener)
			{
			l_Listener.hospitalisationClassChanged(m_CurrentInvoice);
			}
		}
	}

//---------------------------------------------------------------------------

public static void changeToHospitalisation ()
{
	String currentAcronym = m_Instance.getHospitalisationClass().getAcronym();
	
	if (HospitalisationClass.c_Ambulant.equals(currentAcronym))
		{
		m_Instance.setHospitalisationClass(m_HospitalisationClass2);
		}
}

public HospitalisationClass getHospitalisationClassForNewInvoice() {
	HospitalisationClass	l_Class;
	
	if (editableClass) {
		l_Class = (HospitalisationClass) m_HospitalisationClass.getSelectedItem();		
	} else {
		l_Class = m_HospitalisationClassAmbulant;
	}
	return l_Class;
}

//---------------------------------------------------------------------------
//***************************************************************************
//* End of Class															*
//***************************************************************************
}
