/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.event.invoice;

import java.io.Serializable;
import java.util.Collection;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Invoice;

//***************************************************************************
//* Class Definition		                                            	*
//***************************************************************************

/**
 * The InvoiceChangeEvent class is used to notify InvoiceListeners about
 * invoice related changes. 
 * @author nico.mack@tudor.lu
 */

public class InvoiceChangeEvent implements Serializable 
	{
	/**
	 * 
	 */
	private	Object					m_Source;
	private String					m_Originator;
	private int						m_Type;
	private Invoice					m_Invoice;
	private Collection <Integer>	m_InvoiceIDs;	
	
//***************************************************************************
//* Constants	                                                            *
//***************************************************************************

	private static final long serialVersionUID = 1L;

	public static final int c_None			   			= 0;
	public static final int c_InvoiceSelectionChanged 	= 1;
	public static final int c_InvoiceStatusChanged		= 2;
	public static final int c_InvoiceUpdated 			= 3;
	public static final int c_InvoiceDeleted 			= 4;
	
	public static final String ORIGIN_REMOTE	= "REMOTE";
	
	public static final int	c_MaxTypes = 5;
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * Creates a new empty instance of InvoiceChangeEvent
 */
//---------------------------------------------------------------------------

public 	InvoiceChangeEvent ()
	{
	m_Source   		= null;
	m_Originator    = null;
	m_Type	   		= c_None;
	m_Invoice 		= null;
	m_InvoiceIDs 	= null;
	}
	
//---------------------------------------------------------------------------
/**
 * Creates a new instance of InvoiceChangeEvent initialized with the
 * specified properties.
 * @param p_Source specifies the object that generated to event.
 * @param p_Type specifies the type of event. Possible values are:
 * <ul>
 * <li>c_None : Nothing happened. (Hardly every used)</li>
 * <li>c_InvoiceSelectionChanged : Another invoice has been selected</li>
 * <li>c_SingleInvoiceUpdated : A single invoice has been updated</li>
  * </ul>
 * @param p_Invoice specifies the invoice this event refers to.
 */
//---------------------------------------------------------------------------

public InvoiceChangeEvent (Object p_Source, int p_Type, Invoice p_Invoice)
	{
	this.setSource(p_Source);
	this.setOriginator (null);
	this.setType(p_Type);
	this.setInvoice(p_Invoice);
	}

//---------------------------------------------------------------------------
/**
 * Creates a new instance of InvoiceChangeEvent initialized with the
 * specified properties.
 * @param p_Source specifies the object that generated to event.
 * @param p_Type specifies the type of event. Possible values are:
 * <ul>
 * <li>c_None : Nothing happened. (Hardly every used)</li>
 * <li>c_MultipleInvoiceUpdated : More than one invoice have been updated</li>
 * </ul>
 * @param p_InvoiceIDs specifies the IDs of invoice that have been updated.
 */
//---------------------------------------------------------------------------

public InvoiceChangeEvent (Object p_Source, int p_Type, Collection <Integer> p_InvoiceIDs)
	{
	this.setSource(p_Source);
	this.setOriginator (null);
	this.setType(p_Type);
	this.setInvoiceIDs(p_InvoiceIDs);
	}

//---------------------------------------------------------------------------
		
//***************************************************************************
//* Class Body	                                                            *
//***************************************************************************

//---------------------------------------------------------------------------

public Object getSource ()
	{
	return m_Source;
	}

//---------------------------------------------------------------------------

public void setSource (Object p_Source)
	{
	m_Source = p_Source;
	}

//---------------------------------------------------------------------------

public String getOriginator ()
	{
	return m_Originator;
	}

//---------------------------------------------------------------------------

public void setOriginator (String p_Originator)
	{
	m_Originator = p_Originator;
	}

//---------------------------------------------------------------------------

public int getType ()
	{
	return m_Type;
	}

//---------------------------------------------------------------------------

public void setType (int p_Type)
	{
	if ((p_Type >= c_None) && (p_Type < c_MaxTypes))
		{
		m_Type = p_Type;
		}
	else m_Type = c_None;
	}

//---------------------------------------------------------------------------

public Invoice getInvoice ()
	{
	return m_Invoice;
	}

//---------------------------------------------------------------------------

public void setInvoice (Invoice p_Invoice)
	{
	m_Invoice = p_Invoice;
	}

//---------------------------------------------------------------------------

public Collection <Integer> getInvoiceIDs ()
	{
	return m_InvoiceIDs;
	}

//---------------------------------------------------------------------------

public void setInvoiceIDs (Collection <Integer> p_InvoiceIDs) 
	{
	m_InvoiceIDs = p_InvoiceIDs;
	}

//---------------------------------------------------------------------------

public boolean isMultiple ()
	{
	return ((m_InvoiceIDs != null) && (m_InvoiceIDs.size() > 0));
	}

public boolean isRemote() {
	return (getOriginator() != null && ORIGIN_REMOTE.equals(getOriginator()));
}

//---------------------------------------------------------------------------
//***************************************************************************
//* End of Class															*
//***************************************************************************
//---------------------------------------------------------------------------
}

