/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.gui.admin;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Collection;
import java.util.Date;

import javax.swing.AbstractButton;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SwingConstants;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.table.TableColumn;
import javax.swing.tree.TreePath;

import lu.tudor.santec.gecamed.billing.ejb.entity.beans.KeyValue;
import lu.tudor.santec.gecamed.billing.ejb.entity.beans.Rate;
import lu.tudor.santec.gecamed.billing.ejb.session.beans.NomenclatureBean;
import lu.tudor.santec.gecamed.billing.ejb.session.interfaces.NomenclatureInterface;
import lu.tudor.santec.gecamed.billing.gui.BillingModule;
import lu.tudor.santec.gecamed.billing.gui.event.keyvalue.KeyValueChangeEvent;
import lu.tudor.santec.gecamed.billing.gui.event.keyvalue.KeyValueEventDispatcher;
import lu.tudor.santec.gecamed.billing.gui.event.keyvalue.KeyValueEventSource;
import lu.tudor.santec.gecamed.billing.gui.event.keyvalue.KeyValueListener;
import lu.tudor.santec.gecamed.core.gui.GECAMedColors;
import lu.tudor.santec.gecamed.core.gui.GECAMedIconNames;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.MainFrame;
import lu.tudor.santec.gecamed.core.utils.GECAMedUtils;
import lu.tudor.santec.gecamed.core.utils.ManagerFactory;
import lu.tudor.santec.gecamed.core.utils.querybuilder.HibernateCondition;
import lu.tudor.santec.gecamed.core.utils.querybuilder.HibernateOperator;
import lu.tudor.santec.gecamed.core.utils.querybuilder.WhereClause;
import lu.tudor.santec.i18n.Relocalizable;
import lu.tudor.santec.i18n.Translatrix;

import org.apache.log4j.Logger;
import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.decorator.HighlighterFactory;

import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

public class KeyValuePanel extends JPanel implements KeyValueEventSource,
													 ActionListener,
													 Relocalizable, 
													 TreeSelectionListener
	{

	private JButton					m_AddKeyValueButton;
	private JButton					m_RemoveKeyValueButton;
	
	private KeyValueDialog			m_KeyValueDialog;
	
	private NomenclatureInterface	m_NomenclatureInterface;

	private KeyValueEventDispatcher	m_KeyValueListeners;
	
	private KeyValueTreeTableModel 	m_KeyValueModel;
	private JXTreeTable 			m_KeyValueTable;
	private KeyValueTreeNode 		m_RootNode;
	private KeyValueRenderer m_KeyValueRenderer;
	private JButton m_ReloadButton;
	
	private static Logger m_Logger = Logger.getLogger (KeyValuePanel.class.getName());

//---------------------------------------------------------------------------
//***************************************************************************
//* Constants	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------
	
	private static final long serialVersionUID = 1L;

	private static final String c_AddKeyValueButton    	= "KeyValuePanel.AddKeyValueButton";
	private static final String c_RemoveKeyValueButton 	= "KeyValuePanel.RemoveKeyValueButton";

	private static final String c_AddKeyValueTip    	= "KeyValuePanel.AddKeyValueTip";
	private static final String c_RemoveKeyValueTip 	= "KeyValuePanel.RemoveKeyValueTip";
	
	private static final String	c_ReferringRatesTitle	= "KeyValuePanel.ReferringRatesTitle";
	private static final String	c_ReferringRatesMessage	= "KeyValuePanel.ReferringRatesMessage";
	
	private static final String	c_DeleteKeyvalTitle		= "KeyValuePanel.DeleteKeyvalTitle";
	private static final String	c_DeleteKeyvalMessage	= "KeyValuePanel.DeleteKeyvalMessage";
	
	private final static String c_Columns= 	"3dlu,fill:pref:grow,3dlu,fill:max(50dlu;pref)";
	
	private final static String c_Rows=    	"3dlu,fill:max(10dlu;pref), 3dlu,fill:max(10dlu;pref)," +
											"3dlu,fill:max(10dlu;pref),fill:pref:grow";
	
    static final public int c_ColumnWidths [] = {55,  10, 30, 25};
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Constructor	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

public KeyValuePanel ()
	{
	m_KeyValueListeners = new KeyValueEventDispatcher ();
		
	this.buildPanel();
	}
	
//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

private void buildPanel ()
	{
	CellConstraints	l_Constraints;
	FormLayout		l_Layout;	
	MouseAdapter	l_MouseAdapter;
	ImageIcon		l_Icon;
	
	l_Constraints  	= new CellConstraints();
	l_Layout		= new FormLayout(c_Columns, c_Rows);

	this.setLayout(l_Layout);
	
	this.setOpaque(false);

	m_KeyValueModel = new KeyValueTreeTableModel();
	m_KeyValueTable = new JXTreeTable(m_KeyValueModel);
	m_KeyValueRenderer = new KeyValueRenderer (m_KeyValueTable);
	
	for (int l_Count = 0; l_Count < m_KeyValueModel.getColumnCount(); l_Count++) {
		if (m_KeyValueModel.getColumnClass(l_Count) != null)
			m_KeyValueTable.setDefaultRenderer(m_KeyValueModel.getColumnClass(l_Count), m_KeyValueRenderer);
	}
	
	m_KeyValueTable.setHighlighters(HighlighterFactory.createAlternateStriping(
    							 GECAMedColors.c_EvenLineBackground, 
    							 GECAMedColors.c_OddLineBackground));
		
	m_KeyValueTable.addTreeSelectionListener (this);

	l_MouseAdapter = new MouseAdapter()
		{
    	public void mouseClicked(MouseEvent p_Event)
    		{
    		if (p_Event.isPopupTrigger() || (p_Event.getButton() == MouseEvent.BUTTON3))
    			{
     			}
    		else if (p_Event.getClickCount() == 2)
    			{
    			editSelectedKeyValue (); 
    			}
    		}
    	};

    m_KeyValueTable.addMouseListener(l_MouseAdapter);
	
	
	l_Icon = BillingModule.getButtonIcon ("add_act.png");
	m_AddKeyValueButton = new JButton (Translatrix.getTranslationString(c_AddKeyValueButton),l_Icon);
	m_AddKeyValueButton.setToolTipText (Translatrix.getTranslationString(c_AddKeyValueTip));
	m_AddKeyValueButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_AddKeyValueButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_AddKeyValueButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
	m_AddKeyValueButton.setEnabled(true);
	m_AddKeyValueButton.addActionListener(this);
	
	l_Icon = BillingModule.getButtonIcon ("remove_act.png");
	m_RemoveKeyValueButton = new JButton (Translatrix.getTranslationString(c_RemoveKeyValueButton),l_Icon);
	m_RemoveKeyValueButton.setToolTipText (Translatrix.getTranslationString(c_RemoveKeyValueTip));
	m_RemoveKeyValueButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_RemoveKeyValueButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_RemoveKeyValueButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
	m_RemoveKeyValueButton.setEnabled(true);
	m_RemoveKeyValueButton.addActionListener (this);
	
	l_Icon = GECAMedModule.getIcon(GECAMedIconNames.RELOAD);
	m_ReloadButton = new JButton (l_Icon);
	m_ReloadButton.setHorizontalAlignment(SwingConstants.LEFT);
	m_ReloadButton.setVerticalTextPosition(AbstractButton.CENTER);
	m_ReloadButton.setHorizontalTextPosition(AbstractButton.TRAILING); 
	m_ReloadButton.setEnabled(true);
	m_ReloadButton.addActionListener (this);
	
	this.relocalize();
	
	//Disable buttons if demo mode
	if (GECAMedUtils.isDemo() && !MainFrame.isAdmin()) {
	   	// disables all tabs
		m_AddKeyValueButton.setEnabled(false);
		m_RemoveKeyValueButton.setEnabled(false);
		m_ReloadButton.setEnabled(false);
		
	} else {
		m_AddKeyValueButton.setEnabled(true);
		m_RemoveKeyValueButton.setEnabled(true);
		m_ReloadButton.setEnabled(true);	
	}
	
	JScrollPane l_ScrollPane = new JScrollPane(m_KeyValueTable);
	
	this.add (l_ScrollPane, 	  				  l_Constraints.xywh(2, 2, 1, 6));
	this.add (m_AddKeyValueButton, 	  l_Constraints.xywh(4, 2, 1, 1));
	this.add (m_RemoveKeyValueButton, l_Constraints.xywh(4, 4, 1, 1));	
	this.add (m_ReloadButton, 	  	  l_Constraints.xywh(4, 6, 1, 1));
	
	reloadTreeTable();
	}

//---------------------------------------------------------------------------


	private KeyValueTreeNode buildTree() {

		m_RootNode = new KeyValueTreeNode();
		try {
			Collection<KeyValue> keyValues = getNomenclatureInterface().getAllKeyValues();
			KeyValueTreeNode childNode = null;
			for (KeyValue keyValue : keyValues) {

				if (childNode == null || !keyValue.getKeyType().equals(childNode.getKeyValue().getKeyType())) {
					childNode = new KeyValueTreeNode();
					childNode.setKeyValue(keyValue);
					m_RootNode.add(childNode);
				} else {
					KeyValueTreeNode leafNode = new KeyValueTreeNode();
					leafNode.setKeyValue(keyValue);
					childNode.add(leafNode);
				}

			}
		} catch (Exception e) {
			m_Logger.error("Error loading KeyValue View");
		}
		return m_RootNode;
	}

	private KeyValueTreeNode getSelectedNode ()
	{
	KeyValueTreeNode		l_Node;
	TreePath			l_SelectionPath;
	int		 			l_SelectedRow;

	l_SelectedRow = m_KeyValueTable.getSelectedRow();
	if (l_SelectedRow < 0) return null;
		
	l_SelectionPath = m_KeyValueTable.getPathForRow (l_SelectedRow);
	l_Node = (KeyValueTreeNode) l_SelectionPath.getLastPathComponent();
		
	return l_Node;
	}
	
	
//---------------------------------------------------------------------------


private KeyValueDialog	getKeyValueDialog ()
	{
	if (m_KeyValueDialog == null) m_KeyValueDialog = new KeyValueDialog ();
	
	
	return m_KeyValueDialog;
	}

//---------------------------------------------------------------------------

private NomenclatureInterface getNomenclatureInterface ()
	{
	if (m_NomenclatureInterface != null) return m_NomenclatureInterface;

	try {
		m_NomenclatureInterface = (NomenclatureInterface) ManagerFactory.getRemote(NomenclatureBean.class);
		} 
	catch (Exception p_Exception) 
		{
		m_Logger.warn(p_Exception.getLocalizedMessage());
		}

	return m_NomenclatureInterface;
	}


//---------------------------------------------------------------------------

private long getRateCountByKeyValue (KeyValue p_KeyValue)
	{
	NomenclatureInterface	l_NomenclatureInterface;
	long					l_RateCount = 0L;
	
	if (p_KeyValue == null) return l_RateCount;
	
	l_NomenclatureInterface = this.getNomenclatureInterface();
	if (l_NomenclatureInterface == null) return l_RateCount;
	
	try {
		l_RateCount = l_NomenclatureInterface.getRateCountByKeyValue(p_KeyValue);
		} 
	catch (Exception p_Exception) 
		{
		m_Logger.warn(p_Exception.getLocalizedMessage());
		}
	
	return 	l_RateCount;	
	}

//---------------------------------------------------------------------------

private KeyValue saveKeyValue (KeyValue p_KeyValue)
	{
	NomenclatureInterface	l_NomenclatureInterface;
	
	if (p_KeyValue == null) return p_KeyValue;
	
	l_NomenclatureInterface = this.getNomenclatureInterface();
	if (l_NomenclatureInterface == null) return null;
	
	try {
		p_KeyValue = l_NomenclatureInterface.saveKeyValue(p_KeyValue);
		} 
	catch (Exception p_Exception) 
		{
		m_Logger.warn(p_Exception.getLocalizedMessage());
		}
	
	return 	p_KeyValue;		
	}

//---------------------------------------------------------------------------

private void deleteKeyValue (KeyValue p_KeyValue)
	{
	NomenclatureInterface	l_NomenclatureInterface;
	
	if (p_KeyValue == null) return;
	
	l_NomenclatureInterface = this.getNomenclatureInterface();
	if (l_NomenclatureInterface == null) return;
	
	try {
		l_NomenclatureInterface.deleteKeyValue(p_KeyValue);
		} 
	catch (Exception p_Exception) 
		{
		m_Logger.warn(p_Exception.getLocalizedMessage());
		}
	}


//---------------------------------------------------------------------------

private void deleteRates (KeyValue p_KeyValue)
	{
	NomenclatureInterface	l_NomenclatureInterface;
	
	if (p_KeyValue == null) return;
	
	l_NomenclatureInterface = this.getNomenclatureInterface();
	if (l_NomenclatureInterface == null) return;
	
	try {
		WhereClause c = new WhereClause();
		HibernateCondition con = new HibernateCondition("keyType", HibernateOperator.c_EqualOperator, p_KeyValue.getKeyType());
		c.addCondition(con);
		Collection<Rate> rates = l_NomenclatureInterface.getRatesByWhereClause(c);
		for (Rate rate : rates) {			
			l_NomenclatureInterface.deleteRate(rate);
		}
		} 
	catch (Exception p_Exception) 
		{
		m_Logger.warn(p_Exception.getLocalizedMessage());
		}
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body	                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

public void addKeyValueListener (KeyValueListener p_Listener)
	{
	m_KeyValueListeners.addKeyValueListenerListener(p_Listener);
	}

//---------------------------------------------------------------------------

public void removeKeyValueListener (KeyValueListener p_Listener)
	{
	m_KeyValueListeners.removeKeyValueListenerListener(p_Listener);
	}
	
//---------------------------------------------------------------------------

public void addNewKeyValue ()
	{
	KeyValueDialog		l_KeyValueDialog;
	KeyValue 			l_KeyValue;
	KeyValueChangeEvent	l_Event;
//	int					l_DialogOption;
//	
//	// Warn the user not to add a key, if an existing key changed
//	l_DialogOption	= GECAMedBaseDialogImpl.showMessageDialog(
//			MainFrame.getInstance(), 
//			Translatrix.getTranslationString("KeyValueDialog.WarnBeforeAddingKey_title"),
//			Translatrix.getTranslationString("KeyValueDialog.WarnBeforeAddingKey_message"), 
//			GECAMedBaseDialogImpl.YES_NO_BUTTON_MODE, 
//			GECAMedModule.getIcon(GECAMedIconNames.WARNING));
//	
//	if (l_DialogOption != GECAMedBaseDialogImpl.YES_OPTION)
//		return;
	
	l_KeyValue = new KeyValue ();
	l_KeyValue.setProtected(false);
	l_KeyValue.setEditable(true);
	
	KeyValueTreeNode l_KeyValueTreeNode = getSelectedNode();
	if (l_KeyValueTreeNode != null ) {
		KeyValue l_SelectedKeyValue = l_KeyValueTreeNode.getKeyValue();
		if (l_SelectedKeyValue != null) {
			l_KeyValue.setKeyType(		l_SelectedKeyValue.getKeyType());
			l_KeyValue.setLabel(		l_SelectedKeyValue.getLabel());
			l_KeyValue.setDescription(	l_SelectedKeyValue.getDescription());
			l_KeyValue.setValue(		l_SelectedKeyValue.getValue());
			l_KeyValue.setApplicability(new Date());
			l_KeyValue.setProtected(	l_SelectedKeyValue.getProtected());
			l_KeyValue.setEditable(		l_SelectedKeyValue.getEditable());
			l_KeyValue.setFractionDigits(l_SelectedKeyValue.getFractionDigits());
		}
	}
		
	l_KeyValueDialog = this.getKeyValueDialog();
	try {
		l_KeyValueDialog.setKeyValues(getNomenclatureInterface().getNewestKeyValues());		
	} catch (Exception e) {
		m_Logger.warn("Error fetching KeyValues!", e);
	}
	l_KeyValueDialog.setKeyValue(l_KeyValue);
	l_KeyValueDialog.pack();
	MainFrame.showDialogCentered (l_KeyValueDialog);	
	
	if (!l_KeyValueDialog.wasCanceled())
		{
		l_KeyValue = l_KeyValueDialog.getKeyValue();
		l_KeyValue = this.saveKeyValue(l_KeyValue);

		reloadTreeTable();
		
		l_Event = new KeyValueChangeEvent (this,l_KeyValue,KeyValueChangeEvent.c_KeyValueCreated);
		m_KeyValueListeners.notifyKeyValueListeners(l_Event);
		}
	}

//---------------------------------------------------------------------------

public void editSelectedKeyValue ()
	{
	KeyValueDialog		l_KeyValueDialog;
	KeyValue 			l_KeyValue;
	KeyValueChangeEvent	l_Event;

	KeyValueTreeNode l_KeyValueTreeNode = getSelectedNode();
	
	if (l_KeyValueTreeNode == null ) return;
	
	l_KeyValue = l_KeyValueTreeNode.getKeyValue();
	
	l_KeyValueDialog = this.getKeyValueDialog();
	try {
		l_KeyValueDialog.setKeyValues(getNomenclatureInterface().getNewestKeyValues());		
	} catch (Exception e) {
		m_Logger.warn("Error fetching KeyValues!", e);
	}
	l_KeyValueDialog.setKeyValue(l_KeyValue);
	l_KeyValueDialog.pack();
	MainFrame.showDialogCentered (l_KeyValueDialog);	
	
	if (!l_KeyValueDialog.wasCanceled())
		{
		l_KeyValue = l_KeyValueDialog.getKeyValue();
		l_KeyValue = this.saveKeyValue(l_KeyValue);
		
		reloadTreeTable();
		
		l_Event = new KeyValueChangeEvent (this,l_KeyValue,KeyValueChangeEvent.c_KeyValueUpdated);
		m_KeyValueListeners.notifyKeyValueListeners(l_Event);
		}
	}

//---------------------------------------------------------------------------

public void removeSelectedKeyValue ()
	{
	KeyValue 			l_KeyValue;
	long				l_RateCount;
	String[]			l_Filler;
	int					l_UserChoice;
	KeyValueChangeEvent	l_Event;
	boolean				l_DeleteIt = false;
	
	
	l_KeyValue = getSelectedNode().getKeyValue();
	
	if (l_KeyValue == null ) return;
	
	int valuesOfSameType = 0;
	try {
		for (KeyValue kv : m_NomenclatureInterface.getAllKeyValues()) {
			if (l_KeyValue.getKeyType().equals(kv.getKeyType())) {
				valuesOfSameType++;
			}
		}		
	} catch (Exception e) {
		
	}
	
	if (valuesOfSameType < 1) {
		l_RateCount = this.getRateCountByKeyValue (l_KeyValue);
		if (l_RateCount > 0)
		{
			l_Filler = new String [2];
			l_Filler[0] = l_KeyValue.getLabel();	
			l_Filler[1] = Long.valueOf(l_RateCount).toString();
			
			l_UserChoice = BillingModule.getUserConfirmation(
					c_ReferringRatesTitle, 
					c_ReferringRatesMessage, 
					l_Filler);	
			
			l_DeleteIt = (l_UserChoice == JOptionPane.YES_OPTION);
		}
		else {
			l_UserChoice = BillingModule.getUserConfirmation(
					c_DeleteKeyvalTitle, 
					c_DeleteKeyvalMessage, 
					new String[0]);	
			
			l_DeleteIt = (l_UserChoice == JOptionPane.YES_OPTION);
		}
	} else {
		l_UserChoice = BillingModule.getUserConfirmation(
				c_DeleteKeyvalTitle, 
				c_DeleteKeyvalMessage, 
				new String[0]);	
		
		l_DeleteIt = (l_UserChoice == JOptionPane.YES_OPTION);
	}
	
		
	if (l_DeleteIt)
		{
		if (valuesOfSameType <= 1) {
			deleteRates(l_KeyValue);
		}
		
		this.deleteKeyValue(l_KeyValue);	
		
		reloadTreeTable();
		
		l_Event = new KeyValueChangeEvent (this,l_KeyValue,KeyValueChangeEvent.c_KeyValueDeleted);
		m_KeyValueListeners.notifyKeyValueListeners(l_Event);
		}
	}

//---------------------------------------------------------------------------

	private void reloadTreeTable() {
//		m_KeyValueTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
//		m_KeyValueTable.getTableHeader().setReorderingAllowed(false);

		m_KeyValueModel.setRoot(buildTree());
		m_KeyValueTable.setTreeTableModel(m_KeyValueModel);
		m_KeyValueTable.validate();

		 doLayout();
		 
//		 m_KeyValueTable.getColumnModel().getColumn(KeyValueTreeTableModel.c_LabelColumn).setPreferredWidth(200);
//		 m_KeyValueTable.getColumnModel().getColumn(KeyValueTreeTableModel.c_TypeColumn).setPreferredWidth(20);
//		 m_KeyValueTable.getColumnModel().getColumn(KeyValueTreeTableModel.c_ValueColumn).setPreferredWidth(50);
//		 m_KeyValueTable.getColumnModel().getColumn(KeyValueTreeTableModel.c_ApplicableDateColumn).setPreferredWidth(50);

	}

//---------------------------------------------------------------------------
/**
* doLayout is the main entry point for JTable resizing and row/column space
* distribution. The ActListBox class overrides the method to impose its own
* distribution scheme. Width of columns is expressed as percents and we want
* the table to reflect this distribution. 
*/
//---------------------------------------------------------------------------

public void doLayout ()
	{
  TableColumn  l_Column;
	double		l_Width;
	int			l_Index;
	int			l_ColumnWidth;
	
	super.doLayout ();
	
	l_Width = (double) m_KeyValueTable.getWidth () / 100;
	
	for (l_Index = 0; l_Index < c_ColumnWidths.length; l_Index++)
		{
		l_Column = m_KeyValueTable.getColumnModel().getColumn(l_Index);
		l_ColumnWidth = (int)(l_Width * c_ColumnWidths[l_Index]);
		l_Column.setPreferredWidth(l_ColumnWidth);
		l_Column.setPreferredWidth(l_ColumnWidth);
		}
	}

public void relocalize()
	{
	if (m_AddKeyValueButton != null)
		{
		m_AddKeyValueButton.setText (Translatrix.getTranslationString(c_AddKeyValueButton));
		m_AddKeyValueButton.setToolTipText (Translatrix.getTranslationString(c_AddKeyValueTip));
		}
	
	if (m_RemoveKeyValueButton != null)
		{
		m_RemoveKeyValueButton.setText (Translatrix.getTranslationString(c_RemoveKeyValueButton));
		m_RemoveKeyValueButton.setToolTipText (Translatrix.getTranslationString(c_RemoveKeyValueTip));
		}
	if (m_ReloadButton != null)
	{
		m_ReloadButton.setText (Translatrix.getTranslationString("RatesPanel.reload"));
		m_ReloadButton.setToolTipText (Translatrix.getTranslationString("RatesPanel.reload"));
	}
	
	}

//---------------------------------------------------------------------------

public void actionPerformed (ActionEvent p_Event)
	{
	if (p_Event.getSource().equals(m_AddKeyValueButton))
		{
		this.addNewKeyValue();
		}	
	else if (p_Event.getSource().equals(m_RemoveKeyValueButton))
		{
		this.removeSelectedKeyValue();
		}
	else if (p_Event.getSource().equals (m_ReloadButton))
		{
		this.reloadTreeTable();
		}
	}

//---------------------------------------------------------------------------

	public void valueChanged(TreeSelectionEvent p_Event) {
		KeyValueTreeNode l_Node;
		TreePath l_SelectionPath;
		int l_SelectedRow;
		KeyValue l_KeyValue;

		if (!p_Event.getSource().equals(m_KeyValueTable)) {
			l_SelectedRow = m_KeyValueTable.getSelectedRow();
			if (l_SelectedRow < 0)
				return;

			l_SelectionPath = m_KeyValueTable.getPathForRow(l_SelectedRow);
			l_Node = (KeyValueTreeNode) l_SelectionPath.getLastPathComponent();
			if (l_Node.getKeyValue() != null) {
				l_KeyValue = l_Node.getKeyValue();
				//Disable buttons if demo mode
				if (!GECAMedUtils.isDemo() && MainFrame.isAdmin()) {
					m_RemoveKeyValueButton.setEnabled(l_KeyValue.getEditable());
					
				}
				
			} else {
				m_RemoveKeyValueButton.setEnabled(false);
			}
		}
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* End of Class                                                            *
//***************************************************************************
//---------------------------------------------------------------------------

	}
