/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.entity.beans;

import java.io.Serializable;
import java.util.Date;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;
import lu.tudor.santec.gecamed.usermanagement.ejb.entity.beans.GecamedUser;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The TrashedMemo bean is intended to hold data of deleted memos. The TrashedMemo bean 
 * is an exact clone (property wise) of the Memo bean. The only difference is that it 
 * is mapped to a different database table. Doing so, allows deleted memos to be saved
 * for later restore if necessary.
 * @author nico.mack@tudor.lu
 * @since 08/01/14
 */

@Entity
@Table(name = "trashed_memo", schema = "billing")

public class TrashedMemo extends GECAMedEntityBean implements Serializable 
	{
	private static final long serialVersionUID = 1L;

	private Integer		m_InvoiceId;
	private Date		m_CreationDate;
	private GecamedUser	m_Author;
	private String		m_Note;
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	

//---------------------------------------------------------------------------

public TrashedMemo ()
	{
	this.init ();
	}	

//---------------------------------------------------------------------------
/**
 * Creates a new instance of TrashedMemo initialized with the properties of the
 * specified memo.
 * @param p_Memo specifies the memo to get data from to initialize the newly
 * created TrashedMemo.
 */
//---------------------------------------------------------------------------


public TrashedMemo (Memo p_Memo)
	{
	if (p_Memo != null)
		{
		this.setInvoiceId		(p_Memo.getInvoiceId());
		this.setCreationDate	(p_Memo.getCreationDate());
		this.setAuthor			(p_Memo.getAuthor());
		this.setNote			(p_Memo.getNote());
		}
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives					                                            *
//***************************************************************************	
//---------------------------------------------------------------------------

private void init ()
	{
	m_CreationDate = new Date ();
	m_Note		  = new String ();
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Getter and Setter Methods	                                            *
//***************************************************************************	
//---------------------------------------------------------------------------
/**
 * Returns the ID of the trashed invoice this memo is attached to
 * @return the ID of the trashed invoice
 */
//---------------------------------------------------------------------------

@Column(name = "invoice_id", updatable=false, insertable=false)
public Integer getInvoiceId() 
	{
	return m_InvoiceId;
	}

//---------------------------------------------------------------------------
/**
 * Sets the ID of the trashed invoice this memo is attached to
 * @param p_InvoiceId specifies the ID of the trashed invoice this memo is 
 * attached to
 */
//---------------------------------------------------------------------------

public void setInvoiceId (Integer p_InvoiceId)
	{
	m_InvoiceId = p_InvoiceId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the creation date of this memo
 * @return memo creation date
 */
//---------------------------------------------------------------------------

@Column (name ="created")

public Date getCreationDate() 
	{
	return m_CreationDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the creation of this memo
 * @param p_CreationDate specifies the creation date of this memo
 */
//---------------------------------------------------------------------------

public void setCreationDate (Date p_CreationDate) 
	{
	m_CreationDate = (p_CreationDate != null) ? new Date (p_CreationDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the author of this memo
 * @return The user who created this memo
 */
//---------------------------------------------------------------------------

@ManyToOne (fetch = javax.persistence.FetchType.LAZY)
@JoinColumn (name ="author_id")

public GecamedUser getAuthor() 
	{
	return m_Author;
	}

//---------------------------------------------------------------------------
/**
 * Sets the author of this memo
 * @param p_Author specifies the user who created this memo
 */
//---------------------------------------------------------------------------

public void setAuthor (GecamedUser p_Author) 
	{
	m_Author = p_Author;
	}

//---------------------------------------------------------------------------
/**
 * Returns the actual text of this memo
 * @return memo text
 */
//---------------------------------------------------------------------------

@Column (name ="note")

public String getNote() 
	{
	return m_Note;
	}

//---------------------------------------------------------------------------
/**
 * Sets the actual text of this memo
 * @param p_Note specifies the new text for this memo
 */
//---------------------------------------------------------------------------

public void setNote (String p_Note) 
	{
	m_Note = p_Note;
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Recreation Methods														*
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * The method recreates a new instance of memo from this trashed memo instance.
 * @return a new memo initialized with the properties of this trashed memo.
 */
//---------------------------------------------------------------------------

@Transient
public Memo recreateMemo ()
	{
	Memo	l_Memo;
	
	l_Memo = new Memo ();

	l_Memo.setInvoiceId		(this.getInvoiceId());
	l_Memo.setCreationDate	(this.getCreationDate());
	l_Memo.setAuthor		(this.getAuthor());
	l_Memo.setNote			(this.getNote());
	
	return l_Memo;
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* End of Class															*
//***************************************************************************
//---------------------------------------------------------------------------
}
