/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.entity.beans;

import java.io.Serializable;

import javax.persistence.Entity;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;

import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The Settlement class represents a settlement method for an invoice. An invoice
 * can be settled either via a payment, i.e. cash, debit or credit ccard, or via
 * a money transfer to one of the office's (or physician's) bank accounts. 
 * @author nico.mack@tudor.lu
 * @since 06/04/19
 */

@Entity
@Cache(usage = CacheConcurrencyStrategy.NONSTRICT_READ_WRITE) 
@Table(name = "settlement", schema = "billing")

@NamedQueries 
	({
	@NamedQuery(name = Settlement.c_AllSettlements, 				query = "SELECT OBJECT(o) FROM Settlement o"),
	@NamedQuery(name = Settlement.c_TransferSettlements, 			query = "SELECT OBJECT(o) FROM Settlement o WHERE o.transferAccount is not null"),
	@NamedQuery(name = Settlement.c_PaymentSettlements, 			query = "SELECT OBJECT(o) FROM Settlement o WHERE o.payment is not null"),
	@NamedQuery(name = Settlement.c_SettlementByAccount, 			query = "SELECT OBJECT(o) FROM Settlement o WHERE o.transferAccount = :account"),
	@NamedQuery(name = Settlement.c_SettlementByPayment, 			query = "SELECT OBJECT(o) FROM Settlement o WHERE o.payment = :payment"),
	@NamedQuery(name = Settlement.c_TransferSettlementsByHolder, 	query = "SELECT OBJECT(o) FROM Settlement o WHERE o.transferAccount is not null AND o.transferAccount.holder = :holder ORDER BY o.id"),
	@NamedQuery(name = Settlement.c_TransferSettlementsWithoutHolder,query = "SELECT OBJECT(o) FROM Settlement o WHERE o.transferAccount is not null AND o.transferAccount.holder is null ORDER BY o.id")
	})

public class Settlement extends GECAMedEntityBean implements Serializable 
	{
	private static final long serialVersionUID = 1L;

	private Account	m_TransferAccount;
	private Payment	m_Payment;
	
//***************************************************************************
//* Constants		                                                        *
//***************************************************************************	

	public static final int	c_None	   = 0;
	public static final int	c_Transfer = 1;
	public static final int	c_Payment  = 2;
	
	public static final transient String c_AllSettlements 					= "getAllSettlements";
	public static final transient String c_TransferSettlements 				= "getTransferSettlements";
	public static final transient String c_PaymentSettlements 				= "getPaymentSettlements";
	public static final transient String c_SettlementByAccount 				= "getSettlementByAccount";
	public static final transient String c_SettlementByPayment 				= "getSettlementByPayment";
	public static final transient String c_TransferSettlementsByHolder 		= "getTransferSettlementsByHolder";
	public static final transient String c_TransferSettlementsWithoutHolder = "getTransferSettlementsWithoutHolder";
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	
	
//***************************************************************************
//* Getter and Setter Methods	                                            *
//***************************************************************************	
	
//---------------------------------------------------------------------------
/**
 * Returns the Account object associated with this money transfer settlement 
 * @return Account the money was transfered to if settlement is a money
 * transfer, <code>null</code> otherwise.
 */
//---------------------------------------------------------------------------

@ManyToOne
@JoinColumn(name = "account_id")

public Account getTransferAccount() 
	{
	return m_TransferAccount;
	}

//---------------------------------------------------------------------------
/**
 * Sets this settlements' transfer account
 * @param p_TransferAccount specifies the new transfer account for this settlement
 */
//---------------------------------------------------------------------------

public void setTransferAccount (Account p_TransferAccount) 
	{
	m_TransferAccount = p_TransferAccount;
	}

//---------------------------------------------------------------------------
/**
 * Returns this settlements payment method
 * @return The Payment method used if settlement is a payment, <code>null</code>
 * otherwise.
 */
//---------------------------------------------------------------------------

@ManyToOne
@JoinColumn(name = "payment_id")

public Payment getPayment() 
	{
	return m_Payment;
	}

//---------------------------------------------------------------------------
/**
 * Sets this settlements' payment method
 * @param p_Payment specifies the new payment method for this settlement
 */
//---------------------------------------------------------------------------

public void setPayment (Payment p_Payment) 
	{
	m_Payment = p_Payment;
	}

//---------------------------------------------------------------------------
/**
 * Allows to find out what kind of settlement this is.
 * @return The method returns one of the following values:
 * c_None    : If settlement defines neither payment nor transfer account.
 * c_Payment : If settlement defines a payment method like cash, credit or
 *             denit card.
 * c_Trasnfer: If settlement defines a transfer account.
 */
//---------------------------------------------------------------------------

@Transient
public Integer getSettlementMethod ()
	{
	Integer	l_Mode = c_None;
	
	if (this.getPayment() != null) l_Mode = c_Payment;
	else if (this.getTransferAccount() != null) l_Mode = c_Transfer;
	
	return l_Mode;
	}

//---------------------------------------------------------------------------

@Override
public String toString () 
	{
	Integer	l_Method;
	
	l_Method = this.getSettlementMethod();
	
	switch (l_Method)
		{
		case c_None     : return "None";
		case c_Payment  : return getPayment().toString();
		case c_Transfer : return getTransferAccount().toString();
		}
	
	return "";
	}


public String toLabel ()
{
	return toLabel(true);
}


public String toLabel (boolean labelOnly)
{
	Integer	l_Method;
	
	l_Method = this.getSettlementMethod();
	
	switch (l_Method)
	{
		case c_None:
			return "None";
			
		case c_Payment:
			return getPayment().toString();
			
		case c_Transfer:
			return getTransferAccount().toLabel(labelOnly);
	}
	
	return "";
}

//***************************************************************************
//* End of Class															*
//***************************************************************************
}
