/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.entity.beans;

import java.io.Serializable;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashSet;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.NamedQuery;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;
import lu.tudor.santec.gecamed.core.utils.GECAMedUtils;
import lu.tudor.santec.gecamed.patient.utils.PatientNameFormatter;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The InvoiceStub Bean is a composite view of invoices, containing also
 * data about the patient the invoice is charged to. InvoiceStubs were mainly
 * introduced to improve performance when loading large numbers of invoices for
 * all kind of lists.
 * @author nico.mack@tudor.lu
  */

@Entity
@Table(name = "invoice_stub", schema = "billing")

@javax.persistence.NamedQueries 
	({
	@NamedQuery(name = InvoiceStub.c_InvoiceStubById, 				 	 	query = "SELECT OBJECT(o) FROM InvoiceStub o WHERE o.id = :" + InvoiceStub.c_InvoiceStubById),
	@NamedQuery(name = InvoiceStub.c_InvoiceStubsByStatementId, 			query = "SELECT OBJECT(o) FROM InvoiceStub o WHERE o.statementId = :" + InvoiceStub.c_StatementIdParameter + " ORDER BY o.patientName ASC, o.patientFirstName ASC"),
	@NamedQuery(name = InvoiceStub.c_InvoiceStubCountByStatementId, 	 	query = "SELECT COUNT(o) FROM InvoiceStub o WHERE o.statementId = :" + InvoiceStub.c_StatementIdParameter),
	@NamedQuery(name = InvoiceStub.c_SettledInvoiceStubCountByStatementId,  query = "SELECT COUNT(o) FROM InvoiceStub o WHERE o.statementId = :" + InvoiceStub.c_StatementIdParameter + " AND o.balance = 0d"),
	@NamedQuery(name = InvoiceStub.c_LastInvoiceForPatient,				 	query = "SELECT OBJECT(o) FROM InvoiceStub o WHERE o.patientId = :" + InvoiceStub.c_PatientIdParameter + " ORDER BY o.invoiceDate, o.id DESC")
	})
	
public class InvoiceStub extends GECAMedEntityBean implements Serializable
	{
	private static final long serialVersionUID = 1L;

	private Boolean					m_Ticked;
	private Integer					m_HospitalisationClassId;
	private Integer					m_PatientId;
	private Integer					m_PhysicianId;
	private Integer					m_ThirdPartyPayerId;
	private String  				m_PatientFirstName;
	private String  				m_PatientMaidenName;
	private String  				m_PatientName;
	private String					m_PatientSSN;
	private Date 					m_InvoiceDate;
	private Date 					m_PrintedDate;
	private Date					m_DueDate;
	private Integer					m_State;
	private Integer					m_OldState;
	private Integer					m_Reminders;
	private Date					m_ReminderDate;
	private Integer					m_StatementId;
	private Integer					m_SettlementId;
	private Date					m_SettlementDate;
	private Double					m_Amount;
	private Double					m_Payment;
	private Double					m_Deduction;
	private Double					m_Balance;
	private Integer					m_InsuranceId;
	private Date 					m_ClosureDate;
	private String					m_Bailif;
	private String					m_GroupingString = "";

	private String m_Physician;
	
	
	private static PatientNameFormatter m_Formatter = new PatientNameFormatter ();
	
//***************************************************************************
//* Class Constants                                                         *
//***************************************************************************
	
	public static final transient String c_InvoiceStubById  					= "getInvoiceStubById";
	public static final transient String c_InvoiceStubsByStatementId  			= "getInvoiceStubByStatementId";
	public static final transient String c_InvoiceStubCountByStatementId  		= "getInvoiceStubCountByStatementId";
	public static final transient String c_SettledInvoiceStubsByStatementId  	= "getSettledInvoiceStubByStatementId";
	public static final transient String c_SettledInvoiceStubCountByStatementId = "getSettledInvoiceStubCountByStatementId";
	public static final transient String c_LastInvoiceForPatient			  	= "getLastInvoiceForPatient";
	
    public static final transient String c_IdParameter							= "id";
    public static final transient String c_StatementIdParameter					= "statementId";
    public static final transient String c_PatientIdParameter					= "patientId";
	
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	



public InvoiceStub ()
	{
	m_State						= Integer.valueOf (0);
	m_Reminders					= Integer.valueOf (0);
	m_Amount					= Double.valueOf (0);
	m_Payment					= Double.valueOf (0);
	m_Deduction					= Double.valueOf (0);
	m_Balance					= Double.valueOf (0);
	m_Ticked					= Boolean.valueOf (false);
	}	

//---------------------------------------------------------------------------
//***************************************************************************
//* Primitives                                       						*
//***************************************************************************
//---------------------------------------------------------------------------
/**
 * The addToTotal method is a helper method intended to sum amounts of 
 * invoices on a list. The method adds the specified amount to the specified
 * total and returns the new total. The method relies on BigDecimals to
 * perform the addition.
 * @param p_Total specifies the total to add amount to.
 * @param p_Amount specifies the amount to be added.
 * @return the value of p_Total plus p_Amount. In case p_Amount was
 * <code>null</code> the method will return the unmodified p_Total.
 */
//---------------------------------------------------------------------------

@Transient
public static BigDecimal addToTotal (BigDecimal p_Total, Double p_Amount)
	{
	BigDecimal l_Amount;				
	
	if ((p_Total == null) || (p_Amount == null)) return p_Total;
	
	l_Amount = new BigDecimal (p_Amount).setScale (2,RoundingMode.HALF_DOWN);				
	p_Total = p_Total.add (l_Amount);

	return p_Total;
	}

//---------------------------------------------------------------------------
/**
 * The addToTotal method is a helper method intended to sum amounts of 
 * invoices on a list. The method subtracts the specified amount from the 
 * specified total and returns the new total. The method relies on BigDecimals to
 * perform the subtraction.
 * @param p_Total specifies the total to subtract amount from.
 * @param p_Amount specifies the amount to be subtracted.
 * @return the value of p_Total minus p_Amount. In case p_Amount was
 * <code>null</code> the method will return the unmodified p_Total.
 */
//---------------------------------------------------------------------------

@Transient
public static BigDecimal subFromTotal (BigDecimal p_Total, Double p_Amount)
	{
	BigDecimal l_Amount;				
	
	if ((p_Total == null) || (p_Amount == null)) return p_Total;
	
	l_Amount = new BigDecimal (p_Amount).setScale (2,RoundingMode.HALF_DOWN);				
	p_Total = p_Total.subtract (l_Amount);

	return p_Total;
	}

//---------------------------------------------------------------------------
/**
 * Returns the double value of the specified total. This method will disappear
 * when billing module will be completely switched oto BigDecimal arithemtics.
 * @param p_Total specifies the total to get double value of.
 * @return double value of specified total. If specified total was <code>null</code>,
 * the value 0 will be returned.
 */
//---------------------------------------------------------------------------

@Transient
public static Double getTotal (BigDecimal p_Total)
	{
	if (p_Total != null) return (p_Total.doubleValue());
	else return 0d;
	}

//---------------------------------------------------------------------------
/**
 * the formatInvoiceNumber formates the invoice number either as a short
 * number or as a long number. 
 * @param p_Short specifies whether short format should be used for invoice
 * number or long format. Specify <code>true</code> for short format, <code>
 * false</code> for long format.
 * @return the formatted invoice number.
 * @see #Invoice.formatInvoiceNumber (Integer,Integer,Integer,boolean)
 */
//---------------------------------------------------------------------------

@Transient
public String formatInvoiceNumber (boolean p_Short, boolean p_LeadingZeros)
	{
	return Invoice.formatInvoiceNumber (this.getId(),this.getPhysicianId(),this.getPatientId(),p_Short,p_LeadingZeros);
	}

//---------------------------------------------------------------------------
/**
 * the stripTime method neutralizes time data contained in specified date.
 * Neutralization is done by setting hour, minute, second and milliseconds
 * to zero.
 * @param p_OriginalDate specifies original date to strip time data from.
 * @return A date object reflecting the date specified p_OriginalDate but
 * being devoid of any time data.
 * @see #Invoice.stripTime (Date)
 */
//---------------------------------------------------------------------------

public static Date stripTime (Date p_OriginalDate)
	{
	return GECAMedUtils.stripTime (p_OriginalDate);
	}

//---------------------------------------------------------------------------
/**
 * The method checks whether the invoice this stub represents can still be 
 * modified or not. 
 * @return <code>true</code> if invoice is still open an can be modified,
 * <code>false</code> otherwise.
 * @see #Invoice.isLocked (Integer)
 */
//---------------------------------------------------------------------------

@Transient
public boolean isLocked ()
	{
	return Invoice.isLocked (m_State);
	}

//---------------------------------------------------------------------------
/**
 * The method checks whether this invoice this stub represents has expired, i.e. 
 * its due date is in the past or not. The state of the invoice also influences 
 * the result of this method.
 * @return <code>true</code> if invoice has expired, <code>false</code> otherwise.
 * @see #isExpired (Date,Integer,Double)
 */
//---------------------------------------------------------------------------

@Transient
public boolean isExpired ()
	{
	return Invoice.isExpired (this.getDueDate(), this.getState (), this.getBalance());
	}

//---------------------------------------------------------------------------
/**
 * returns the full name of the patient.
 * @return full name of patient.
 */
//---------------------------------------------------------------------------

@Transient
public String getPatientFullName() 
	{
	return m_Formatter.getPatientFullName (m_PatientFirstName, m_PatientMaidenName, m_PatientName);
	}

//---------------------------------------------------------------------------
/**
 * given a collection of InvoiceStubs, this method returns a collection
 * holding the IDs of all the invoices represented by specified InvoiceStubs.
 * @param p_InvoiceStubs specifies the collection of InvoiceStubs to get
 * Invoice IDs of.
 * @return A collection holding the IDs of the invoices represented by the
 * specified InvoiceStubs.
 */
//---------------------------------------------------------------------------

@Transient
public static Collection <Integer> getInvoiceIDs (Collection <InvoiceStub> p_InvoiceStubs)
	{
	Collection <Integer> 		l_InvoiceIDs;
	Iterator <InvoiceStub>	 	l_StubIterator;
	InvoiceStub				 	l_Stub;
	
	l_InvoiceIDs = new LinkedHashSet <Integer> ();
	
	if (p_InvoiceStubs == null) return l_InvoiceIDs;
	
	l_StubIterator = p_InvoiceStubs.iterator();
	while (l_StubIterator.hasNext())
		{
		l_Stub = l_StubIterator.next();
		l_InvoiceIDs.add(l_Stub.getId());
		}

	return l_InvoiceIDs;
	}

//---------------------------------------------------------------------------
//***************************************************************************
//* Class Body                                                              *
//***************************************************************************
//---------------------------------------------------------------------------

@Transient
public static void setMaidenNameIsMarriedName (Boolean p_MaidenNameIsMarriedName)
	{
	m_Formatter.setMaidenNameIsMarriedName(p_MaidenNameIsMarriedName);
	}

//---------------------------------------------------------------------------
/**
 * Returned the state of the ticked property. The ticked property is no
 * intrinsic property of this bean, i.e. it has to be explicitely set.
 * @return <code>true</code> if this invoice stub has its ticked flag set,
 * <code>false</code> otherwise.
 */
//---------------------------------------------------------------------------

@Transient
public Boolean getTicked() 
	{
	return m_Ticked;
	}

//---------------------------------------------------------------------------
/**
 * Explicitely sets the state of the ticked property for this invoice stub.
 * @param p_Ticked specifies the new state of the ticked flag.
 */
//---------------------------------------------------------------------------

@Transient
public void setTicked (Boolean p_Ticked) 
	{
	m_Ticked = p_Ticked;
	}

//---------------------------------------------------------------------------
/**
 * Returns the ID of the hospitalisation class of this invoice stub
 * @return The ID the hospitalisation class of this invoice stub
 */
//---------------------------------------------------------------------------

@Column(name = "class_id", insertable =false, updatable = false)

public Integer getHospitalisationClassId() 
	{
	return m_HospitalisationClassId;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified hospitalisation class with this invoice stub
 * @param p_HospitalisationClassId specifies the ID of the hospitalisation  
 * class to be associated with this invoice stub
 */
//---------------------------------------------------------------------------

public void setHospitalisationClassId (Integer p_HospitalisationClassId) 
	{
	m_HospitalisationClassId = p_HospitalisationClassId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the ID of the patient associated with this invoice stub.
 * @return The ID of the patient associated with this stub.
 */
//---------------------------------------------------------------------------

@Column(name = "patient_id", insertable =false, updatable = false)

public Integer getPatientId() 
	{
	return m_PatientId;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified patient with this invoice stub
 * @param p_PatientId specifies the ID of the patient to be associated with this
 * stub
 */
//---------------------------------------------------------------------------

public void setPatientId (Integer p_PatientId) 
	{
	m_PatientId = p_PatientId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the ID of the physician associated with this invoice stub
 * @return The ID of the physician associated with this stub
 */
//---------------------------------------------------------------------------

@Column(name = "physician_id", insertable =false, updatable = false)

public Integer getPhysicianId() 
	{
	return m_PhysicianId;
	}

//---------------------------------------------------------------------------
/**
 * Associates the specified physician with this invoice stub
 * @param p_PhysicianId specifies ID of the physician to be associated with this
 * stub
 */
//---------------------------------------------------------------------------

public void setPhysicianId (Integer p_PhysicianId) 
	{
	m_PhysicianId = p_PhysicianId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the first name of the patient associated with this invoice stub
 * @return first name of patient this stub is associated with
 */
//---------------------------------------------------------------------------

@Column(name = "patient_first_name", insertable =false, updatable = false)

public String getPatientFirstName ()
	{
	return m_PatientFirstName;
	}

//---------------------------------------------------------------------------
/**
 * Sets the first name of the patient associated with this invoice stub
 * @param p_PatientFirstName specifies the first name of the patient this stub 
 * is associated with
 */
//---------------------------------------------------------------------------

public void setPatientFirstName (String p_PatientFirstName)
	{
	m_PatientFirstName = p_PatientFirstName;
	}

//---------------------------------------------------------------------------
/**
 * Returns the maiden/married name of the patient associated with this invoice 
 * stub
 * @return maiden/married name of patient this stub is associated with
 */
//---------------------------------------------------------------------------

@Column(name = "patient_maiden_name", insertable =false, updatable = false)

public String getPatientMaidenName ()
	{
	return m_PatientMaidenName;
	}

//---------------------------------------------------------------------------
/**
 * Sets the maiden/married name of the patient associated with this invoice stub
 * @param p_PatientMaidenName specifies the maiden/married name of the patient 
 * this stub is associated with
 */
//---------------------------------------------------------------------------

public void setPatientMaidenName (String p_PatientMaidenName)
	{
	m_PatientMaidenName = p_PatientMaidenName;
	}

//---------------------------------------------------------------------------
/**
 * Returns the name of the patient associated with this invoice stub
 * @return name of patient this stub is associated with
 */
//---------------------------------------------------------------------------

@Column(name = "patient_name", insertable =false, updatable = false)

public String getPatientName ()
	{
	return m_PatientName;
	}

//---------------------------------------------------------------------------
/**
 * Sets the name of the patient associated with this invoice stub
 * @param p_PatientName specifies the name of the patient this stub 
 * is associated with
 */
//---------------------------------------------------------------------------

public void setPatientName (String p_PatientName)
	{
	m_PatientName = p_PatientName;
	}

//---------------------------------------------------------------------------
/**
 * Returns the social security number of the patient associated with this invoice stub
 * @return social security number of patient this stub is associated with
 */
//---------------------------------------------------------------------------

@Column(name = "patient_ssn", insertable =false, updatable = false)

public String getPatientSSN ()
	{
	return m_PatientSSN;
	}

//---------------------------------------------------------------------------
/**
 * Sets the social security number of the patient associated with this invoice stub
 * @param p_PatientSSN specifies the social security number of the patient 
 * this stub is associated with
 */
//---------------------------------------------------------------------------

public void setPatientSSN (String p_PatientSSN)
	{
	m_PatientSSN = p_PatientSSN;
	}

//---------------------------------------------------------------------------
/**
 * Returns the third party paying insurance covering this invoices' due amount
 * @return The third party paying insurance associated with this invoice
 */
//---------------------------------------------------------------------------

@Column(name = "third_party_id", insertable =false, updatable = false)

public Integer getThirdPartyPayerId() 
	{
	return m_ThirdPartyPayerId;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified insurance as the third party payer for this invoices'
 * due amount
 * @param p_ThirdPartyPayerId specifies the third party paying insurance to be
 * associated with this invoice
 */
//---------------------------------------------------------------------------

public void setThirdPartyPayerId (Integer p_ThirdPartyPayerId) 
	{
	m_ThirdPartyPayerId = p_ThirdPartyPayerId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the creation date of the invoice this stub represents
 * @return The invoice creation date
 */
//---------------------------------------------------------------------------

@Column(name = "created", insertable =false, updatable = false)

public Date getInvoiceDate() 
	{
	return m_InvoiceDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the creation date of the invoice this stub represents
 * @param p_InvoiceDate specifies the new invoice creation date
 */
//---------------------------------------------------------------------------

public void setInvoiceDate (Date p_InvoiceDate) 
	{
	m_InvoiceDate = (p_InvoiceDate != null) ? new Date (p_InvoiceDate.getTime()) : null;
	}


//---------------------------------------------------------------------------
/**
* Returns the Printed date of the invoice this stub represents
* @return The invoice Printed date
*/
//---------------------------------------------------------------------------

@Column(name = "printed", insertable =false, updatable = false)

public Date getPrintedDate() 
	{
	return m_PrintedDate;
	}

//---------------------------------------------------------------------------
/**
* Sets the Printed date of the invoice this stub represents
* @param p_InvoiceDate specifies the new invoice Printed date
*/
//---------------------------------------------------------------------------

public void setPrintedDate (Date p_PrintedDate) 
	{
	m_PrintedDate = (p_PrintedDate != null) ? new Date (p_PrintedDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the due date of the invoice this stub represents
 * @return The invoice due date
 */
//---------------------------------------------------------------------------

@Column(name = "due", insertable =false, updatable = false)

public Date getDueDate() 
	{
	return m_DueDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the specified date as this invoices' due number
 * @param p_DueDate specifies the new invoice due date
 */
//---------------------------------------------------------------------------

public void setDueDate (Date p_DueDate) 
	{
	m_DueDate = (p_DueDate != null) ? new Date (p_DueDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the state this invoice is currently in
 * @return The invoice state
 */
//---------------------------------------------------------------------------

@Column(name = "state", insertable =false, updatable = false)

public Integer getState() 
	{
	return m_State;
	}

//---------------------------------------------------------------------------
/**
 * Sets the state this invoice is currently in
 * @param p_State specifies the new invoice state
 */
//---------------------------------------------------------------------------

public void setState (Integer p_State) 
	{
	m_State = p_State;
	}

//---------------------------------------------------------------------------
/**
 * Returns the old state this invoice was previously in
 * @return The old invoice state
 */
//---------------------------------------------------------------------------

@Column(name = "old_state", insertable =false, updatable = false)

public Integer getOldState() 
	{
	return m_OldState;
	}

//---------------------------------------------------------------------------
/**
 * Sets the state this invoice was in
 * @param p_OldState specifies the old invoice state
 */
//---------------------------------------------------------------------------

public void setOldState (Integer p_OldState) 
	{
	m_OldState = p_OldState;
	}

//---------------------------------------------------------------------------
/**
 * Returns the number of reminders already issued for the invoice this stub
 * represents
 * @return The number of already issued reminders.
 */
//---------------------------------------------------------------------------

@Column(name = "reminders", insertable =false, updatable = false)

public Integer getNumberOfReminders() 
	{
	return m_Reminders;
	}

//---------------------------------------------------------------------------
/**
 * Sets the number of already issued reminders
 * @param p_Reminders specifies the number of issued reminders
 */
//---------------------------------------------------------------------------

public void setNumberOfReminders (Integer p_Reminders) 
	{
	m_Reminders = p_Reminders;
	}

//---------------------------------------------------------------------------
/**
 * Returns the date the last reminder was issued
 * @return Latest reminder date
 */
//---------------------------------------------------------------------------

@Column(name = "reminder_date", insertable =false, updatable = false)

public Date getReminderDate() 
	{
	return m_ReminderDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the date the last reminder was issued
 * @param p_ReminderDate specifies the latest reminder date
 */
//---------------------------------------------------------------------------

public void setReminderDate (Date p_ReminderDate) 
	{
	m_ReminderDate = (p_ReminderDate != null) ? new Date (p_ReminderDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the Id of the statement the invoice this stub represents is on.
 * @return The Id of the statement, <code>null</code> if none.
 */
//---------------------------------------------------------------------------

@Column(name = "statement_id", insertable =false, updatable = false)

public Integer getStatementId() 
	{
	return m_StatementId;
	}

//---------------------------------------------------------------------------
/**
 * Sets the Id of the statement this invoice stub is part of
 * @param p_StatementId specifies the new Id of the statement this stub is
 * to be part of, <code>null</code> if none.
 */
//---------------------------------------------------------------------------

public void setStatementId (Integer p_StatementId) 
	{
	m_StatementId = p_StatementId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the Id of the settlement method the invoice this stub represents 
 * was settled with
 * @return The Id of the settlement method, <code>null</code> if none.
 */
//---------------------------------------------------------------------------

@Column(name = "settlement_id", insertable =false, updatable = false)

public Integer getSettlementId() 
	{
	return m_SettlementId;
	}

//---------------------------------------------------------------------------
/**
 * Sets the Id of the settlement method used to settle this invoice stub
 * @param p_SettlementId specifies the new Id of the settlement method of
 * this stub, <code>null</code> if none.
 */
//---------------------------------------------------------------------------

public void setSettlementId (Integer p_SettlementId) 
	{
	m_SettlementId = p_SettlementId;
	}

//---------------------------------------------------------------------------
/**
 * Returns the date the invoice this stub represents was settled
 * @return Settlement Date of invoice
 */
//---------------------------------------------------------------------------

@Column(name = "settlement_date", insertable =false, updatable = false)

public Date getSettlementDate() 
	{
	return m_SettlementDate;
	}

//---------------------------------------------------------------------------
/**
 * Sets the date this invoice stub was settled
 * @param p_SettlementDate specifies the new settlement date.
 */
//---------------------------------------------------------------------------

public void setSettlementDate (Date p_SettlementDate) 
	{
	m_SettlementDate = (p_SettlementDate != null) ? new Date (p_SettlementDate.getTime()) : null;
	}

//---------------------------------------------------------------------------
/**
 * Returns the value the invoice this stub represents amounts to
 * @return The itotal amount
 */
//---------------------------------------------------------------------------

@Column(name = "amount", insertable =false, updatable = false)

public Double getAmount() 
	{
	return m_Amount;
	}

//---------------------------------------------------------------------------
/**
 * Sets the value this invoice stub amounts to.
 * @param p_Amount specifies the new amount
 */
//---------------------------------------------------------------------------

public void setAmount (Double p_Amount) 
	{
	m_Amount = p_Amount;
	}

//---------------------------------------------------------------------------
/**
 * Returns the already paid amount for the invoice this stub represents
 * @return The already paid amount
 */
//---------------------------------------------------------------------------

@Column(name = "payment", insertable =false, updatable =false)

public Double getPayment() 
	{
	return m_Payment;
	}

//---------------------------------------------------------------------------
/**
 * Sets the value already paid for this stub
 * @param p_Payment specifies the already paid amount
 */
//---------------------------------------------------------------------------

public void setPayment (Double p_Payment) 
	{
	m_Payment = p_Payment;
	}

//---------------------------------------------------------------------------
/**
 * Returns an eventual deduction applied to the invoice this stub represents. 
 * Deductions may be applied to invoice to compensate for non-reimbursed amounts.
 * @return The deduced amount
 */
//---------------------------------------------------------------------------

@Column(name = "deduction", insertable =false, updatable =false)

public Double getDeduction() 
	{
	return m_Deduction;
	}

//---------------------------------------------------------------------------
/**
 * Sets the amount to be deduced from this invoice stub
 * @param p_Deduction specifies the amount to be deduced
 */
//---------------------------------------------------------------------------

public void setDeduction (Double p_Deduction) 
	{
	m_Deduction = p_Deduction;
	}

//---------------------------------------------------------------------------
/**
 * Returns the amount that still needs to be paid to settle the invoice this 
 * stub represents. The balance is normaly obtained by subtracting payment and 
 * deduction from total invoice amount.
 * @return The invoice balance
 */
//---------------------------------------------------------------------------

@Column(name = "balance", insertable =false, updatable =false)

public Double getBalance() 
	{
	return m_Balance;
	}

//---------------------------------------------------------------------------
/**
 * Sets the amount that still needs to paid to settle this invoice stub
 * @param p_Balance specifies the new balance
 */
//---------------------------------------------------------------------------

public void setBalance (Double p_Balance) 
	{
	m_Balance = p_Balance;
	}

@Column(name = "insurance_id", insertable =false, updatable =false)
public Integer getInsuranceId() {
	return m_InsuranceId;
}

public void setInsuranceId(Integer m_InsuranceId) {
	this.m_InsuranceId = m_InsuranceId;
}

@Column(name = "closure_date", insertable =false, updatable =false)
public Date getClosureDate() {
	return m_ClosureDate;
}

public void setClosureDate(Date m_ClosureDate) {
	this.m_ClosureDate = m_ClosureDate;
}

//--   Transient Helpers    -------------------------------------------

@Transient
public String getGroupingString() {
	return m_GroupingString;
}

public void setGroupingString(String m_GroupingString) {
	this.m_GroupingString = m_GroupingString;
}

@Transient
public String getBailif() {
	return m_Bailif;
}

public void setBailif(String m_Bailif) {
	this.m_Bailif = m_Bailif;
}

@Transient
public String getPhysician() {
	return m_Physician;
}

public void setPhysician(String l_Physician) {
	this.m_Physician = l_Physician;
}

public String toString() {
	StringBuffer sb = new StringBuffer("InvoiceStub: ");
	sb.append(formatInvoiceNumber(true, true));
	sb.append(" ").append(getInvoiceDate());
	sb.append(" ").append(getPatientFullName());
	
	return sb.toString();
}

//---------------------------------------------------------------------------
//***************************************************************************
//* End of Class															*
//***************************************************************************
}

