/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.billing.ejb.entity.beans;

import java.io.Serializable;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.NamedQuery;
import javax.persistence.Table;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;

import org.hibernate.annotations.Cache;
import org.hibernate.annotations.CacheConcurrencyStrategy;

//***************************************************************************
//* Class Definition and Members                                            *
//***************************************************************************

/**
 * The Account class represents the bank accounts of physicians or the medical 
 * office(s) which can be used to settle invoices.
 * @author nico.mack@tudor.lu
 * @since 06/04/27 
 */

@Entity
@Cache(usage = CacheConcurrencyStrategy.NONSTRICT_READ_WRITE) 
@Table(name = "account", schema = "billing")

@javax.persistence.NamedQueries 
	({
	@NamedQuery(name = Account.c_AllAccounts, 	   query = "SELECT OBJECT(o) FROM Account o ORDER BY o.id"),
	@NamedQuery(name = Account.c_AllActiveAccounts, 	   query = "SELECT OBJECT(o) FROM Account o WHERE o.obsolete IS NOT true ORDER BY o.id"),
	@NamedQuery(name = Account.c_AccountsByHolder, query = "SELECT OBJECT(o) FROM Account o WHERE o.holder = :holder ORDER BY o.id")
	})

public class Account extends GECAMedEntityBean implements Serializable 
	{
	private static final long serialVersionUID = 1L;
	
	private String		m_Bankname;
	private String		m_BIC;
	private String		m_IBAN;
	private String		m_Label;
	private Physician	m_Holder;
	private Boolean		m_Obsolete;
	
//***************************************************************************
//* Constructor(s)                                                          *
//***************************************************************************	

//***************************************************************************
//* Constants                                                               *
//***************************************************************************	
	
	public static final transient String c_AllAccounts 	    = "getAllAcconts";
	public static final transient String c_AllActiveAccounts 	    = "getAllActiveAcconts";
	public static final transient String c_AccountsByHolder = "getAccontsByHolder";

//***************************************************************************
//* Getter and Setter Methods	                                            *
//***************************************************************************	

//---------------------------------------------------------------------------
/**
 * Returns the BIC (Bank Identifier Code) of this account
 * @return Account BIC code
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="bic")

public String getBic() 
	{
	return m_BIC;
	}

//---------------------------------------------------------------------------
/**
 * Sets the BIC code of this account
 * @param p_BIC specifies the new BIC code for this account
 */
//---------------------------------------------------------------------------

public void setBic (String p_BIC) 
	{
	m_BIC = p_BIC;
	}

//---------------------------------------------------------------------------
/**
 * Returns the IBAN (International Bank Account Number) account number
 * @return IBAN account number
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="iban")

public String getIban() 
	{
	return m_IBAN;
	}

//---------------------------------------------------------------------------
/**
 * Sets this accounts' IBAN number
 * @param p_IBAN specifies the new IBAN account number
 */
//---------------------------------------------------------------------------

public void setIban (String p_IBAN) 
	{
	m_IBAN = p_IBAN;
	}


//---------------------------------------------------------------------------
/**
 * Returns the bankname of the account
 * @return the bankname of the account
 */
//---------------------------------------------------------------------------

@javax.persistence.Column (name ="bankname")

public String getBankname() 
	{
	return m_Bankname;
	}

//---------------------------------------------------------------------------
/**
 * Sets this accounts' bankname
 * @param p_Bankname specifies the new bankname of the account
 */
//---------------------------------------------------------------------------

public void setBankname(String p_Bankname) 
	{
	m_Bankname = p_Bankname;
	}


@Column (name ="obsolete")
public Boolean getObsolete() {
	return m_Obsolete;
}

public void setObsolete(Boolean p_obsolete) {
	this.m_Obsolete = p_obsolete;
}

@Column (name = "label")
public String getLabel()
	{
	return m_Label;
	}


public void setLabel(String m_Label)
	{
	this.m_Label = m_Label;
	}

//---------------------------------------------------------------------------
/**
 * Returns the holder of this account 
 * @return Physician who owns this account
 */
//---------------------------------------------------------------------------

@ManyToOne
@JoinColumn(name = "holder_id")

public Physician getHolder() 
	{
	return m_Holder;
	}

//---------------------------------------------------------------------------
/**
 * Sets this accounts' holder
 * @param p_Holder specifies the physician who owns this account
 */
//---------------------------------------------------------------------------

public void setHolder (Physician p_Holder) 
	{
	m_Holder = p_Holder;
	}

//---------------------------------------------------------------------------
/**
 * returns a readable representation of the account
 */
//---------------------------------------------------------------------------

public String toString() 
	{
		String s =  getBankname() + " IBAN: " + getIban();
		if (getBic() != null && ! getBic().equals("")) 
			{
			s += " code BIC: " + getBic();
			}
		return s;
	}

public String toLabel ()
{
	return toLabel(true);
}

public String toLabel (boolean labelOnly)
{
	String	label	= getLabel();
	
	if (label != null && label.trim().length() > 0)
		return labelOnly ? label : (toString() + " (" + label + ")");
	else
		return toString();
}



//***************************************************************************
//* End of Class															*
//***************************************************************************
}
