/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.addressbook.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JEditorPane;
import javax.swing.JPanel;
import javax.swing.text.html.HTMLEditorKit;

import lu.tudor.santec.gecamed.address.gui.widgets.MapOpener;
import lu.tudor.santec.gecamed.addressbook.ejb.entity.beans.Contact;
import lu.tudor.santec.gecamed.addressbook.ejb.entity.beans.ContactAddress;
import lu.tudor.santec.gecamed.core.gui.GECAMedModule;
import lu.tudor.santec.gecamed.core.gui.IconFetcher;
import lu.tudor.santec.gecamed.core.gui.utils.PrintUtilities;
import lu.tudor.santec.gecamed.core.gui.widgets.HeaderedPanel;
import lu.tudor.santec.gecamed.office.ejb.entity.beans.Physician;
import lu.tudor.santec.i18n.Translatrix;
import lu.tudor.santec.widgets.gui.ButtonFactory;

import com.jgoodies.forms.builder.ButtonBarBuilder;

/**
 * JPanel that shows a preview of the contact selected in the list of the
 * Addressbook
 * 
 *
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @version
 * <br>$Log: ContactSidebar.java,v $
 * <br>Revision 1.13  2010-04-22 07:25:15  hermen
 * <br>fixed vcard import
 * <br>
 * <br>Revision 1.12  2010-03-12 14:12:56  hermen
 * <br>icon cleanup
 * <br>
 * <br>Revision 1.11  2009-01-14 14:30:43  hermen
 * <br>added opening of maps to show address
 * <br>
 * <br>Revision 1.10  2008-12-05 15:13:08  hermen
 * <br>added menue entry for online manual and fixed printing of addressbook entries
 * <br>
 * <br>Revision 1.9  2008-11-28 09:30:21  hermen
 * <br>*** empty log message ***
 * <br>
 * <br>Revision 1.8  2008-09-25 09:42:27  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.7  2008-08-19 10:25:08  heinemann
 * <br>cleanup
 * <br>
 * <br>Revision 1.6  2008-05-06 14:29:13  hermen
 * <br>updated javadoc
 * <br>
 *
 */
public class ContactSidebar extends HeaderedPanel{

    private static final long serialVersionUID = 1L;
    private JEditorPane infoLabel;

    private DateFormat df = DateFormat.getDateInstance(DateFormat.SHORT);
    private JButton printButton;
    private JButton mapsButton;
    private Contact contact;
    
    public ContactSidebar() {
    	this.setHeader(Translatrix.getTranslationString("Addressbook.contact")+":");
    	this.setHeaderIcon(IconFetcher.getMediumIcon(AddressbookModule.class, AddressbookModule.ICON_EDIT_CONTACT));
    	this.enableAlpha(true);
    	this.setBackground(Color.WHITE);
    	this.setLayout(new BorderLayout());
    	this.infoLabel = new JEditorPane();
    	this.infoLabel.setEditorKit(new HTMLEditorKit());
    	this.infoLabel.setEditable(false);
    	this.infoLabel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
    	this.add(this.infoLabel, BorderLayout.CENTER);
    	
    	
    	
    	mapsButton = ButtonFactory.createNarrowButton(IconFetcher.getMediumIcon(GECAMedModule.class, GECAMedModule.SERVER));
    	mapsButton.setToolTipText(Translatrix.getTranslationString("MapOpener.openMaps"));
    	mapsButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		   if (contact != null && contact.getContactAddress() != null) {
		       MapOpener.openMap(contact.getContactAddress().toString());			
		  }
	    }
    	});
    	
    	printButton = ButtonFactory.createNarrowButton(IconFetcher.getMediumIcon(GECAMedModule.class, GECAMedModule.PRINT));
    	printButton.setToolTipText(Translatrix.getTranslationString("Addressbook.printContact"));
    	printButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		try {
		    PrintUtilities.printComponent(infoLabel, false);
		} catch (Exception e2) {
		    e2.printStackTrace();
		}
	    }
    	});
    	
    	
    	ButtonBarBuilder bb = new ButtonBarBuilder();
    	bb.addGlue();
    	bb.addFixed(mapsButton);
    	bb.addFixed(printButton);
    	JPanel jp = bb.getPanel();
    	jp.setBackground(Color.white);
    	this.add(jp, BorderLayout.SOUTH);
    	this.setOpaque(true);
    	this.setPreferredSize(new Dimension(200,200));
    }
    
    /**
     * set the contact to the panel
     * @param contact
     */
    @SuppressWarnings("unchecked")
    public void setContact(Contact contact)  {
	
	this.contact = contact;
	
	if (contact == null) {
	    infoLabel.setText("");
	    return;
	}
	
	StringBuffer sb = new StringBuffer("<html><br><font size=-1>");
	
	// Name Info
	sb.append("<b>" + Translatrix.getTranslationString("Addressbook.name") + ":</b><br>");
	if (contact.getTitle() != null && ! contact.getTitle().equals("")) {
	    if (Translatrix.getTranslationString("Title."+contact.getTitle()).equals("Title."+contact.getTitle())) {
		sb.append("&nbsp;&nbsp;" + contact.getTitle() + "<br>");
	    } else {
		sb.append("&nbsp;&nbsp;" + Translatrix.getTranslationString("Title."+contact.getTitle()) + "<br>");		
	    }
	    
	}
	if (contact.getNameFirst() != null && ! contact.getNameFirst().equals(""))
		sb.append("&nbsp;&nbsp;" + contact.getNameFirst() + "<br>");
	
	if (contact.getName() != null && ! contact.getName().equals(""))
		sb.append("&nbsp;&nbsp;" + contact.getName() + "<br>");
	
	// Physician Info
	if (contact.getUcmCode() != null && ! contact.getUcmCode().equals(Physician.UCM_DEFAULT)) {
		sb.append("<br>");
		sb.append("<b>" + Translatrix.getTranslationString("core.physician") + "</b><br>");
		
		sb.append("&nbsp;&nbsp;" + Translatrix.getTranslationString("Addressbook.ucmCode") + ": " + contact.getUcmCode() + "<br>");
		if (contact.getMedpres() != null) {
			sb.append("&nbsp;&nbsp;" + Translatrix.getTranslationString("Addressbook.MedPres") + ": " + contact.getMedpres() + "<br>");			
		}
	}
	
	// Phone Info
	if ((contact.getPhoneWork() != null && ! contact.getPhoneWork().equals("")) || 
			(contact.getPhonePrivate() != null	&& !contact.getPhonePrivate().equals("")) ||
			(contact.getPhoneMobile() != null && !contact.getPhoneMobile().equals("")) ||
			(contact.getFax() != null && !contact.getFax().equals("")) ||
			(contact.getEmail() != null && !contact.getEmail().equals(""))
	) {
		sb.append("<br>");
		sb.append("<b>" + Translatrix.getTranslationString("Addressbook.phones") + ":</b><br>");
		if (contact.getPhoneWork() != null && ! contact.getPhoneWork().equals("")) {
			sb.append("&nbsp;&nbsp;work: " + contact.getPhoneWork()+ "<br>");
		}
		if (contact.getPhonePrivate() != null	&& !contact.getPhonePrivate().equals("")) {
			sb.append("&nbsp;&nbsp;private: " + contact.getPhonePrivate() + "<br>");
		}
		if (contact.getPhoneMobile() != null && !contact.getPhoneMobile().equals("")) {
			sb.append("&nbsp;&nbsp;mobile: " + contact.getPhoneMobile() + "<br>");
		}
		if (contact.getFax() != null && !contact.getFax().equals("")) {
			sb.append("&nbsp;&nbsp;fax: " + contact.getFax() + "<br>");
		}
		if (contact.getEmail() != null && !contact.getEmail().equals("")) {
			sb.append("&nbsp;&nbsp;mail: " + contact.getEmail() + "<br>");
		}
	}
		
	
	// Address Info
	if (contact.getAddress() != null) { 
			sb.append("<br>");
			sb.append("<b>" + Translatrix.getTranslationString("Addressbook.addresses") + ":</b>");
        	for (Iterator iter = contact.getAddress().iterator(); iter.hasNext();) {
        		ContactAddress address = (ContactAddress) iter.next();
        		sb.append("<br>&nbsp;&nbsp;" + address.toStringMultiLine().replaceAll("\n", "<br>&nbsp;&nbsp;") + "");
        		sb.append("<br>");
        	}
	}

	// Comment Info
	if (contact.getComment() != null && !contact.getComment().equals("")) {
		sb.append("<br>");
		sb.append("<b>" + Translatrix.getTranslationString("Addressbook.comment") + ":</b><br>");
	    sb.append("&nbsp;&nbsp;" + contact.getComment().replaceAll("\n", "<br>&nbsp;&nbsp;") + "<br>");
	}
	
	// Additional Info
	if (
			contact.getBirthDate() != null || 
			(contact.getLanguage() != null && !contact.getLanguage().equals("")) ||
			(contact.getNationality() != null && !contact.getNationality().equals(""))
	) {
		sb.append("<br>");
		sb.append("<b>" + Translatrix.getTranslationString("Addressbook.additional") + ":</b><br>");
		if (contact.getBirthDate() != null ) {
			sb.append("&nbsp;&nbsp;" +Translatrix.getTranslationString("Addressbook.dob") + ": " + df.format(contact.getBirthDate()) + "<br>");
		}
		if (contact.getLanguage() != null
				&& !contact.getLanguage().equals("")) {
			sb.append("&nbsp;&nbsp;" +Translatrix.getTranslationString("Addressbook.language") + ": " + 
			(Translatrix.hasTranslationFor("Language." + contact.getLanguage()) ? Translatrix.getTranslationString("Language." + contact.getLanguage()): contact.getLanguage()) + "<br>");
		}
		if (contact.getNationality() != null && !contact.getNationality().equals("")) {
			sb.append("&nbsp;&nbsp;" +Translatrix.getTranslationString("Addressbook.nationality") + ": " + 
			(Translatrix.hasTranslationFor("Nationality." + contact.getNationality()) ? Translatrix.getTranslationString("Nationality." + contact.getNationality()): contact.getNationality()) + "<br>");
		}	
	}
	
	infoLabel.setText(sb.toString());
    }
    

    
}
