/*******************************************************************************
 * This file is part of GECAMed.
 * 
 * GECAMed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License (L-GPL) as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * GECAMed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License (L-GPL)
 * along with GECAMed.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * GECAMed is Copyrighted by the Centre de Recherche Public Henri Tudor (http://www.tudor.lu)
 * (c) CRP Henri Tudor, Luxembourg, 2008
 *******************************************************************************/
package lu.tudor.santec.gecamed.address.ejb.entity.beans;

import java.io.Serializable;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Table;
import javax.persistence.Transient;

import lu.tudor.santec.gecamed.core.ejb.entity.beans.GECAMedEntityBean;
import lu.tudor.santec.gecamed.core.utils.AutoTranslatable;
import lu.tudor.santec.i18n.Translatrix;


@javax.persistence.NamedQueries({

	@javax.persistence.NamedQuery(name="findAllCountry",
		query="SELECT OBJECT(o) FROM Country o"),
	@javax.persistence.NamedQuery(name="getCountryByName",
		query="SELECT OBJECT(o) FROM Country o WHERE lower(o.name) = lower(:name)"),
    @javax.persistence.NamedQuery(name="findAllCountryByName",
        query="SELECT OBJECT(o) FROM Country o WHERE o.name LIKE :name")
})

/**
 * Entity Bean that maps the <b>address.country</b> table
 * which stores a list of all available coutries
 * 
 * @author Johannes Hermen johannes.hermen(at)tudor.lu
 *
 * @Version
 * <br>$Log: Country.java,v $
 * <br>Revision 1.11  2012-11-30 08:01:00  ferring
 * <br>postal code added to country and to formatter
 * <br>Short patient title corrected and long patient title added
 * <br>
 * <br>Revision 1.10  2010-04-27 14:56:21  mack
 * <br>Cleanup and completion of country translations
 * <br>
 * <br>Revision 1.9  2008-09-25 09:42:27  heinemann
 * <br>fixed copyrights
 * <br>
 * <br>Revision 1.8  2008-02-06 16:25:49  mack
 * <br>Fixed french translation of GERMAN country
 * <br>
 * <br>Revision 1.7  2008-01-15 09:29:39  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 * <br>Revision 1.6  2007-12-12 15:10:23  hermen
 * <br>updated Javadoc and refactured code
 * <br>
 *
 */
@Entity
@Table(name="country", schema="address")
public class Country extends GECAMedEntityBean implements AutoTranslatable, Serializable {

	// EU member states
	
	public static final String AUSTRIA			= "austria";
	public static final String BELGUIM 			= "belguim";
	public static final String BULGARIA 		= "bulgaria";
	public static final String CYPRUS 			= "cyprus";
	public static final String CZECH 			= "czech";	
	public static final String DENMARK 			= "denmark";
	public static final String ESTONIA 			= "estonia";
	public static final String FINLAND 			= "finland";
	public static final String FRANCE 			= "france";
	public static final String GERMANY 			= "germany";
	public static final String GREECE 			= "greece";
	public static final String HUNGARY 			= "hungary";
	public static final String IRELAND 			= "ireland";
	public static final String ITALY 			= "italy";
	public static final String LATVIA 			= "latvia";
	public static final String LITHUANIA 		= "lithuania";
	public static final String LUXEMBOURG 		= "luxembourg";
	public static final String MALTA	 		= "malta";
	public static final String NETHERLANDS 		= "netherlands";
	public static final String POLAND	 		= "poland";
	public static final String PORTUGAL 		= "portugal";
	public static final String ROMANIA	 		= "romania";
	public static final String SLOVAKIA 		= "slovakia";
	public static final String SLOVENIA 		= "slovenia";
	public static final String SPAIN	 		= "spain";
	public static final String SWEDEN	 		= "sweden";
	public static final String UNITED_KINGDOM	= "united_kingdom";
	public static final String ENGLAND 			= "england";	// DEPRECATED
	
	// Other european countries

	public static final String ALBANIA				= "albania";
	public static final String ANDORRA				= "andorra";
	public static final String ARMENIA				= "armenia";
	public static final String AZERBAIJAN			= "azerbaijan";
	public static final String BELARUS				= "belarus";
	public static final String BOSNIA_HERZEGOVINA	= "bosnia_herzegovina";
	public static final String CROATIA				= "croatia";
	public static final String GEORGIA				= "georgia";
	public static final String ICELAND				= "iceland";
	public static final String LICHTENSTEIN			= "lichtenstein";
	public static final String MACEDONIA			= "macedonia";
	public static final String MOLDOVA				= "moldova";
	public static final String MONACO				= "monaco";
	public static final String MONTENEGRA			= "montenegro";
	public static final String NORWAY				= "norway";
	public static final String SAN_MARINO			= "san_marino";
	public static final String SERBIA				= "serbia";
	public static final String SWITZERLAND			= "switzerland";
	public static final String TURKEY				= "turkey";
	public static final String UKRAINE				= "ukraine";
	
	public static final String DEFAULT		= "default";
	
	/**
	 * This is the hashmap for the country names for addresses abroad.
	 * They are unique for luxembourg.
	 * If the software is going to be used in other countries, this should be
	 * adapted
	 */
//	public static final HashMap<String, HashMap<String, String>> addressTranslation = new HashMap<String, HashMap<String,String>>();
//	{
//		/* ================================================== */
//		// the luxembourgish translation set
//		HashMap<String, String> luxembourgTranslations = new HashMap<String, String>();
//		luxembourgTranslations.put(LUXEMBOURG, 	"Luxembourg");
//		luxembourgTranslations.put(GERMANY, 	"Allemagne");
//		luxembourgTranslations.put(FRANCE, 		"France");
//		luxembourgTranslations.put(ENGLAND, 	"Angleterre");
//		luxembourgTranslations.put(BELGUIM, 	"Belgique");
//		luxembourgTranslations.put(NETHERLANDS, "Pays-Bas");
//		luxembourgTranslations.put(PORTUGAL, 	"Portugal");
//		luxembourgTranslations.put(ITALY, 	 	"Italie");
//		/* ------------------------------------------------------- */
//		addressTranslation.put(LUXEMBOURG, luxembourgTranslations);
//		
//		// the default translation set
//		// it is taken from the iso standard country codes
//		HashMap<String, String> defaultTranslations = new HashMap<String, String>();
//		defaultTranslations.put(LUXEMBOURG, 	"Luxembourg");
//		defaultTranslations.put(GERMANY, 		"Germany");
//		defaultTranslations.put(FRANCE, 		"France");
//		defaultTranslations.put(ENGLAND, 		"United Kingdom");
//		defaultTranslations.put(BELGUIM, 		"Belgium");
//		defaultTranslations.put(NETHERLANDS, 	"Netherlands");
//		defaultTranslations.put(PORTUGAL, 		"Portugal");
//		defaultTranslations.put(ITALY, 	 		"Italy");
//		/* ------------------------------------------------------- */
//		addressTranslation.put(DEFAULT, defaultTranslations);
//		/* ================================================== */
//	}
	
	
	private static final long serialVersionUID = 1L;

	private String code;
	private String name;
	private String postalCode;
	private String alpha3Code;

	/**
	 * @return Returns the country-code.
	 */
	@javax.persistence.Column(name = "code")
	public String getCode() {
		return code;
	}

	/**
	 * @param code The country-code to set.
	 */
	public void setCode(String code) {
		if (code != null && code.length() > 2)
			this.code	= code.substring(0, 2);
		else
			this.code = code;
	}

	/**
	 * @return Returns the name of the country.
	 */
	@javax.persistence.Column(name = "name")
	public String getName() {
		return name;
	}

	/**
	 * @param name The name of the coutry to set.
	 */
	public void setName(String name) {
		this.name = name;
	}
	
	@Column(name = "postal_code")
	public String getPostalCode ()
	{
		return this.postalCode;
	}
	
	public void setPostalCode (String code)
	{
		this.postalCode = code;
	}
	
	
	@Column(name = "iso3166_alpha3")
	public String getAlpha3Code() 
	{
		return alpha3Code;
	}
	
	
	public void setAlpha3Code(String alpha3Code) 
	{
		if (alpha3Code != null && alpha3Code.length() > 3)
			this.alpha3Code	= alpha3Code.substring(0, 3);
		else
			this.alpha3Code = alpha3Code;
	}

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return this.getName();
    }

 //---------------------------------------------------------------------------
/**
 * Translates the name of the foreign country using the language specified
 * for the office country.
 * 
 * @param p_OfficeCountry
 *            specifies the country whose language ought to be used to
 *            translate name of foreign country.
 * @param p_ForeignCountry
 *            specifies the foreign country to get translation for.
 * @return translated name of foreign country.
 */
//---------------------------------------------------------------------------
    
    @Transient
    public String translate () 
    	{
    	String	l_Name;
    	String 	l_TranslationKey;
    	String 	l_Translation;
    	
    	l_Name = this.getName();
    	
    	if (l_Name != null)
    		{
    		l_TranslationKey = "Country." + l_Name.toLowerCase();
    		l_Translation    = Translatrix.getTranslationString(l_TranslationKey);
    	
    		if (l_TranslationKey.equals(l_Translation))
    			{
    			l_Translation = l_Name;
    			}
    		}
    	else l_Translation = l_Name;
    	
    	return l_Translation;
        }
    
    
	@Transient
	public String getTranslationKey ()
	{
		return getClass().getSimpleName() + "." + getName();
	}
	
	
	@Transient
	public String getDefaultTranslation ()
	{
		return getName();
	}
    
}
