//===============================================================
// package : com.tetrasix.util
// class : com.tetrasix.util.Configuration
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */

package com.tetrasix.util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Locale;
import java.util.Properties;

import com.jclark.xml.parse.EndElementEvent;
import com.jclark.xml.parse.EntityManagerImpl;
import com.jclark.xml.parse.OpenEntity;
import com.jclark.xml.parse.StartElementEvent;
import com.jclark.xml.parse.io.ApplicationImpl;
import com.jclark.xml.parse.io.Parser;
import com.jclark.xml.parse.io.ParserImpl;

public class Configuration {

	static Properties _properties;
    static Hashtable  _propertylists;
	static String     _filename;
   
	public static void init(String app, boolean load)
	{
		_filename = getConfigurationFileName(app);
		if (load) {
			reload();
		}
		else {
			Configuration ppl = new Configuration();
			_properties = new Properties();
            _propertylists = new Hashtable();
		}
	}

	public static String getConfigurationFileName(String app)
	{
		String filesep  = System.getProperty("file.separator");
		String tempdir  = System.getProperty("java.io.tmpdir");
		String appdir   = System.getProperty(app + ".dir", tempdir);
		String filename = System.getProperty(app + ".conf", "majix.conf");

		if (! new File(appdir + filesep + filename).canRead()) {
			try {
				InputStream fis = Configuration.class.getResourceAsStream("majix.conf");
				OutputStream fos = new FileOutputStream(tempdir + filesep + "majix.conf");
				byte[] buf = new byte[1024];
			    int i = 0;
			    while((i=fis.read(buf))!=-1) {
			      fos.write(buf, 0, i);
			      }
			    fis.close();
			    fos.close();
			} catch (Exception e) {
				e.printStackTrace();
			}
			return tempdir + filesep + "majix.conf";
		}
		
		return appdir + filesep + filename;
	}

	public static void reload()
	{
		Configuration ppl = new Configuration();
		ppl.LoadProperties(_filename);
	}

	public static void save()
	{
		SaveProperties(_filename);
	}

	public static String getProperty(String key)
	{
		String prop = _properties.getProperty(key);
		return (prop == null) ? "" : prop;
	}

	public static String getProperty(String key, String defaultvalue)
	{
		String prop = _properties.getProperty(key);
		return (prop == null) ? defaultvalue : prop;
	}

    public static Properties getProperties()
    {
        return _properties;
    }

    public static Properties getProperties(String key)
    {
        if ((key == null) || key.equals("")) {
            return _properties;
        }
        else {
            return (Properties) _propertylists.get(key);
        }
    }

	public static void setProperty(String name, String value)
	{
		_properties.remove(name);
		_properties.put(name, value);
	}

	private void LoadProperties(String filename)
	{
		try {
			_properties = new Properties();
            _propertylists = new Hashtable();

			Parser parser = new ParserImpl();
			OpenEntity entity = EntityManagerImpl.openFile(filename);

			parser.setApplication(new ConfigurationLoaderApplication());
			parser.parseDocument(entity);

			SaveProperties(filename + ".save");
		}
		catch (IOException e) {
			System.out.println("Error when loading the configuration file : " + e.getMessage());
		}
	}

	private static void SaveProperties(String filename)
	{
		try {
			OutputStream ostream = new FileOutputStream(filename);
			ISO8859XMLWriter writer = new ISO8859XMLWriter(ostream, ISO8859XMLWriter.MINIMIZE_EMPTY_ELEMENTS);

			writer.markup("<?xml version=\"1.0\" encoding=\"ISO-8859-1\" standalone=\"yes\"?>");writer.write("\n");
			writer.markup("<!DOCTYPE configuration [");writer.write("\n");
			writer.markup("<!ELEMENT configuration (entry+)>");writer.write("\n");
			writer.markup("<!ELEMENT entry EMPTY>");writer.write("\n");
			writer.markup("<!ATTLIST entry ");writer.write("\n");
			writer.markup("          key  CDATA #REQUIRED");writer.write("\n");
			writer.markup("          value CDATA #IMPLIED>");writer.write("\n");
			writer.markup("]>");writer.write("\n");

			writer.startElement("configuration");writer.write("\n");

			for (Enumeration e = _properties.propertyNames(); e.hasMoreElements(); ) {
				String name = e.nextElement().toString();
				writer.startElement("entry");
				writer.attribute("key", name);
				writer.attribute("value", _properties.getProperty(name));
				writer.endElement("entry");
				writer.write("\n");
			}
			
            for (Enumeration e = _propertylists.keys(); e.hasMoreElements(); ) {
                String name = e.nextElement().toString();
                Properties properties = (Properties) _propertylists.get(name);
                writer.startElement("list");
                writer.attribute("key", name);
                writer.write("\n");
                for (Enumeration e1 = properties.propertyNames(); e1.hasMoreElements(); ) {
                    String name1 = e1.nextElement().toString();
                    writer.startElement("entry");
                    writer.attribute("key", name1);
                    writer.attribute("value", properties.getProperty(name1));
                    writer.endElement("entry");
                    writer.write("\n");
                }
                
                writer.endElement("list");
                writer.write("\n");
            }
            
			writer.endElement("configuration");writer.write("\n");

			writer.flush();
			ostream.close();
		}
		catch(IOException e) {
			System.out.println("IO error : " + e.getMessage());
		}
	}

	static public void setDefaultLocale(String arg[])
	{
		String next = null;
		Locale loc = Locale.getDefault();
		String lang = loc.getLanguage();
		String country = loc.getCountry();
		boolean locale_changed = false;

		if (! Configuration.getProperty("language").equals("")) {
			lang = Configuration.getProperty("language");
			locale_changed = true;
		}
		if (! Configuration.getProperty("country").equals("")) {
			country = Configuration.getProperty("country");
			locale_changed = true;
		}

		for (int ii = 0; ii < arg.length; ii++) {

			if (next != null) {
				if (next.equals("lang")) {
					lang = arg[ii];
					next = null;
					locale_changed = true;
				}
				else if (next.equals("country")) {
					country = arg[ii];
					next = null;
					locale_changed = true;
				}
			}
			else {
				if (arg[ii].equals("-language")) {
					next = "lang";
				}
				else if (arg[ii].equals("-country")) {
					next = "country";
				}
			}
		}


		if (locale_changed) {
			Locale.setDefault(new Locale(lang, country));
		}
	}

    static public int getTemplateCount()
    {
        int count = Integer.parseInt(_properties.getProperty("majix.templates.count", "0"));
        return count;
    }

    static private String getTemplatePropertyValue(int idx, String key)
    {
        String listkey = "majix.templates." + String.valueOf(idx);
        Properties properties = (Properties) _propertylists.get(listkey);
        if (properties != null) {
            return properties.getProperty(key);
        }
        else {
            return null;
        }
    }

    static public String getTemplateName(int idx)
    {
        return getTemplatePropertyValue(idx, "name");
    }

    static public String getTemplatePath(int idx)
    {
        return getTemplatePropertyValue(idx, "path");
    }

    static public String getTemplatePath(String name)
    {
        int count = getTemplateCount();
        for (int ii = 0; ii < count; ii++) {
            String listkey = "majix.templates." + String.valueOf(ii);
            Properties properties = (Properties) _propertylists.get(listkey);
            if (properties.getProperty("name").equals(name)) {
                return properties.getProperty("path");
            }
        }
        
        return "";
    }

    static public void clearTemplates()
    {
        int count = getTemplateCount();
        for (int ii = 0; ii < count; ii++) {
            String listkey = "majix.templates." + String.valueOf(ii);
            _propertylists.remove(listkey);
        }
        // put should be replaced by putProperty in JDK 1.2
        _properties.put("majix.templates.count", "0");
    }

    static public void addTemplate(String name, String path)
    {
        Properties properties = new Properties();
        properties.put("name", name);
        properties.put("path", path);
        String listkey = "majix.templates." + String.valueOf(getTemplateCount());
        _propertylists.put(listkey, properties);
        // put should be replaced by putProperty in JDK 1.2
        _properties.put("majix.templates.count",
                         String.valueOf(getTemplateCount() + 1));
    }

	private class ConfigurationLoaderApplication extends ApplicationImpl {
        
        private Properties _listProperty;
        private String     _listPropertyKey;

        public void startDocument() throws IOException
        {
            _listProperty = null;
        }

        public void startElement(StartElementEvent event) throws IOException
        {
            if (event.getName().equals("entry")) {
                String key = event.getAttributeValue("key");
                String value = event.getAttributeValue("value");

                if (value != null) {
                    value = value.trim();
                }

                if ((value != null) && (! value.equals(""))) {
                    if (_listProperty != null) {
                        _listProperty.put(key, value);
                    }
                    else {
                        _properties.put(key, value);
                    }
                }
            }
            else if (event.getName().equals("list")) {
                String key = event.getAttributeValue("key");
                _listProperty = new Properties();
                _propertylists.put(key, _listProperty);
            }
        }
        
        public void endElement(EndElementEvent event) throws IOException
        {
            if (event.getName().equals("list")) {
                _listProperty = null;                
            }
        }
	}
}