//===============================================================
// package : com.tetrasix.majix.xml
// class : com.tetrasix.majix.XmlTagMap
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */


package com.tetrasix.majix.xml;

import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import com.jclark.xml.output.XMLWriter;
import com.jclark.xml.parse.StartElementEvent;

public class XmlTagMap implements Cloneable {

	private Hashtable _map    = new Hashtable();
	private Vector    _vector = new Vector();

	public static final int TC_DEFAULT       = 0;
	public static final int TC_ROOT          = 1;
	public static final int TC_INFO          = 2;
	public static final int TC_SECTION       = 3;
	public static final int TC_BODY          = 4;
	public static final int TC_TABLE         = 5;
	public static final int TC_INLINE        = 6;
	public static final int TC_COLOR         = 7;
	public static final int TC_CHARPROP      = 8;
	public static final int TC_ENTITIES      = 9;
	public static final int TC_SPECIAL       = 10;
	public static final int TC_MISC          = 11;

	public static final int TC_LAST_CODE     = 11;

	public static final int TC_TYPE_UNDEF    = 0;
	public static final int TC_TYPE_TAG      = 1;
	public static final int TC_TYPE_TAG2     = 2;
	public static final int TC_TYPE_TAG3     = 3;
	public static final int TC_TYPE_ATTR     = 4;
	public static final int TC_TYPE_BOOL     = 5;
	public static final int TC_TYPE_ENT      = 6;
	public static final int TC_TYPE_SPECIAL  = 7;

	public static final int TC_LAST_TYPE     = 7;

	public XmlTagMap()
	{
	}

	protected Object clone()
	{
		XmlTagMap obj = new XmlTagMap();

		obj._map = (Hashtable) _map.clone();
		obj._vector = (Vector) _vector.clone();

		return obj;
	}

	public void Clear()
	{
		for (Enumeration e = _map.elements(); e.hasMoreElements(); ) {
			XmlTagMapEntry entry = (XmlTagMapEntry) e.nextElement();
			entry._actual_tag = "";
			entry._attribute = "";
		}
	}

	public void add(int category, int type,
					String default_tag, String description,
					String actual_tag, String attributes, boolean special)
	{
		_map.put(default_tag,
				 new XmlTagMapEntry(category, type, default_tag, description, actual_tag, attributes, special));
		_vector.addElement(default_tag);
	}

	public void add(int category, int type, 
					String default_tag, String description,
					String actual_tag, String attributes)
	{
		_map.put(default_tag,
				 new XmlTagMapEntry(category, type, default_tag, description, actual_tag, attributes));
		_vector.addElement(default_tag);
	}

	public void add(int category, int type, 
					String default_tag, String description,
					String actual_tag, String attributes,
					String description2, String attribute2)
	{
		_map.put(default_tag,
				 new XmlTagMapEntry(category, type, 
									default_tag, description,
									actual_tag, attributes,
									description2, attribute2));
		_vector.addElement(default_tag);
	}

	public void add(int category, int type, 
					String default_tag, String description,
					String actual_tag, String attributes,
					String description2, String attribute2,
					String description3, String attribute3)
	{
		if (description3 != null) {
			_map.put(default_tag,
					 new XmlTagMapEntry(category, type, 
										default_tag, description,
										actual_tag, attributes,
										description2, attribute2,
										description3, attribute3));
		}
		else {
			_map.put(default_tag,
					 new XmlTagMapEntry(category, type, 
										default_tag, description,
										actual_tag, attributes,
										description2, attribute2));
		}
		_vector.addElement(default_tag);
	}

	public void add(int category, int type, 
					String default_tag, String description, 
					String actual_tag)
	{
		add(category, type, default_tag, description, actual_tag, null);
	}

	public void add(int category, int type, String default_tag,
					String description)
	{
		add(category, type, default_tag, description, null);
	}

	/**
	 * Method to determine if a tag-element is defined in the used template.
	 * @param tag  the tag to be examined
	 * @return true if the tag is defined in the template
	 */
	public boolean isDefined(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		return entry != null;
	}

	public void setActualTag(String default_tag, String actual_tag)
	{
		XmlTagMapEntry entry = getEntry(default_tag);
		if (entry != null) {
			entry._actual_tag = actual_tag;
		}
	}

	public String getActualTag(String default_tag, boolean with_attribute)
	{
		XmlTagMapEntry entry = getEntry(default_tag);
		if (entry != null) {
			String tag = entry._actual_tag;
			if (tag == null) {
				tag = default_tag;
			}
			if (with_attribute 
					&& (entry._attribute != null)
					&& (! entry._attribute.equals(""))
					) {
				tag += " " + entry._attribute;
			}
			return tag;
		}
		else {
			return default_tag;
		}
	}

	public String getDescription(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry._description;
		}
		else {
			return null;
		}
	}

	public String getDescription(int idx)
	{
		XmlTagMapEntry entry = getEntry(idx);
		if (entry != null) {
			return entry._description;
		}
		else {
			return null;
		}
	}

	public int getCategory(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry._category;
		}
		else {
			return TC_SPECIAL;
		}
	}

	public int getCategory(int idx)
	{
		XmlTagMapEntry entry = getEntry(idx);
		if (entry != null) {
			return entry._category;
		}
		else {
			return TC_SPECIAL;
		}
	}

	public int getType(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry._type;
		}
		else {
			return TC_TYPE_UNDEF;
		}
	}

	public boolean getSpecial(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry.getSpecial();
		}
		else {
			return false;
		}
	}

	public String getAttributes(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry.getAttribute();
		}
		else {
			return null;
		}
	}

	public String getAttribute2(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry.getAttribute2();
		}
		else {
			return null;
		}
	}

	public String getDescription2(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry.getDescription2();
		}
		else {
			return null;
		}
	}

	public String getAttribute3(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry.getAttribute3();
		}
		else {
			return null;
		}
	}

	public String getDescription3(String tag)
	{
		XmlTagMapEntry entry = getEntry(tag);
		if (entry != null) {
			return entry.getDescription3();
		}
		else {
			return null;
		}
	}

	public void setAttributes(String default_tag, String attrs)
	{
		XmlTagMapEntry entry = getEntry(default_tag);
		if (entry != null) {
			entry._attribute = attrs;
			entry._description2 = null;
			entry._attribute2 = null;
			entry._description3 = null;
			entry._attribute3 = null;
		}
	}

	public void setAttributes(String default_tag, String attrs,
							  String descr2, String attr2)
	{
		XmlTagMapEntry entry = getEntry(default_tag);
		if (entry != null) {
			entry._attribute = attrs;
			entry._description2 = descr2;
			entry._attribute2 = attr2;
			entry._description3 = null;
			entry._attribute3 = null;
		}
	}

	public void setAttributes(String default_tag, String attrs,
							  String descr2, String attr2,
							  String descr3, String attr3)
	{
		XmlTagMapEntry entry = getEntry(default_tag);
		if (entry != null) {
			entry._attribute = attrs;
			entry._description2 = descr2;
			entry._attribute2 = attr2;
			entry._description3 = descr3;
			entry._attribute3 = attr3;
		}
	}

	private XmlTagMapEntry getEntry(String key)
	{
		return (XmlTagMapEntry) _map.get(key);
	}

	private XmlTagMapEntry getEntry(int idx)
	{
		if (idx != -1) {
			String key = (String) _vector.elementAt(idx);
			return (XmlTagMapEntry) _map.get(key);
		}
		else {
			return null;
		}
	}

	public Enumeration getTags()
	{
		return _vector.elements();
	}

	public void load(StartElementEvent event, boolean allow_creation)
	{
		String category     = event.getAttributeValue("category");
		String type         = event.getAttributeValue("type");
		String id           = event.getAttributeValue("id");
		String tag          = event.getAttributeValue("tag");
		String attrs        = event.getAttributeValue("attrs");
		String attr2        = event.getAttributeValue("attr2");
		String attr3        = event.getAttributeValue("attr3");
		String special      = event.getAttributeValue("special");
		String description  = event.getAttributeValue("description");
		String description2 = event.getAttributeValue("description2");
		String description3 = event.getAttributeValue("description3");
		String atag         = event.getAttributeValue("atag");

		XmlTagMapEntry entry = getEntry(id);
		if (entry != null || allow_creation) {
			if (entry == null) {
				entry = new XmlTagMapEntry();
				_map.put(id, entry);
				_vector.addElement(id);
			}
			entry._category = CategoryMap.getCategoryCode(category);
			if (entry._category == 0) {
				System.out.println("Unknown tag category : " + category);
			}
			if (type != null) {
				entry._type = TypeMap.getTypeCode(type);
			}
			if (allow_creation) {
				entry._actual_tag = atag;
			}
			else {
				entry._actual_tag = tag;
			}
			entry._attribute = attrs;
			if (attr2 != null) {
				entry._attribute2 = attr2;
			}
			else {
				entry._attribute2 = "";
			}
			if (attr3 != null) {
				entry._attribute3 = attr3;
			}
			else {
				entry._attribute3 = "";
			}
			if (description != null) {
				entry._description = description;
			}
			if (description2 != null) {
				entry._description2 = description2;
			}
			if (description3 != null) {
				entry._description3 = description3;
			}
			if ((special != null) && (special.equals("true"))) {
				entry._special = true;
			}
			else {
				entry._special = false;
			}
		}
		else {
			System.out.println("Unknown tag : " + tag);
		}
	}

	public void Save(XMLWriter writer, boolean with_description) throws IOException
	{
		writer.write("\n");
		writer.startElement("tagmap");
		
		for (Enumeration e = _vector.elements(); e.hasMoreElements(); ) {
			String tagname = e.nextElement().toString();
			XmlTagMapEntry entry = (XmlTagMapEntry) _map.get(tagname);
			
			writer.write("\n  ");
			writer.startElement("tag");
			writer.attribute("category", CategoryMap.getCategoryName(entry._category));
			writer.attribute("type", TypeMap.getTypeName(entry._type));
			if (entry._default_tag != null) {
				writer.attribute("id", entry._default_tag);
			}
			writer.attribute("tag",
				(entry._actual_tag == null)
							? entry._default_tag
							: entry._actual_tag);
			if (with_description && entry._actual_tag != null) {
				writer.attribute("atag", entry._actual_tag);
			}
			if (with_description
					&& entry._description != null 
					&& ! entry._description.equals("")) {
				writer.attribute("description", entry._description);
			}
			writer.attribute("attrs", 
				(entry._attribute == null)
							? ""
							: entry._attribute);
			if (with_description
					&& entry._description2 != null 
					&& ! entry._description2.equals("")) {
				writer.attribute("description2", entry._description2);
			}
			if (entry._attribute2 != null && ! entry._attribute2.equals("")) {
				writer.attribute("attr2", entry._attribute2);
			}
			if (with_description
					&& entry._description3 != null 
					&& ! entry._description3.equals("")) {
				writer.attribute("description3", entry._description3);
			}
			if (entry._attribute3 != null && ! entry._attribute3.equals("")) {
				writer.attribute("attr3", entry._attribute3);
			}
			if (entry._special) {
				writer.attribute("special", "true");
			}
			writer.endElement("tag");
		}

		writer.write("\n");
		writer.endElement("tagmap");
	}

	static class CategoryMap  {

		static String getCategoryName(int code)
		{
			if (code > TC_LAST_CODE) {
				code = 0;
			}
			return categoryNames[code];
		}

		static int getCategoryCode(String name)
		{
			for (int ii = 0; ii <= TC_LAST_CODE; ii++) {
				if (name.equals(categoryNames[ii])) {
					return ii;
				}
			}

			return 0;
		}

		static final String[] categoryNames = {
			"default",      // TC_DEFAULT       = 0
			"root",			// TC_ROOT          = 1
			"info",			// TC_INFO          = 2
			"section",		// TC_SECTION       = 3
			"body",			// TC_BODY          = 4
			"table",		// TC_TABLE         = 5
			"inlline",		// TC_INLINE        = 6
			"color",		// TC_COLOR         = 7
			"charprop",		// TC_CHARPROP      = 8
			"entities",		// TC_ENTITIES      = 9
			"special",		// TC_SPECIAL       = 10
			"misc",		    // TC_MISC          = 11
		};
	}

	static class TypeMap  {

		static String getTypeName(int type)
		{
			if (type > TC_LAST_TYPE) {
				type = 0;
			}
			return typeNames[type];
		}

		static int getTypeCode(String name)
		{
			for (int ii = 0; ii <= TC_LAST_TYPE; ii++) {
				if (name.equals(typeNames[ii])) {
					return ii;
				}
			}

			return 0;
		}

		static final String[] typeNames = {
			
			"undef",        // TC_TYPE_UNDEF    = 0
			"tag",          // TC_TYPE_TAG      = 1
			"tag2",			// TC_TYPE_TAG2     = 2
			"tag3",			// TC_TYPE_TAG3     = 3
			"attr",			// TC_TYPE_ATTR     = 4
			"bool",			// TC_TYPE_BOOL     = 5
			"ent",			// TC_TYPE_ENT      = 6
			"special",		// TC_TYPE_SPECIAL  = 7
		};
	}

	static class XmlTagMapEntry {

		int     _category;
		int     _type;
		String  _default_tag;
		String  _description;
		String  _actual_tag;
		String  _attribute;
		String  _description2;
		String  _attribute2;
		String  _description3;
		String  _attribute3;
		boolean _special;

		XmlTagMapEntry()
		{
			_category     = 0;
			_type         = 0;
			_default_tag  = null;
			_description  = null;
			_actual_tag   = null;
			_attribute    = null;
			_description2 = null;
			_attribute2   = null;
			_description3 = null;
			_attribute3   = null;
			_special      = false;
		}

		XmlTagMapEntry(int category, int type, String default_tag,
					   String description,
					   String actual_tag, String attribute,
					   String description2, String attribute2,
					   String description3, String attribute3)
		{
			_category     = category;
			_type         = type;
			_default_tag  = default_tag;
			_description  = description;
			_actual_tag   = actual_tag;
			_attribute    = attribute;
			_description2 = description2;
			_attribute2   = attribute2;
			_description3 = description3;
			_attribute3   = attribute3;
			_special      = false;
		}

		XmlTagMapEntry(int category, int type, String default_tag, String description,
					   String actual_tag, String attribute,
					   String description2, String attribute2)
		{
			_category     = category;
			_type         = type;
			_default_tag  = default_tag;
			_description  = description;
			_actual_tag   = actual_tag;
			_attribute    = attribute;
			_description2 = description2;
			_attribute2   = attribute2;
			_description3 = "";
			_attribute3   = "";
			_special      = false;
		}

		XmlTagMapEntry(int category, int type,
					   String default_tag, String description,
					   String actual_tag, String attribute)
		{
			_category     = category;
			_type         = type;
			_default_tag  = default_tag;
			_description  = description;
			_actual_tag   = actual_tag;
			_attribute    = attribute;
			_description2 = "";
			_attribute2   = "";
			_description3 = "";
			_attribute3   = "";
			_special      = false;
		}

		XmlTagMapEntry(int category, int type,
					   String default_tag, String description,
					   String actual_tag, String attribute, boolean special)
		{
			_category     = category;
			_type         = type;
			_default_tag  = default_tag;
			_description  = description;
			_actual_tag   = actual_tag;
			_attribute    = attribute;
			_description2 = "";
			_attribute2   = "";
			_description3 = "";
			_attribute3   = "";
			_special      = special;
		}

		String getDefaultTag()
		{
			return _default_tag;
		}

		String getActualTag()
		{
			return _actual_tag;
		}

		String getAttribute()
		{
			return _attribute;
		}

		String getDescription()
		{
			return _description;
		}

		String getAttribute2()
		{
			return _attribute2;
		}

		String getDescription2()
		{
			return _description2;
		}

		String getAttribute3()
		{
			return _attribute3;
		}

		String getDescription3()
		{
			return _description3;
		}

		boolean getSpecial()
		{
			return _special;
		}
	}
}
