//===============================================================
// package : com.tetrasix.majix.uis
// class : com.tetrasix.majix.uis.ConverterFrame
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.2
 */


package com.tetrasix.majix.uis;

import java.awt.AWTEvent;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.SystemColor;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.ResourceBundle;

import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.filechooser.FileFilter;

import com.tetrasix.majix.rtf.RtfStyleMap;
import com.tetrasix.majix.templates.ConversionTemplateFactory;
import com.tetrasix.majix.xml.ConversionTemplate;
import com.tetrasix.majix.xml.XmlGeneratorParam;
import com.tetrasix.util.Alerter;
import com.tetrasix.util.Configuration;
import com.tetrasix.util.Logo;
import com.tetrasix.util.LogoTetraSix;
import com.tetrasix.util.MessageBox;
import com.tetrasix.util.MessageSink;

public class ConverterFrame extends JFrame
		implements ActionListener, ItemListener, FocusListener, KeyListener, MajixEventListener, MessageSink {

	private JTextField _rtfTextField;
	private JTextField _xmlTextField;
	private JTextField _transTextField;

	private JButton _convertButton;
	private JButton _editStylesButton;
	private JButton _editTagsButton;
	private JButton _quitButton;
	private JButton _aboutButton;
	private JButton _helpButton;
	private JButton _findDocumentButton;
	private JButton _findDirectoryButton;
	private JButton _findFileButton;
	private JButton _toolsButton;

	private JCheckBox _dumpBox;
	private JCheckBox _debugBox;

	private JLabel _status;
	private JPanel _statusPanel;

	private JComboBox _templateChoice;

	private ConversionTemplate _template;

	private boolean _development_environment = false;
	private boolean _multiple_translations = true;
    
	ResourceBundle _res = null;

	static private final String _HELP_TOPIC = "converter_frame";

	public ConverterFrame()
	{
        LFlist.add(this);
		enableEvents(AWTEvent.WINDOW_EVENT_MASK);
		setBackground(SystemColor.control);
	}

	public void init()
	{
		Alerter.initInteractive(this);

		_res = 
			ResourceBundle.getBundle("com.tetrasix.majix.uis.ConverterFrameResources");

		setTitle(_res.getString("frameTitle"));

		Box p0 = Box.createVerticalBox();
        
        Box box = Box.createHorizontalBox();
        box.add(new LogoTetraSix());
        box.add(Box.createHorizontalGlue());
        box.add(new Logo("com/tetrasix/util/LogoMajix.gif", null, "majix"));
        p0.add(box);

		JPanel p1 = new JPanel();
        p1.setLayout(new GridBagLayout());
        
		addComponent(p1, new JLabel(_res.getString("doc_text_label")),
                     0, 0, 1, 1,
                     GridBagConstraints.NONE, GridBagConstraints.WEST);
        addComponent(p1, _rtfTextField = new JTextField(48),
                     1, 0, 1, 1,
                     GridBagConstraints.NONE, GridBagConstraints.WEST);
		_rtfTextField.addActionListener(this);
		_rtfTextField.addFocusListener(this);
		_rtfTextField.addKeyListener(this);
        addComponent(p1, _findDocumentButton = new JButton(_res.getString("find_document_button")),
                     2, 0, 1, 1,
                     GridBagConstraints.NONE, GridBagConstraints.WEST);
		_findDocumentButton.setActionCommand("find_document");
		_findDocumentButton.addActionListener(this);

        addComponent(p1, new JLabel(_res.getString("xml_text_label")),
                     0, 1, 1, 1,
                     GridBagConstraints.NONE, GridBagConstraints.WEST);
        addComponent(p1, _xmlTextField = new JTextField(48),
                     1, 1, 1, 1,
                     GridBagConstraints.NONE, GridBagConstraints.WEST);
		_xmlTextField.addActionListener(this);
		_xmlTextField.addFocusListener(this);
		_xmlTextField.addKeyListener(this);
        p0.add(p1);

		if (_multiple_translations) {
            box = Box.createHorizontalBox();
			box.add(new JLabel(_res.getString("template_choice_label")));
			_templateChoice = new JComboBox();

            int count = Configuration.getTemplateCount();
            for (int ii = 0; ii < count; ii++) {
                _templateChoice.addItem(Configuration.getTemplateName(ii));
            }
            _templateChoice.setSelectedIndex(0);

			box.add(_templateChoice);
			_templateChoice.addItemListener(this);
            box.add(Box.createHorizontalGlue());
            
            p0.add(box);
		}

		if (_development_environment) {
            box = Box.createHorizontalBox();
			box.add(_dumpBox = new JCheckBox(_res.getString("dump_box")));
			box.add(_debugBox = new JCheckBox(_res.getString("debug_box")));
            p0.add(box);
		}

        box = Box.createHorizontalBox();
		box.add(new JLabel(_res.getString("status_label")));
		box.add(_status = new JLabel(""));
        box.add(Box.createHorizontalGlue());
        p0.add(box);
		
        box = Box.createHorizontalBox();
		box.add( _convertButton = new JButton(_res.getString("convert_button")));
		_convertButton.setActionCommand("convert");
		_convertButton.addActionListener(this);

		box.add( _editStylesButton = new JButton(_res.getString("edit_styles_button")));
		_editStylesButton.addActionListener(this);
		_editStylesButton.setActionCommand("edit_styles");

		box.add( _editTagsButton = new JButton(_res.getString("edit_tags_button")));
		_editTagsButton.addActionListener(this);
		_editTagsButton.setActionCommand("edit_tags");

		box.add(	_toolsButton = new JButton(_res.getString("tools_button")));
		_toolsButton.addActionListener(this);
		_toolsButton.setActionCommand("tools");

		box.add(Box.createHorizontalStrut(20));

		if (Configuration.getProperty("majix.enable.doc").equals("1")) {
			box.add( _helpButton = new JButton(_res.getString("help_button")));
			_helpButton.addActionListener(this);
			_helpButton.setActionCommand("help");
		}
 
		box.add( _aboutButton = new JButton(_res.getString("about_button")));
		_aboutButton.addActionListener(this);
		_aboutButton.setActionCommand("about");

        box.add(Box.createHorizontalStrut(20));
		box.add( _quitButton = new JButton(_res.getString("quit_button")));
		_quitButton.addActionListener(this);
		_quitButton.setActionCommand("quit");

        p0.add(box);
        
		getContentPane().add(p0);

		File stylemapfile = new File("default.sty");
		File tagsmapfile = new File("default.tdef");

		ConversionTemplateFactory factory = new ConversionTemplateFactory();
		if (_multiple_translations) {
			String templatename = (String) _templateChoice.getSelectedItem();
			_template = factory.getConversionTemplate(Configuration.getTemplatePath(templatename));
			stylemapfile = new File(templatename + ".sty");
			tagsmapfile = new File(templatename + ".tdef");
		}
		else {
			_template = factory.getConversionTemplate("mydoc");
			stylemapfile = new File("default.sty");
			tagsmapfile = new File("default.tdef");
		}

		if (stylemapfile.exists() && stylemapfile.canRead()) {
			RtfStyleMap stylemap = new RtfStyleMap(stylemapfile.getAbsolutePath());
			_template.initRtfAbstractStyleSheet(stylemap.getRtfAbstractStylesheet());
		 	_template.setStyleMap(stylemap);
		}

		if (tagsmapfile.exists() && tagsmapfile.canRead()) {

			_template.setGeneratorParam(new XmlGeneratorParam(_template, tagsmapfile.getAbsolutePath()));
		}

		_template.setAbstractStyleProvider(new InteractiveAbstractStyleProvider(this));

		pack();

		CheckConvertButton();
	}

	protected void processWindowEvent(WindowEvent e)
	{
		if (e.getID() == WindowEvent.WINDOW_CLOSING) {
			System.exit(0);
		}
		else {
			super.processWindowEvent(e);
		}
	}

	public void focusGained(FocusEvent e)
	{
	}

	public void focusLost(FocusEvent e)
	{
		if (e.getSource() == _rtfTextField) {
			CheckConvertButton();
		}
		if (e.getSource() == _xmlTextField) {
			CheckConvertButton();
		}
	}

	public void keyPressed(KeyEvent e)
	{
	}

	public void keyReleased(KeyEvent e)
	{
		if (e.getSource() == _rtfTextField) {
			CheckConvertButton();
		}
		if (e.getSource() == _xmlTextField) {
			CheckConvertButton();
		}
	}

	public void keyTyped(KeyEvent e)
	{
	}

	public boolean checkFile(String filename)
	{
		File f = new File(filename);

		if (f.exists() && f.isFile()) {
			return true;
		}
		else {
			String msg[] = new String[2];
			msg[0] = _res.getString("no_such_file");
			msg[1] = "   " + filename;
			new MessageBox(this, msg);
			return false;
		}
	}

	public void actionPerformed(ActionEvent e)
	{
		if (e.getSource() instanceof JButton) {
			String s = e.getActionCommand();
	
			if (s.equals("quit")) {
				System.exit(0);
			}
			else if (s.equals("convert")) {
				if ( checkFile(_rtfTextField.getText())) {

					Message("Processing a MS-Word file");
					ConverterThread t = new ConverterThread(
							this, 
							this,
							_template,
							_rtfTextField.getText(),
							_xmlTextField.getText(),
							_dumpBox != null && _dumpBox.isSelected(),
							_debugBox != null && _debugBox.isSelected());
					t.start();
				}
			}
			else if (s.equals("edit_styles")) {
				RtfStylesEditor editor = new RtfStylesEditor();
				editor.Edit(_template, this);
			}
			else if (s.equals("edit_tags")) {
				XmlTagsEditor editor = new XmlTagsEditor();
				editor.Edit(_template, this);
			}
			else if (s.equals("tools")) {
				ToolsFrame toolsframe = new ToolsFrame(this);
				toolsframe.Run();
			}
			else if (s.equals("find_document")) {
				FindDocument();
			}
			else if (s.equals("about")) {
				AboutFrame.showAbout();
			}
			else if (s.equals("help")) {
				showHelp();
			}
			else {
				System.out.println("unexpected action command : " + s);
			}
		}
		else if (e.getSource() instanceof JTextField) {
			JTextField f = (JTextField) e.getSource();

			if (e.getSource() == _rtfTextField) {
				String rtf = _rtfTextField.getText().toLowerCase().trim();
				if (rtf.endsWith(".rtf") || rtf.endsWith(".doc")) {
					String xmlfilename = rtf.substring(0, rtf.length() - 4) + ".xml";
					_xmlTextField.setText(xmlfilename);
				}
			}

			CheckConvertButton();
		}
		else {
			System.out.println("unexpected action command : " + e.toString());
		}
	}

	void FindDocument()
	{
        JFileChooser chooser = new JFileChooser(System.getProperty("user.dir"));
        chooser.setDialogTitle(_res.getString("select_word"));
        FileFilter filter1 = new ExtensionFileFilter("doc", "Word files");
        FileFilter filter2 = new ExtensionFileFilter("rtf", "RTF files");
        chooser.setFileFilter(filter1);
        chooser.addChoosableFileFilter(filter2);

		if (Configuration.getProperty("majix.enable.msword").equals("1")) {
            chooser.setFileFilter(filter1);
            chooser.addChoosableFileFilter(filter2);
		}
		else {
            chooser.setFileFilter(filter2);
		}

        int option = chooser.showOpenDialog(this);
        if (option == JFileChooser.APPROVE_OPTION) {
			String docfilename = chooser.getSelectedFile().getAbsolutePath();
			_rtfTextField.setText(docfilename);
			CheckConvertButton();

			if (   docfilename.toLowerCase().endsWith(".rtf")
				|| docfilename.toLowerCase().endsWith(".doc")) {

				String xmlfilename = docfilename.substring(0, docfilename.length() - 4) + ".xml";
				_xmlTextField.setText(xmlfilename);
			}
			else {
				_xmlTextField.setText("");
			}
		}

		CheckConvertButton();
	}

	public void CheckConvertButton()
	{
		if (   (! _rtfTextField.getText().equals(""))
			&& (! _xmlTextField.getText().equals("")) ) {

			_convertButton.setEnabled(true);
		}
		else {
			_convertButton.setEnabled(false);
		}
	}

	public void Message(String msg)
	{
		_status.setText(msg);
	}

	public void MessageUsingResource(String msgrsrc)
	{
		String msg = _res.getString(msgrsrc);
		Message(msg);
	}

	public void center()
	{
		Dimension wd = getSize();
		Dimension sd = Toolkit.getDefaultToolkit().getScreenSize();

		setLocation((sd.width - wd.width) / 2, (sd.height - wd.height) / 2);
	}

	private void showHelp()
	{
		ShowHelp.load(_HELP_TOPIC);
	}

	public void enableDevelopmentEnvironment(boolean enable)
	{
		_development_environment = enable;
	}

	public void enableMultipleTranslations(boolean enable)
	{
		_multiple_translations = enable;
	}

	public void itemStateChanged(ItemEvent e)
	{
		if (e.getStateChange() == ItemEvent.SELECTED) {
			ConversionTemplateFactory factory = new ConversionTemplateFactory();
			_template = factory.getConversionTemplate((String) _templateChoice.getSelectedItem());
			_template.setAbstractStyleProvider(new InteractiveAbstractStyleProvider(this));
		}
	}

	public void onEvent(MajixEvent event)
	{
		if (event.getEventType() == MajixEvent.EVENT_UPDATE_STYLE_MAP) {
			_template.setStyleMap((RtfStyleMap)event.getSource());
		}
	}

    private static void addComponent (
        Container container, Component component, 
        int gridx, int gridy, int gridwidth, int gridheight,
        int fill, int anchor)
    {
        LayoutManager lm = container.getLayout();
        if (! (lm instanceof GridBagLayout)) {
              System.out.println("Invalid layout : " + lm);
              container.add(component);
              return;
        }
        
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = gridx;
        gbc.gridy = gridy;
        gbc.gridwidth = gridwidth;
        gbc.gridheight = gridheight;
        gbc.fill = fill;
        gbc.anchor = anchor;
        gbc.insets = new Insets(2, 2, 2, 2);
        ((GridBagLayout)lm).setConstraints(component, gbc);
        container.add(component);
    }
}