//===============================================================
// package : com.tetrasix.majix.rtf
// class : com.tetrasix.majix.RtfParagraphProperties
//===============================================================
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is TetraSys code.
//
// The Initial Developer of the Original Code is TetraSys..
// Portions created by TetraSys are 
// Copyright (C) 1998-2000 TetraSys. All Rights Reserved.
//
// Contributor(s): Arne Jans, Quinscape GmbH (www.quinscape.de)
//===============================================================


/**
 *
 * @version 1.1
 */


package com.tetrasix.majix.rtf;

import java.io.*;

public class RtfParagraphProperties implements Cloneable {
	// unordered types
	public static final int STYLE_SIMPLE = 0;
	public static final int STYLE_BULLET = 1;
	// ordered types
	public static final int STYLE_NUMBERED_DEC = 2;
	public static final int STYLE_NUMBERED_ALPHA = 3;
	public static final int STYLE_NUMBERED_ALPHA_UP = 4;
	public static final int STYLE_NUMBERED_ROMAN = 5;
	public static final int STYLE_NUMBERED_ROMAN_UP = 6;

	// attention: use the constant STYLE_* and subtract 2 because of
	// STYLE_SIMPLE and STYLE_BULLET 
	public static final String[] NUMBERTYPE_CODES = {"decimal",
										             "alphanumeric",
										             "alphanumeric_up",
										             "roman",
										             "roman_up"};

	public static final int ALIGNMENT_LEFT = 0;
	public static final int ALIGNMENT_CENTER = 1;
	public static final int ALIGNMENT_JUSTIFIED = 2;
	public static final int ALIGNMENT_RIGHT = 3;
	public static final int ALIGNMENT_DISTRIBUTED = 4;
	
	public static final String[] ALIGNMENT_CODES = {"left",
	                                                "center",
	                                                "justified",
	                                                "right",
	                                                "distributed"};
	private int _style = -1;
	private String _bullet = null;
	// HACK: non-resetted, non-inherited, non-cloned bullettext of the group pntext
	private String _bulletText = null;
	private String _textBefore = null;
	private String _textAfter = null;
	private int _numstyle = STYLE_SIMPLE;
	private int _orderingStart = 1;
	private boolean _dontShowNumbering = false;
	private boolean _pbreak = false;
	private int _alignment = ALIGNMENT_LEFT;

	private int _spaceBefore = 0;
	private int _spaceAfter = 0;
	
	private int _lineSpacing = 0;
	
	private int _leftIndent = 0;
	private int _rightIndent = 0;
	private int _firstIndent = 0;

	private RtfTabDefList _tabDefList;
	
	void setStyle(int style)
	{
		_style = style;
	}

	void setNumStyle(int numstyle)
	{
		_numstyle = numstyle;
	}

	void setOrderingStart(int start)
	{
		_orderingStart = start;
	}

	public int getOrderingStart()
	{
		return _orderingStart;
	}

	void setDontShowNumbering(boolean show)
	{
		_dontShowNumbering = show;
	}

	public boolean getDontShowNumbering()
	{
		return _dontShowNumbering;
	}

	void setLineSpacing(int spacing)
	{
		_lineSpacing = spacing;
	}

	public int getLineSpacing()
	{
		return _lineSpacing;
	}

	void setFirstIndent(int firstindent)
	{
		_firstIndent = firstindent;
	}

	public int getFirstIndent()
	{
		return _firstIndent;
	}

	void setLeftIndent(int leftindent)
	{
		_leftIndent = leftindent;
	}

	public int getLeftIndent()
	{
		return _leftIndent;
	}

	void setRightIndent(int rightindent)
	{
		_rightIndent = rightindent;
	}

	public int getRightIndent()
	{
		return _rightIndent;
	}

	public int getStyle()
	{
		return _style;
	}

	public int getNumStyle()
	{
		return _numstyle;
	}

	void setBullet(String bullet)
	{
		_bullet = bullet;
	}

	public String getBullet()
	{
		return _bullet;
	}

	void setBulletText(String bullettext)
	{
		_bulletText = bullettext;
	}

	public String getBulletText()
	{
		return _bulletText;
	}

	void setTextBefore(String textbefore)
	{
		_textBefore = textbefore;
	}

	public String getTextBefore()
	{
		return _textBefore;
	}

	void setTextAfter(String textafter)
	{
		_textAfter = textafter;
	}

	public String getTextAfter()
	{
		return _textAfter;
	}

	void setPageBreak(boolean pbreak)
	{
		_pbreak = pbreak;
	}

	public boolean getPageBreak()
	{
		return _pbreak;
	}
	
	void setSpaceBefore(int spacebefore)
	{
		_spaceBefore = spacebefore;
	}

	public int getSpaceBefore()
	{
		return _spaceBefore;
	}
	
	void setSpaceAfter(int spaceafter)
	{
		_spaceAfter = spaceafter;
	}

	public int getSpaceAfter()
	{
		return _spaceAfter;
	}
	
	/**
	 * Textalignment in this paragraph.
	 * Valid values are:
	 * <p>
	 * ALIGNMENT_LEFT<br>
	 * ALIGNMENT_CENTER<br>
	 * ALIGNMENT_JUSTIFIED<br>
	 * ALIGNMENT_RIGHT<br>
	 * ALIGNMENT_DISTRIBUTED<br>
	 * 
	 * @param value die Textausrichtung des Absatzes
	 */
	void setParagraphAlignment(int value)
	{
	    _alignment = value;
	}

	/**
	 * Returns the text-alignment of this paragraph.
	 * See also <code>setParagraphAlignment</code>.
	 * 
	 * @see    #setParagraphAlignment(int)
	 * @return the alignment
	 */
	public int getParagraphAlignment()
	{
	    return _alignment;
	}
	
	/**
	 * This method resets all known properties of the paragraph.
	 * You can set the relative default-tabsize for the next paragraph.
	 * If defaultTab is null, then the value RtfTabDef.TAB_DEFAULT_WIDTH will be used
	 * as the relative tab size.
	 * 
	 * @param defaultTab the relative default-tabsize for
	 */
	void reset(RtfTabDef defaultTab)
	{
		_style = -1;
		_bullet = null;
		_textBefore = null;
		_textAfter = null;
		_numstyle = STYLE_SIMPLE;
		_pbreak = false;
		_orderingStart = 1;
		_dontShowNumbering = false;
		_alignment = ALIGNMENT_LEFT;
		_spaceAfter = 0;
		_spaceBefore = 0;
		_lineSpacing = 0;
		_firstIndent = 0;
		_leftIndent = 0;
		_rightIndent = 0;
		// tabs
		this._tabDefList = new RtfTabDefList();
		if (defaultTab != null)
			_tabDefList.setDefaultTabDef(defaultTab);
		else
			_tabDefList.setDefaultTabDef(
			        new RtfTabDef(RtfTabDef.TABALIGN_LEFT,
			                RtfTabDef.TAB_DEFAULT_WIDTH,
			                RtfTabDef.TAB_TYPE_DEFAULT));
	}

	/**
	 * This method resets only the properties of the list-paragraphs.
	 *
	 */
	void resetList()
	{
		_bullet = null;
		_textBefore = null;
		_textAfter = null;
		_numstyle = STYLE_SIMPLE;
		_orderingStart = 1;
		_dontShowNumbering = false;
	}

	protected Object clone()
	{
		RtfParagraphProperties properties = new RtfParagraphProperties();

		properties.setStyle(this.getStyle());
		properties.setBullet(this.getBullet());
		properties.setBulletText(this.getBulletText());
		properties.setTextBefore(this.getTextBefore());
		properties.setTextAfter(this.getTextAfter());
		properties.setNumStyle(this.getNumStyle());
		properties.setOrderingStart(this.getOrderingStart());
		properties.setDontShowNumbering(this.getDontShowNumbering());
		properties.setPageBreak(this.getPageBreak());
		properties.setParagraphAlignment(this.getParagraphAlignment());
		properties.setSpaceBefore(this.getSpaceBefore());
		properties.setSpaceAfter(this.getSpaceAfter());
		properties.setFirstIndent(this.getFirstIndent());
		properties.setLeftIndent(this.getLeftIndent());
		properties.setRightIndent(this.getRightIndent());
		properties.setLineSpacing(this.getLineSpacing());
		properties.setTabDefList(this.getTabDefList());

		return properties;
	}

	void setTabDefList(RtfTabDefList newList) {
	    _tabDefList = newList;
	}
	
	public RtfTabDefList getTabDefList() {
	    return _tabDefList;
	}
	
	void Dump(PrintWriter out)
	{
		if (_alignment > 0) {
			switch (_alignment) {
				case ALIGNMENT_CENTER:
					out.print(" alignment=\"center\"");
					break;
				case ALIGNMENT_DISTRIBUTED:
					out.print(" alignment=\"distributed\"");
					break;
				case ALIGNMENT_JUSTIFIED:
					out.print(" alignment=\"justified\"");
					break;
				case ALIGNMENT_LEFT:
					out.print(" alignment=\"left\"");
					break;
				case ALIGNMENT_RIGHT:
					out.print(" alignment=\"right\"");
			}
		}
		if (_style > 0) {
			out.print(" style=\"" + _style + "\"");
		}
		if (_bullet != null) {
			out.print(" bullet=\"" + _bullet + "\"");
		}
		if (_textBefore != null) {
			out.print(" textbefore=\"" + _textBefore + "\"");
		}
		if (_textAfter != null) {
			out.print(" textafter=\"" + _textAfter + "\"");
		}
		if (_numstyle > 0) {
			out.print(" numstyle=\"" + _numstyle + "\"");
		}
		if (_orderingStart > 1) {
			out.print(" orderingstart=\"" + _orderingStart + "\"");
		}
		if (_dontShowNumbering == true) {
			out.print(" dontshownumbers=\"1\"");
		}
		if (_spaceBefore > 0) {
			out.print(" spacebefore=\"" + _spaceBefore + "\"");
		}
		if (_spaceAfter > 0) {
			out.print(" spaceafter=\"" + _spaceAfter + "\"");
		}
		if (_firstIndent != 0) {
			out.print(" firstindent=\"" + _firstIndent + "\"");
		}
		if (_leftIndent != 0) {
			out.print(" leftindent=\"" + _leftIndent + "\"");
		}
		if (_rightIndent != 0) {
			out.print(" rightindent=\"" + _rightIndent + "\"");
		}
		if (_lineSpacing > 0) {
			out.print(" linespacing=\"" + _lineSpacing + "\"");
		}
		if (_pbreak) {
			out.print(" pagebreak=\"1\"");
		}
		
		// TODO tabdeflist dumping
	}

}

